package org.mule.weave.v2.helper

import java.io.BufferedOutputStream
import java.io.ByteArrayInputStream
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.InputStream
import java.nio.charset.Charset
import java.nio.charset.StandardCharsets
import groovy.lang.GroovyShell
import org.mule.weave.v2.core.io.FileHelper

import javax.mail.internet.MimeMultipart
import javax.mail.util.ByteArrayDataSource
import org.mule.weave.v2.matchers.WeaveMatchers.matchBin
import org.mule.weave.v2.matchers.WeaveMatchers.matchJson
import org.mule.weave.v2.matchers.WeaveMatchers.matchProperties
import org.mule.weave.v2.matchers.WeaveMatchers.matchString
import org.mule.weave.v2.matchers.WeaveMatchers.matchXml
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.values.BinaryValue
import org.mule.weave.v2.module.reader.AutoPersistedOutputStream
import org.scalatest.matchers.should.Matchers

import scala.io.Source.fromFile
import scala.util.Try

trait AbstractMatcherTest extends Matchers with BaseDataWeaveTest {

  def readFile(expectedFile: File): String = {
    val expectedText: String = {
      if (expectedFile.getName endsWith ".bin")
        ""
      else
        Try(fileToString(expectedFile)).getOrElse(fromFile(expectedFile)(scala.io.Codec("UTF-16")).mkString)
    }
    expectedText
  }

  final protected def assertMatches(writerResult: Any, encoding: Charset, expectedFile: File)(implicit ctx: EvaluationContext): Any = {
    val outputExtension: String = FileHelper.getExtension(expectedFile)
    val memoryService = ctx.serviceManager.memoryService

    writerResult match {
      case result: AutoPersistedOutputStream =>
        val stream = result.toInputStream
        val byteArray = BinaryValue.getBytesFromInputStream(stream, memoryService, true)
        doAssert(byteArray, encoding, expectedFile, outputExtension)
      case result: InputStream =>
        val byteArray = BinaryValue.getBytesFromInputStream(result, memoryService, true)
        doAssert(byteArray, encoding, expectedFile, outputExtension)
      case result =>
        outputExtension match {
          case ".groovy" =>
            val expectedObject: AnyRef = new GroovyShell().evaluate(readFile(expectedFile))
            val difference: Difference = JavaComparator.diff(expectedObject, result)
            difference match {
              case dif: Different =>
                val source: Any = dif.finalCause().source
                val sourceString: String = if (source != null) source + " (class: " + source.getClass.getSimpleName + ")" else "null"
                val targetValue: Any = dif.finalCause().target
                val targetString: String = if (targetValue != null) targetValue + " (class: " + targetValue.getClass.getSimpleName + ")" else "null"
                fail(
                  dif.message() + ", Path: " + dif.path() + ", Expected : " + sourceString + ", Actual " + targetString
                    + "\nActual Value:" + result.toString + "\nExpected Value:" + expectedObject.toString)
              case _ =>
            }
        }
    }
  }

  private def doAssert(actualBytes: Array[Byte], encoding: Charset, expectedFile: File, outputExtension: String)(implicit ctx: EvaluationContext) = {
    try {
      outputExtension match {
        case ".json" =>
          val actual: String = new String(actualBytes, encoding)
          actual should matchJson(readFile(expectedFile))
        case ".xml" =>
          val actual: String = new String(actualBytes, encoding)
          actual should matchXml(readFile(expectedFile))
        case ".dwl" =>
          val actual: String = new String(actualBytes, "UTF-8")
          actual should matchString(readFile(expectedFile))(after being whiteSpaceNormalised)
        case ".csv" =>
          val actual: String = new String(actualBytes, encoding).trim
          actual should matchString(readFile(expectedFile).trim)
        case ".txt" =>
          val actual: String = new String(actualBytes, encoding)
          actual should matchString(readFile(expectedFile))
        case ".bin" =>
          assertBinaryFile(actualBytes, expectedFile)
        case ".urlencoded" =>
          val actual: String = new String(actualBytes, "UTF-8")
          actual should matchString(readFile(expectedFile).trim)
        case ".properties" =>
          val actual: String = new String(actualBytes, "UTF-8")
          actual should matchProperties(readFile(expectedFile).trim)
        case ".multipart" => matchMultipart(expectedFile, actualBytes)
        case ext =>
          assertResult(ext, actualBytes, encoding, expectedFile)
      }
    } catch {
      case e: Exception => { //If it fails then check if we should update result
        if (shouldUpdateResult) {
          updateResult(outputExtension, actualBytes, expectedFile)
        }
        throw e
      }
      case e: AssertionError => { //If it fails then check if we should update result
        if (shouldUpdateResult) {
          updateResult(outputExtension, actualBytes, expectedFile)
        }
        throw e
      }
    }
  }

  def updateResult(ext: String, result: Array[Byte], expectedFile: File): Unit = {
    val testCaseFolder = getTestCaseFolder(expectedFile)
    val srcAst: File = new File(testCaseFolder, expectedFile.getParentFile.getName + File.separator + expectedFile.getName)
    println("WARNING -------------------- Updating ->" + srcAst.getCanonicalPath)
    val bos = new BufferedOutputStream(new FileOutputStream(srcAst))
    bos.write(result)
    bos.close()
  }

  private def matchMultipart(output: File, result: Array[Byte])(implicit ctx: EvaluationContext): Unit = {

    val expected = new MimeMultipart(new ByteArrayDataSource(new FileInputStream(output), "multipart/form-data"))
    val actual = new MimeMultipart(new ByteArrayDataSource(new ByteArrayInputStream(result), "multipart/form-data"))
    actual.getPreamble should matchString(expected.getPreamble)
    actual.getCount shouldBe expected.getCount
    var i = 0
    while (i < expected.getCount) {
      val expectedBodyPart = expected.getBodyPart(i)
      val actualBodyPart = actual.getBodyPart(i)
      actualBodyPart.getContentType should matchString(expectedBodyPart.getContentType)
      actualBodyPart.getDisposition should matchString(expectedBodyPart.getDisposition)
      actualBodyPart.getFileName should matchString(expectedBodyPart.getFileName)

      val actualContent = actualBodyPart.getContent
      val expectedContent = expectedBodyPart.getContent

      val actualContentString: Array[Byte] = actualContent match {
        case inputStream: InputStream => BinaryValue.getBytesFromInputStream(inputStream, ctx.serviceManager.memoryService)
        case str: String              => str.getBytes
      }

      val expectedContentString = expectedContent match {
        case inputStream: InputStream => BinaryValue.getBytesFromInputStream(inputStream, ctx.serviceManager.memoryService)
        case str: String              => str.getBytes
      }

      try {
        actualContentString should matchBin(expectedContentString)
      } catch {
        case e: Throwable =>
          val a = new String(actualContentString, StandardCharsets.UTF_8)
          val b = new String(expectedContentString, StandardCharsets.UTF_8)

          a should matchString(b)
      }
      // actualContentString should matchString(expectedContentString)
      i = i + 1
    }

  }

  protected def assertBinaryFile(result: Array[Byte], expectedFile: File): Any = {
    result should matchBin(expectedFile)
  }

  protected def assertResult(ext: String, result: Array[Byte], encoding: Charset, expectedFile: File): Unit = {
    val actual: String = new String(result, encoding)
    actual.trim should matchString(readFile(expectedFile).trim)
  }

}
