package org.mule.weave.v2.helper

import groovy.lang.GroovyShell
import org.mule.weave.v2.core.io.FileHelper
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.module.core.json.reader.JsonReader
import org.mule.weave.v2.module.core.csv.DefaultCSVDataFormat
import org.mule.weave.v2.module.core.csv.reader.CSVReader
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.pojo.reader.JavaReader
import org.mule.weave.v2.module.core.xml.DefaultXmlDataFormat
import org.mule.weave.v2.module.core.xml.reader.DelegatingXmlReader
import org.mule.weave.v2.module.core.xml.reader.indexed.IndexedXmlReader
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider

import java.io.File
import java.io.FileInputStream
import java.nio.charset.Charset
import java.util.Properties

trait ReadCapableTest {
  def buildReaders(inputs: Array[File], defaultEncoding: String = "UTF-8", forceIndexed: Boolean = false)(implicit ctx: EvaluationContext): Map[String, Reader] = {
    inputs
      .map((file: File) => {
        val reader: Reader = buildReader(file, defaultEncoding, forceIndexed)
        (FileHelper.baseName(file), reader)
      })
      .toMap
  }

  def buildReader(file: File, defaultEncoding: String = "UTF-8", forceIndexed: Boolean = false)(implicit ctx: EvaluationContext): Reader = {
    val configuration = new File(file.getParentFile, FileHelper.baseName(file) + "-config.properties")
    val configurationProperties = new Properties()
    if (configuration.exists()) {
      val stream = new FileInputStream(configuration)
      configurationProperties.load(stream)
      stream.close()
    }

    doBuildReader(file, defaultEncoding, forceIndexed, configurationProperties)
  }

  def doBuildReader(file: File, defaultEncoding: String, forceIndexed: Boolean, configurationProperties: Properties)(implicit ctx: EvaluationContext): Reader = {
    val extension: String = file.getName.drop(file.getName lastIndexOf '.')
    val encoding: String = configurationProperties.getProperty("encoding", defaultEncoding)
    val reader: Reader = extension match {
      case ".xml" =>
        val settings = DefaultXmlDataFormat.createReaderSettings()
        settings.indexedReader = forceIndexed
        if (forceIndexed)
          IndexedXmlReader(file, encoding, settings)
        else
          DelegatingXmlReader(file, encoding, settings)
      case ".csv" =>
        CSVReader(file, encoding, DefaultCSVDataFormat.createReaderSettings(), forceIndexed)
      case ".groovy" => new JavaReader(new GroovyShell().evaluate(file))
      case ".json"   => JsonReader(new FileInputStream(file), encoding, forceIndexed)
      case _ =>
        val maybeFormat = DataFormatManager.byExtension(extension)
        val format = maybeFormat.getOrElse(throw new IllegalArgumentException("Unable to find reader for extension `" + extension + "`"))
        val provider = SourceProvider(file, Charset.forName(encoding), format.defaultMimeType)
        format.reader(provider)

    }
    reader
  }
}

object ReadCapableTest extends ReadCapableTest