package org.mule.weave.v2.interpreted.node.structure.function

import org.mule.weave.v2.core.functions.BaseBinaryFunctionValue
import org.mule.weave.v2.core.functions.BaseQuaternaryFunctionValue
import org.mule.weave.v2.core.functions.BaseTernaryFunctionValue
import org.mule.weave.v2.core.functions.BaseUnaryFunctionValue
import org.mule.weave.v2.core.functions.EmptyFunctionValue
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types.FunctionParamType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values.BaseFunctionValue
import org.mule.weave.v2.model.values.FunctionParameter
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.ValueProvider
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.location.LocationCapable

trait ExecutionContextAwareFunctionValue extends BaseFunctionValue {

  /**
    * Returns the context frame of this function. This frames has references of all the variables that can be referenced from inside the function that are
    * declared outside of it.
    *
    * @return The frame
    */
  def functionContextFrame: Frame

}

class DefaultExecutionContextAwareFunction(
  val parameters: Array[FunctionParameter], //
  val params: Array[FunctionParameterNode], //
  val body: ValueNode[_], //
  val returnTypeNode: Option[ValueNode[Type]],
  override val name: Option[String], //
  val functionContextFrame: Frame, //
  val locationCapable: LocationCapable,
  override val minParams: Int,
  override val maxParams: Int,
  override val paramsTypesRequiresMaterialize: Boolean)
    extends BaseFunctionValue()
    with ExecutionContextAwareFunctionValue {

  override def returnType(implicit ctx: EvaluationContext): Option[Type] = {
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionContextFrame, {
          returnTypeNode.map(_.execute(executionContext).evaluate)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        returnTypeNode.map(_.execute(ExecutionContext(functionContextFrame, ctx)).evaluate)
      }
    }
  }

  override def location(): Location = locationCapable.location()

  def setArgsInBodyFrame(args: Array[Value[_]], functionBodyFrame: Frame)(implicit ctx: EvaluationContext): Unit = {
    val functionArgs = args
    var i: Int = 0
    while (params.length > i) {
      val parameter = params(i)
      val value = functionArgs(i)
      val parameterValue = if (parameter.materialize) value.materialize else value
      functionBodyFrame.updateVariable(parameter.variable.slot, parameterValue)
      i = i + 1
    }
  }

  override def doCall(args: Array[Value[_]])(implicit ctx: EvaluationContext): Value[_] = {
    val functionBodyFrame: Frame = functionContextFrame.child(locationCapable, Some(label))
    setArgsInBodyFrame(args, functionBodyFrame)
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionBodyFrame, {
          body.execute(executionContext)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        body.execute(ExecutionContext(functionBodyFrame, ctx))
      }
    }
  }

  override def doCallInline(args: Array[Value[_]])(implicit ctx: EvaluationContext): Value[_] = {
    ctx match {
      case executionContext: ExecutionContext =>
        val frame = executionContext.executionStack().activeFrame()
        setArgsInBodyFrame(args, frame)
        body.execute(executionContext)
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        call(args)
      }
    }
  }
}

object DefaultExecutionContextAwareFunction {
  def apply(parameters: Array[FunctionParameter], params: Array[FunctionParameterNode], body: ValueNode[_], returnTypeNode: Option[ValueNode[Type]], functionName: Option[String], functionContextFrame: Frame, locationCapable: LocationCapable, minParams: Int, maxParams: Int, requiresMaterialize: Boolean): DefaultExecutionContextAwareFunction =
    new DefaultExecutionContextAwareFunction(parameters, params, body, returnTypeNode, functionName, functionContextFrame, locationCapable, minParams, maxParams, requiresMaterialize)
}

object ExecutionContextAwareFunctionValue {

  def calculateParams(params: Array[FunctionParameterNode], ctx: ExecutionContext): Array[FunctionParameter] = {
    val frame = ctx.executionStack().activeFrame()
    val result = new Array[FunctionParameter](params.length)
    var i = 0
    while (i < params.length) {
      val param = params(i)
      val paramType: Value[Type] = param.wtype.execute(ctx)
      result.update(
        i,
        FunctionParameter(
          param.variable.name,
          paramType.evaluate(ctx),
          param.defaultValue.map((node) => {
            new ValueProvider() {
              override def value()(implicit ctx: EvaluationContext): Value[_] = {
                node.execute(ExecutionContext(frame, ctx))
              }
            }
          }),
          param.typeRequiresMaterialize))
      i = i + 1
    }
    result
  }
}

//Specialized Functions Arity

class EmptyFunctionExecutionContextAwareFunction(
  val functionContextFrame: Frame, //
  body: ValueNode[_], //
  returnTypeNode: Option[ValueNode[Type]],
  locationCapable: LocationCapable,
  override val name: Option[String])
    extends ExecutionContextAwareFunctionValue
    with EmptyFunctionValue {

  override def location(): Location = locationCapable.location()

  override def returnType(implicit ctx: EvaluationContext): Option[Type] = {
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionContextFrame, {
          returnTypeNode.map(_.execute(executionContext).evaluate)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        returnTypeNode.map(_.execute(ExecutionContext(functionContextFrame, ctx)).evaluate)
      }
    }
  }

  def doExecute()(implicit ctx: EvaluationContext): Value[_] = {
    val functionBodyFrame = functionContextFrame.child(locationCapable, name)
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionBodyFrame, {
          body.execute(executionContext)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        body.execute(ExecutionContext(functionBodyFrame, ctx))
      }
    }
  }

  override lazy val functionParamTypes: Array[FunctionParamType] = Array.empty

  override def doExecuteInline()(implicit ctx: EvaluationContext): Value[_] = {
    ctx match {
      case executionContext: ExecutionContext =>
        body.execute(executionContext)
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        doExecute()
      }
    }
  }
}

class UnaryFunctionExecutionContextAwareFunction(
  override val rightParam: FunctionParameter, //
  rightVariable: FunctionParameterNode, //
  val functionContextFrame: Frame, //
  body: ValueNode[_], //
  returnTypeNode: Option[ValueNode[Type]],
  locationCapable: LocationCapable, //
  override val name: Option[String],
  override val minParams: Int,
  override val paramsTypesRequiresMaterialize: Boolean)
    extends BaseUnaryFunctionValue
    with ExecutionContextAwareFunctionValue {

  override def location(): Location = locationCapable.location()

  override def returnType(implicit ctx: EvaluationContext): Option[Type] = {
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionContextFrame, {
          returnTypeNode.map(_.execute(executionContext).evaluate)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        returnTypeNode.map(_.execute(ExecutionContext(functionContextFrame, ctx)).evaluate)
      }
    }
  }

  override def doExecuteInline(rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {

    ctx match {
      case executionContext: ExecutionContext =>
        val functionBodyFrame = executionContext.executionStack().activeFrame()
        functionBodyFrame.updateVariable(rightVariable.variable.slot, if (rightVariable.materialize) rightValue.materialize else rightValue)
        body.execute(executionContext)

      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        doExecute(rightValue)
      }
    }
  }

  override def doExecute(rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val functionBodyFrame = functionContextFrame.child(locationCapable, name)
    functionBodyFrame.updateVariable(rightVariable.variable.slot, if (rightVariable.materialize) rightValue.materialize else rightValue)
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionBodyFrame, {
          body.execute(executionContext)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        body.execute(ExecutionContext(functionBodyFrame, ctx))
      }
    }
  }

  override val R: Type = rightParam.wtype

  override lazy val functionParamTypes: Array[FunctionParamType] = {
    val result = new Array[FunctionParamType](1)
    result.update(0, FunctionParamType(rightParam.wtype, rightParam.value.isDefined, Some(rightParam.name)))
    result
  }
}

class BinaryFunctionExecutionContextAwareFunction(
  override val leftParam: FunctionParameter, //
  override val rightParam: FunctionParameter, //
  leftVariable: FunctionParameterNode, //
  rightVariable: FunctionParameterNode, //
  val functionContextFrame: Frame, //
  body: ValueNode[_], //
  returnTypeNode: Option[ValueNode[Type]],
  locationCapable: LocationCapable,
  override val name: Option[String],
  override val minParams: Int,
  override val paramsTypesRequiresMaterialize: Boolean)
    extends BaseBinaryFunctionValue
    with ExecutionContextAwareFunctionValue {

  override def location(): Location = locationCapable.location()

  override def returnType(implicit ctx: EvaluationContext): Option[Type] = {
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionContextFrame, {
          returnTypeNode.map(_.execute(executionContext).evaluate)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        returnTypeNode.map(_.execute(ExecutionContext(functionContextFrame, ctx)).evaluate)
      }
    }
  }

  override def doExecuteInline(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    ctx match {
      case executionContext: ExecutionContext => {
        val functionBodyFrame = executionContext.executionStack().activeFrame()
        functionBodyFrame.updateVariable(leftVariable.variable.slot, if (leftVariable.materialize) leftValue.materialize else leftValue)
        functionBodyFrame.updateVariable(rightVariable.variable.slot, if (rightVariable.materialize) rightValue.materialize else rightValue)
        body.execute(executionContext)
      }
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        call(leftValue, rightValue)
      }
    }
  }

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val functionBodyFrame = functionContextFrame.child(locationCapable, name)
    functionBodyFrame.updateVariable(leftVariable.variable.slot, if (leftVariable.materialize) leftValue.materialize else leftValue)
    functionBodyFrame.updateVariable(rightVariable.variable.slot, if (rightVariable.materialize) rightValue.materialize else rightValue)
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionBodyFrame, {
          body.execute(executionContext)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        body.execute(ExecutionContext(functionBodyFrame, ctx))
      }
    }
  }

  override lazy val functionParamTypes: Array[FunctionParamType] = {
    val result = new Array[FunctionParamType](2)
    result.update(0, FunctionParamType(leftParam.wtype, leftParam.value.isDefined, Some(leftParam.name)))
    result.update(1, FunctionParamType(rightParam.wtype, rightParam.value.isDefined, Some(rightParam.name)))
    result
  }

  override val L: Type = leftParam.wtype

  override val R: Type = rightParam.wtype
}

class TernaryFunctionExecutionContextAwareFunction(
  override val firstParam: FunctionParameter, //
  override val secondParam: FunctionParameter, //
  override val thirdParam: FunctionParameter, //
  firstVariable: FunctionParameterNode, //
  secondVariable: FunctionParameterNode, //
  thirdVariable: FunctionParameterNode, //
  val functionContextFrame: Frame, //
  body: ValueNode[_], //
  returnTypeNode: Option[ValueNode[Type]],
  locationCapable: LocationCapable,
  override val name: Option[String],
  override val minParams: Int,
  override val paramsTypesRequiresMaterialize: Boolean)
    extends BaseTernaryFunctionValue
    with ExecutionContextAwareFunctionValue {

  override def location(): Location = locationCapable.location()

  override def returnType(implicit ctx: EvaluationContext): Option[Type] = {
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionContextFrame, {
          returnTypeNode.map(_.execute(executionContext).evaluate)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        returnTypeNode.map(_.execute(ExecutionContext(functionContextFrame, ctx)).evaluate)
      }
    }
  }

  override protected def doExecuteInline(firstValue: First.V, secondValue: Second.V, thirdValue: Third.V)(implicit ctx: EvaluationContext): Value[_] = {
    ctx match {
      case executionContext: ExecutionContext =>
        val functionBodyFrame = executionContext.executionStack().activeFrame()
        functionBodyFrame.updateVariable(firstVariable.variable.slot, if (firstVariable.materialize) firstValue.materialize else firstValue)
        functionBodyFrame.updateVariable(secondVariable.variable.slot, if (secondVariable.materialize) secondValue.materialize else secondValue)
        functionBodyFrame.updateVariable(thirdVariable.variable.slot, if (thirdVariable.materialize) thirdValue.materialize else thirdValue)
        body.execute(executionContext)

      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        doExecute(firstValue, secondValue, thirdValue)
      }
    }
  }

  override def doExecute(firstValue: First.V, secondValue: Second.V, thirdValue: Third.V)(implicit ctx: EvaluationContext): Value[_] = {
    val functionBodyFrame = functionContextFrame.child(locationCapable, name)
    functionBodyFrame.updateVariable(firstVariable.variable.slot, if (firstVariable.materialize) firstValue.materialize else firstValue)
    functionBodyFrame.updateVariable(secondVariable.variable.slot, if (secondVariable.materialize) secondValue.materialize else secondValue)
    functionBodyFrame.updateVariable(thirdVariable.variable.slot, if (thirdVariable.materialize) thirdValue.materialize else thirdValue)
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionBodyFrame, {
          body.execute(executionContext)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        body.execute(ExecutionContext(functionBodyFrame, ctx))
      }
    }
  }

  override lazy val functionParamTypes: Array[FunctionParamType] = {
    val result = new Array[FunctionParamType](3)
    result.update(0, FunctionParamType(firstParam.wtype, firstParam.value.isDefined, Some(firstParam.name)))
    result.update(1, FunctionParamType(secondParam.wtype, secondParam.value.isDefined, Some(secondParam.name)))
    result.update(2, FunctionParamType(thirdParam.wtype, thirdParam.value.isDefined, Some(thirdParam.name)))
    result
  }

  override val First: Type = firstParam.wtype

  override val Second: Type = secondParam.wtype

  override val Third: Type = thirdParam.wtype
}

class QuaternaryFunctionExecutionContextAwareFunction(
  override val firstParam: FunctionParameter, //
  override val secondParam: FunctionParameter, //
  override val thirdParam: FunctionParameter, //
  override val forthParam: FunctionParameter, //
  firstVariable: FunctionParameterNode, //
  secondVariable: FunctionParameterNode, //
  thirdVariable: FunctionParameterNode, //
  forthVariable: FunctionParameterNode, //
  val functionContextFrame: Frame, //
  body: ValueNode[_], //
  returnTypeNode: Option[ValueNode[Type]],
  locationCapable: LocationCapable,
  override val name: Option[String],
  override val minParams: Int,
  override val paramsTypesRequiresMaterialize: Boolean)
    extends BaseQuaternaryFunctionValue
    with ExecutionContextAwareFunctionValue {

  override def location(): Location = locationCapable.location()

  override def returnType(implicit ctx: EvaluationContext): Option[Type] = {
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionContextFrame, {
          returnTypeNode.map(_.execute(executionContext).evaluate)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        returnTypeNode.map(_.execute(ExecutionContext(functionContextFrame, ctx)).evaluate)
      }
    }
  }

  override protected def doExecuteInline(firstValue: First.V, secondValue: Second.V, thirdValue: Third.V, forthValue: Forth.V)(implicit ctx: EvaluationContext): Value[_] = {
    ctx match {
      case executionContext: ExecutionContext =>
        val functionBodyFrame = executionContext.executionStack().activeFrame()
        functionBodyFrame.updateVariable(firstVariable.variable.slot, if (firstVariable.materialize) firstValue.materialize else firstValue)
        functionBodyFrame.updateVariable(secondVariable.variable.slot, if (secondVariable.materialize) secondValue.materialize else secondValue)
        functionBodyFrame.updateVariable(thirdVariable.variable.slot, if (thirdVariable.materialize) thirdValue.materialize else thirdValue)
        functionBodyFrame.updateVariable(forthVariable.variable.slot, if (forthVariable.materialize) forthValue.materialize else forthValue)
        body.execute(executionContext)

      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        doExecute(firstValue, secondValue, thirdValue, forthValue)
      }
    }
  }

  override def doExecute(firstValue: First.V, secondValue: Second.V, thirdValue: Third.V, forthValue: Forth.V)(implicit ctx: EvaluationContext): Value[_] = {
    val functionBodyFrame = functionContextFrame.child(locationCapable, name)
    functionBodyFrame.updateVariable(firstVariable.variable.slot, if (firstVariable.materialize) firstValue.materialize else firstValue)
    functionBodyFrame.updateVariable(secondVariable.variable.slot, if (secondVariable.materialize) secondValue.materialize else secondValue)
    functionBodyFrame.updateVariable(thirdVariable.variable.slot, if (thirdVariable.materialize) thirdValue.materialize else thirdValue)
    functionBodyFrame.updateVariable(forthVariable.variable.slot, if (forthVariable.materialize) forthValue.materialize else forthValue)
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(functionBodyFrame, {
          body.execute(executionContext)
        })
      case _ => {
        //We don't always use this one because multithreading inside dw breaks (http service)
        body.execute(ExecutionContext(functionBodyFrame, ctx))
      }
    }
  }

  override val First: Type = firstParam.wtype

  override val Second: Type = secondParam.wtype

  override val Third: Type = thirdParam.wtype

  override val Forth: Type = forthParam.wtype

  override lazy val functionParamTypes: Array[FunctionParamType] = {
    val result = new Array[FunctionParamType](4)
    result.update(0, FunctionParamType(firstParam.wtype, firstParam.value.isDefined, Some(firstParam.name)))
    result.update(1, FunctionParamType(secondParam.wtype, secondParam.value.isDefined, Some(secondParam.name)))
    result.update(2, FunctionParamType(thirdParam.wtype, thirdParam.value.isDefined, Some(thirdParam.name)))
    result.update(2, FunctionParamType(forthParam.wtype, forthParam.value.isDefined, Some(forthParam.name)))
    result
  }
}
