package org.mule.weave.v2.interpreted.transform

import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.pattern._
import org.mule.weave.v2.interpreted.node.structure.RLiteralTypeNode
import org.mule.weave.v2.interpreted.node.structure.function.DynamicFunctionNode
import org.mule.weave.v2.interpreted.node.structure.function.FunctionParameterNode
import org.mule.weave.v2.parser
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.WeaveType

trait EnginePatternTransformations extends AstTransformation {

  def transformExpressionPatternNode(pattern: AstNode, name: NameIdentifier, function: AstNode): ExpressionPatternNode = {
    transformExpressionPatternNode(pattern, name, function, transform(function))
  }

  def transformExpressionPatternNode(pattern: AstNode, name: NameIdentifier, function: AstNode, body: ValueNode[_]): ExpressionPatternNode = {
    val functionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(name), needsMaterialization(name))), body, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location

    val patternExpression: ValueNode[_] = transform(pattern)

    val patternFunctionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(name), needsMaterialization(name))), patternExpression, None, requiresMaterializedArguments = false, literalParameters = true)
    patternFunctionNode._location = pattern._location
    new ExpressionPatternNode(patternFunctionNode, functionNode)
  }

  def transformLiteralPatternNode(pattern: AstNode, name: NameIdentifier, function: AstNode): LiteralPatternNode = {
    transformLiteralPatternNode(pattern, name, function, transform(function))
  }

  def transformLiteralPatternNode(pattern: AstNode, name: NameIdentifier, onMatch: AstNode, body: ValueNode[_]): LiteralPatternNode = {
    val onMatchFunctionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(name), needsMaterialization(name))), body, None, requiresMaterializedArguments = false, literalParameters = true)
    onMatchFunctionNode._location = onMatch._location
    new LiteralPatternNode(transform(pattern), onMatchFunctionNode)
  }

  def transformRegexPatternNode(pattern: AstNode, name: NameIdentifier, function: AstNode): RegexPatternNode = {
    transformRegexPatternNode(pattern, name, function, transform(function))
  }

  def transformRegexPatternNode(pattern: AstNode, name: NameIdentifier, function: AstNode, body: ValueNode[_]): RegexPatternNode = {
    val functionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(name), needsMaterialization(name))), body, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location
    new RegexPatternNode(transform(pattern), functionNode)
  }

  def transformDefaultPatternNode(nameIdentifier: NameIdentifier, value: AstNode): DefaultPatternNode = {
    transformDefaultPatternNode(nameIdentifier, value, transform(value))
  }

  def transformDefaultPatternNode(nameIdentifier: NameIdentifier, value: AstNode, body: ValueNode[_]): DefaultPatternNode = {
    val onMatchFunctionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(nameIdentifier), needsMaterialization(nameIdentifier))), body, None, requiresMaterializedArguments = false, literalParameters = true)
    onMatchFunctionNode._location = value._location
    new DefaultPatternNode(onMatchFunctionNode)
  }

  def transformTypePatternNode(pattern: WeaveTypeNode, name: NameIdentifier, function: AstNode): TypePatternNode = {
    transformTypePatternNode(pattern, name, function, transform(function))
  }

  def transformTypePatternNode(pattern: WeaveTypeNode, name: NameIdentifier, onMatch: AstNode, body: ValueNode[_]): TypePatternNode = {
    val onMatchFunctionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(name), needsMaterialization(name))), body, None, requiresMaterializedArguments = false, literalParameters = true)
    onMatchFunctionNode._location = onMatch._location
    val theType = new RLiteralTypeNode(transform(pattern))
    val weaveType = WeaveType(pattern, scopeNavigator().rootScope.referenceResolver())
    new TypePatternNode(theType, onMatchFunctionNode, TypeHelper.requiredMaterialize(weaveType))
  }

  def transformPatternMatcherNode(lhs: AstNode, patterns: parser.ast.patterns.PatternExpressionsNode): PatternMatcherNode = {
    new PatternMatcherNode(transform(lhs), transformSeq(patterns.patterns).toArray)
  }

  def transformEmptyArrayNode(function: AstNode): EmptyArrayPatternNode = {
    transformEmptyArrayNode(function, transform(function))
  }

  def transformEmptyArrayNode(function: AstNode, body: ValueNode[_]): EmptyArrayPatternNode = {
    val functionNode = DynamicFunctionNode(Array(), body, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location
    new EmptyArrayPatternNode(functionNode)
  }

  def transformDeconstructArrayNode(head: NameIdentifier, tail: NameIdentifier, function: AstNode): DeconstructArrayPatternNode = {
    transformDeconstructArrayNode(head, tail, function, transform(function))
  }

  def transformDeconstructArrayNode(head: NameIdentifier, tail: NameIdentifier, function: AstNode, body: ValueNode[_]): DeconstructArrayPatternNode = {
    val parameterNodes = Array(FunctionParameterNode(transform(head), needsMaterialization(head)), FunctionParameterNode(transform(tail), needsMaterialization(tail)))
    val functionNode: DynamicFunctionNode = DynamicFunctionNode(parameterNodes, body, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location
    new DeconstructArrayPatternNode(functionNode)
  }

  def transformEmptyObjectNode(function: AstNode): EmptyObjectPatternNode = {
    transformEmptyObjectNode(function, transform(function))
  }

  def transformEmptyObjectNode(function: AstNode, value: ValueNode[_]): EmptyObjectPatternNode = {
    val functionNode = DynamicFunctionNode(Array(), value, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location
    new EmptyObjectPatternNode(functionNode)
  }

  def transformDeconstructObjectNode(headKey: NameIdentifier, headValue: NameIdentifier, tail: NameIdentifier, function: AstNode): DeconstructObjectPatternNode = {
    transformDeconstructObjectNode(headKey, headValue, tail, function, transform(function))
  }

  def transformDeconstructObjectNode(headKey: NameIdentifier, headValue: NameIdentifier, tail: NameIdentifier, function: AstNode, body: ValueNode[_]): DeconstructObjectPatternNode = {
    val parameterNodes = Array(
      FunctionParameterNode(transform(headKey), needsMaterialization(headKey)),
      FunctionParameterNode(transform(headValue), needsMaterialization(headValue)),
      FunctionParameterNode(transform(tail), needsMaterialization(tail)))
    val functionNode = DynamicFunctionNode(parameterNodes, body, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location
    new DeconstructObjectPatternNode(functionNode)
  }
}
