package org.mule.weave.v2.runtime.core.functions.crypto

import javax.crypto.Mac
import javax.crypto.spec.SecretKeySpec
import org.mule.weave.v2.core.functions.TernaryFunctionValue
import org.mule.weave.v2.core.exception.IllegalHMACArgumentException
import org.mule.weave.v2.core.exception.UnknownAlgorithmException
import org.mule.weave.v2.core.io.MemoryService
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.parser.location.UnknownLocation

import java.security.NoSuchAlgorithmException

object HMACFunctionValue extends TernaryFunctionValue {

  override val First = BinaryType

  override val Second = BinaryType

  override val Third = StringType

  override val thirdDefaultValue: Option[ValueProvider] = Some(ValueProvider(StringValue("HmacSHA1")))

  override def doExecute(secretValue: First.V, contentValue: Second.V, algorithmValue: Third.V)(implicit ctx: EvaluationContext): Value[_] = {
    val memoryService: MemoryService = ctx.serviceManager.memoryService
    val secret: Array[Byte] = BinaryValue.getBytesFromSeekableStream(secretValue.evaluate, memoryService)
    val content: Array[Byte] = BinaryValue.getBytesFromSeekableStream(contentValue.evaluate, memoryService)
    val algorithm: String = algorithmValue.evaluate.toString

    try {
      val secretKey = new SecretKeySpec(secret, algorithm)
      val mac = Mac.getInstance(algorithm)
      mac.init(secretKey)
      val result: Array[Byte] = mac.doFinal(content)

      BinaryValue(result)
    } catch {
      case e: NoSuchAlgorithmException => throw new UnknownAlgorithmException(algorithmValue.location(), e.getMessage)
      case e: Throwable                => throw new IllegalHMACArgumentException(UnknownLocation, e.getMessage)
    }
  }

}
