package org.mule.weave.v2.interpreted

import java.io.ByteArrayOutputStream
import java.nio.charset.Charset

import org.mule.weave.v2.interpreted.debugger.server.DefaultWeaveDebuggingSession
import org.mule.weave.v2.interpreted.debugger.server.ServerProtocol
import org.mule.weave.v2.interpreted.debugger.server.WeaveDebuggerExecutor
import org.mule.weave.v2.interpreted.debugger.server.WeaveDebuggingSession
import org.mule.weave.v2.interpreted.listener.NotificationManager
import org.mule.weave.v2.interpreted.listener.WatchdogExecutionListener
import org.mule.weave.v2.interpreted.listener.WeaveExecutionListener
import org.mule.weave.v2.interpreted.module.WeaveDataFormat
import org.mule.weave.v2.interpreted.node.AstWrapperRoot
import org.mule.weave.v2.interpreted.transform.EngineGrammarTransformation
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.ValuesHelper
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.writer.ConfigurableEncoding
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.phase.AstNodeResultAware
import org.mule.weave.v2.parser.phase.CompilationPhase
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.ScopeNavigatorResultAware
import org.mule.weave.v2.parser.phase.SuccessResult
import org.mule.weave.v2.runtime.CompilationResult
import org.mule.weave.v2.runtime.DebugAwareWeave
import org.mule.weave.v2.runtime.ExecutableWeave

import scala.collection.mutable.ArrayBuffer

class InterpreterModuleCompilerPhase(moduleNodeLoader: RuntimeModuleNodeCompiler) extends CompilationPhase[AstNodeResultAware[ModuleNode] with ScopeNavigatorResultAware, CompilationResult[ModuleNode]] {
  override def doCall(source: AstNodeResultAware[ModuleNode] with ScopeNavigatorResultAware, context: ParsingContext): PhaseResult[_ <: CompilationResult[ModuleNode]] = {
    val astNode: ModuleNode = source.astNode
    val transform: node.ModuleNode = EngineGrammarTransformation(context, source.scope.astNavigator(), moduleNodeLoader).transformModule(astNode)
    SuccessResult(CompilationResult(astNode, InterpretedModuleExecutableWeave(transform, astNode), source.scope), context)
  }
}

class InterpretedModuleExecutableWeave(val executableDocument: node.ModuleNode, val astDocument: ModuleNode, val functionName: String = "main") extends ExecutableWeave[ModuleNode] with DebugAwareWeave {

  private val nodeListeners: ArrayBuffer[WeaveExecutionListener] = ArrayBuffer()
  private var materializeValues: Boolean = false
  private var maxExecutionTime: Long = -1

  /**
    * Registers an execution listener
    *
    * @param listener
    */
  override def addExecutionListener(listener: WeaveExecutionListener): Unit = {
    nodeListeners += listener
  }

  override def withMaxTime(maxExecutionTime: Long): InterpretedModuleExecutableWeave = {
    this.maxExecutionTime = maxExecutionTime
    this
  }

  /**
    * Dumps the execution tree
    *
    * @return The result
    */
  def dumpExecutionTree(): (Any, Charset) = {
    implicit val ctx = EvaluationContext()
    val value: DataFormat[_, _] = DataFormatManager.byContentType("application/xml").getOrElse(new WeaveDataFormat())
    val writer: Writer = value.writer(Some(new ByteArrayOutputStream()))
    val context: ExecutionContext = ExecutionContext(
      writer = writer,
      notificationManager = createNotificationManager,
      materializedValue = materializeValues,
      location = astDocument)
    try {
      writer.startDocument(executableDocument)
      writer.writeValue(new AstWrapperRoot(executableDocument))(context)
      writer.endDocument(executableDocument)
    } finally {
      context.close()
    }
    (writer.result, writer.settings match {
      case e: ConfigurableEncoding => e.charset
      case _                       => context.serviceManager.charsetProviderService.defaultCharset()
    })
  }

  /**
    * Enables debugging over this weave script
    *
    * @param debuggerProtocol The debugger protocol
    * @return The session
    */
  override def debug(debuggerProtocol: ServerProtocol): DefaultWeaveDebuggingSession = {
    val session: DefaultWeaveDebuggingSession = new DefaultWeaveDebuggingSession(debuggerProtocol)
    debug(session)
    session
  }

  def debug(session: WeaveDebuggingSession): Unit = {
    val debuggerExecutor: WeaveDebuggerExecutor = new WeaveDebuggerExecutor(session)
    session.start(debuggerExecutor)
    addExecutionListener(debuggerExecutor)
    materializeValues = true
  }

  def materializedValuesExecution(materialize: Boolean): Unit = {
    materializeValues = materialize
  }

  override def execute(readers: Map[String, Reader], values: Map[String, Value[_]])(implicit ctx: EvaluationContext): Value[_] = {
    val mainSlot = executableDocument.variableTable.variables.find(_.name == functionName)
    val notificationManager = createNotificationManager
    implicit val context: ExecutionContext = ExecutionContext(
      variableTable = executableDocument.variableTable,
      moduleTable = executableDocument.moduleTable,
      notificationManager = notificationManager,
      evaluationContext = ctx,
      materializedValue = materializeValues,
      location = astDocument)
    mainSlot match {
      case Some(main) => {
        executableDocument.importDirectives.foreach(_.execute)
        executableDocument.directives.foreach(_.execute)
        val mainFunction = FunctionType.coerce(context.executionStack().getVariable(main.slot))
        mainFunction.call(ValuesHelper.array(ArrayValue(values.values.toSeq)))
      }
      case None => {
        throw new RuntimeException(s"No `${functionName}` function defined in the module")
      };
    }
  }

  def collectVariables(variables: Seq[String])(implicit ctx: EvaluationContext): Map[String, Value[_]] = {

    val notificationManager = createNotificationManager
    implicit val context: ExecutionContext = ExecutionContext(
      variableTable = executableDocument.variableTable,
      moduleTable = executableDocument.moduleTable,
      notificationManager = notificationManager,
      evaluationContext = ctx,
      materializedValue = materializeValues,
      location = astDocument)

    executableDocument.importDirectives.foreach(_.execute)
    executableDocument.directives.foreach(_.execute)
    variables
      .flatMap((name) => {
        val mainSlot = executableDocument.variableTable.variables.find(_.name == name)
        mainSlot.map((main) => {
          (name, context.executionStack().getVariable(main.slot))
        })
      })
      .toMap

  }

  private def createNotificationManager = {
    val listeners = if (maxExecutionTime > -1) {
      nodeListeners.toArray :+ new WatchdogExecutionListener(maxExecutionTime)
    } else {
      nodeListeners.toArray
    }
    NotificationManager(listeners)
  }

  override def writeWith(writer: Writer, readers: Map[String, Reader], values: Map[String, Value[_]], closeAfterWrite: Boolean)(implicit ctx: EvaluationContext): (Any, Charset) = {
    throw new RuntimeException("Module does not support write")
  }

  override def removeExecutionListener(listener: WeaveExecutionListener): Unit = {
    nodeListeners -= listener
  }
}

object InterpretedModuleExecutableWeave {
  def apply(executableDocument: node.ModuleNode, astDocument: ModuleNode): InterpretedModuleExecutableWeave =
    new InterpretedModuleExecutableWeave(executableDocument, astDocument)
}
