package org.mule.weave.v2.interpreted.marker

import org.mule.weave.v2.interpreted.InterpreterPreCompilerMarker
import org.mule.weave.v2.parser.annotation.AstNodeAnnotation
import org.mule.weave.v2.parser.annotation.WithValueAstNodeAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.header.directives.FunctionDirectiveNode
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.scope.ScopesNavigator

class FunctionCallNodeMarker extends InterpreterPreCompilerMarker {

  override def mark(node: AstNode, scope: ScopesNavigator, context: ParsingContext): Unit = {
    val navigator = scope.astNavigator()
    node match {
      case fcn: FunctionCallNode =>
        // Set function name
        if (!fcn.function.isAnnotatedWith(classOf[FunctionCallNameAnnotation])) {
          val functionName = AstNodeHelper.functionRefName(fcn.function, scope)
          fcn.function.annotate(new FunctionCallNameAnnotation(functionName))
        }

        // Set static recursive function call
        if (!fcn.function.isAnnotatedWith(classOf[StaticRecFunctionCallAnnotation])) {
          val maybeFunctionNode = navigator.parentWithType(fcn.function, classOf[FunctionNode])
          if (maybeFunctionNode.isDefined) {
            val fn = maybeFunctionNode.get
            val maybeParent = navigator.parentOf(fn)
            if (maybeParent.isDefined) {
              maybeParent.get match {
                case fdn: FunctionDirectiveNode =>
                  if (AstNodeHelper.isRecursiveFunctionCall(fdn.variable, fcn, scope)) {
                    fcn.annotate(new StaticRecFunctionCallAnnotation())
                  }
                case _ =>
                // Nothing to do
              }
            }
          }
        }

      case _ =>
      // Nothing to do
    }
  }
}

class FunctionCallNameAnnotation(val functionName: String) extends WithValueAstNodeAnnotation[String] {

  override def name(): String = {
    "FunctionCallName"
  }

  override def value(): String = {
    functionName
  }
}

class StaticRecFunctionCallAnnotation extends AstNodeAnnotation {
  override def name(): String = {
    "StaticRecFunctionCall"
  }
}