package org.mule.weave.v2.compilation.loader

import org.mule.weave.v2.compilation.message.WeaveBinaryDeserializationError
import org.mule.weave.v2.compilation.mapper.ast.ModuleNodeSerializer
import org.mule.weave.v2.compilation.serializer.SerializableAstNodeSerializer
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.SafeStringBasedParserInput
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.phase.ModuleLoader
import org.mule.weave.v2.parser.phase.ParsingContentInput
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.ParsingResult
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.sdk.BinaryWeaveResource
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.mule.weave.v2.sdk.WeaveResourceResolverAware

import java.io.ByteArrayInputStream
import java.io.DataInputStream

class WeaveBinaryResourceModuleLoader extends ModuleLoader with WeaveResourceResolverAware {
  private var resolver: WeaveResourceResolver = _

  override def name(): Option[String] = Some("bdwl")

  /**
    * Builds if possible the module ast for the given NameIdentifier
    *
    * @param nameIdentifier The name of the module to be build
    * @param parentContext  The parsing context for this module
    * @return If was able to build the ParsingResult
    */
  override def loadModule(nameIdentifier: NameIdentifier, parentContext: ParsingContext): Option[PhaseResult[ParsingResult[ModuleNode]]] = {
    parentContext.notificationManager.startResolvingResource(nameIdentifier, resolver)
    val resolve = resolver.resolveBinary(nameIdentifier)
    parentContext.notificationManager.endResolvingResource(nameIdentifier, resolver, resolve)
    resolve.map(resource => parseBinaryModule(nameIdentifier, resource))
  }

  override def resolver(resolver: WeaveResourceResolver): Unit = {
    this.resolver = resolver
  }

  private def parseBinaryModule(identifier: NameIdentifier, resource: BinaryWeaveResource): PhaseResult[ParsingResult[ModuleNode]] = {
    val inputStream = new ByteArrayInputStream(resource.binaryContent)
    val dataInputStream = new DataInputStream(inputStream)
    val input: ParsingContentInput = ParsingContentInput(resource, identifier, SafeStringBasedParserInput(resource.content()))
    try {
      val deserializedNode = SerializableAstNodeSerializer.deserialize(dataInputStream)
      val moduleNode = ModuleNodeSerializer.deserialize(deserializedNode, resource, identifier)
      new PhaseResult(Some(ParsingResult(input, moduleNode)), MessageCollector())
    } catch {
      case e: Throwable => new PhaseResult(None, MessageCollector().error(WeaveBinaryDeserializationError(identifier, e.getMessage), UnknownLocation))
    }
  }
}
