package org.mule.weave.v2.compilation.mapper

import org.mule.weave.v2.compilation.ArraySerializableAstNode
import org.mule.weave.v2.compilation.BooleanSerializableValueAstNode
import org.mule.weave.v2.compilation.IntSerializableValueAstNode
import org.mule.weave.v2.compilation.NoneSerializableValueAstNode
import org.mule.weave.v2.compilation.SerializableAstNode
import org.mule.weave.v2.compilation.StringSerializableValueAstNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.WeaveResource

trait DWSerializer[T] {
  def serialize(value: T): SerializableAstNode
  def deserialize(node: SerializableAstNode, resource: WeaveResource, identifier: NameIdentifier): T

  def deserializeMaybe(node: SerializableAstNode, resource: WeaveResource, identifier: NameIdentifier): Option[T] = {
    node match {
      case NoneSerializableValueAstNode() => None
      case _                              => Some(deserialize(node, resource, identifier))
    }
  }

  def deserializeSeq(node: SerializableAstNode, resource: WeaveResource, identifier: NameIdentifier): Seq[T] = {
    node.asInstanceOf[ArraySerializableAstNode].children.map(deserialize(_, resource, identifier)).toList
  }

  def deserializeMaybeSeq(node: SerializableAstNode, resource: WeaveResource, identifier: NameIdentifier): Option[Seq[T]] = {
    node match {
      case NoneSerializableValueAstNode() => None
      case _                              => Some(deserializeSeq(node, resource, identifier))
    }
  }

  def deserializeString(node: SerializableAstNode): String = {
    node.asInstanceOf[StringSerializableValueAstNode].value
  }

  def deserializeBoolean(node: SerializableAstNode): Boolean = {
    node.asInstanceOf[BooleanSerializableValueAstNode].value
  }

  def deserializeInt(node: SerializableAstNode): Int = {
    node.asInstanceOf[IntSerializableValueAstNode].value
  }

  def deserializeMaybeString(node: SerializableAstNode): Option[String] = {
    node match {
      case NoneSerializableValueAstNode()        => None
      case StringSerializableValueAstNode(value) => Some(value)
      case _                                     => None //TODO: throw?
    }
  }

  def serializeMaybe(maybeValue: Option[T]): SerializableAstNode = {
    maybeValue match {
      case Some(value) => serialize(value)
      case None        => NoneSerializableValueAstNode()
    }
  }

  def serializeMaybeSeq(maybeValue: Option[Seq[T]]): SerializableAstNode = {
    maybeValue match {
      case Some(value) => serializeSeq(value)
      case None        => NoneSerializableValueAstNode()
    }
  }

  def serializeSeq(seq: Seq[T]): ArraySerializableAstNode = {
    ArraySerializableAstNode(seq.map(serialize))
  }

  def serializeString(str: String): StringSerializableValueAstNode = StringSerializableValueAstNode(str)

  def serializeBoolean(b: Boolean): BooleanSerializableValueAstNode = BooleanSerializableValueAstNode(b)

  def serializeInt(n: Int): IntSerializableValueAstNode = IntSerializableValueAstNode(n)

  def serializerMaybeString(maybeStr: Option[String]): SerializableAstNode = {
    maybeStr match {
      case Some(str) => serializeString(str)
      case None      => NoneSerializableValueAstNode()
    }
  }
}
