package org.mule.weave.v2.compilation.mapper.ast

import org.mule.weave.v2.compilation.ArraySerializableAstNode
import org.mule.weave.v2.compilation.SerializableAstNode
import org.mule.weave.v2.compilation.SerializableAstNodeLocation
import org.mule.weave.v2.compilation.mapper.DWSerializer
import org.mule.weave.v2.compilation.mapper.SerializableLocationMapper
import org.mule.weave.v2.compilation.mapper.nonAst.AstNodeAnnotationSerializer
import org.mule.weave.v2.parser.annotation.EphemeralAstNodeAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.WeaveResource

trait AstNodeSerializer[T <: AstNode] extends DWSerializer[T] {

  override final def serialize(node: T): SerializableAstNode = {
    val location = SerializableLocationMapper.serialize(node.location())
    val runtimeAnnotations = node.annotations().filterNot(_.isInstanceOf[EphemeralAstNodeAnnotation])
    val _annotations = ArraySerializableAstNode(runtimeAnnotations.map(AstNodeAnnotationSerializer.serialize))

    doSerialize(node, location, _annotations)
  }

  override final def deserialize(node: SerializableAstNode, srcResource: WeaveResource, srcIdentifier: NameIdentifier): T = {
    val location = SerializableLocationMapper.deserialize(node.location(), srcResource, srcIdentifier)
    val annotations = AstNodeAnnotationSerializer.deserializeSeq(node.children().head, srcResource, srcIdentifier)
    val children = node.children().slice(1, node.children().length)

    val astNode = doDeserialize(children, srcResource, srcIdentifier)

    astNode._location = Some(location)
    annotations.foreach(astNode.annotate)
    astNode
  }

  protected def doSerialize(node: T, location: SerializableAstNodeLocation, _annotations: ArraySerializableAstNode): SerializableAstNode

  protected def doDeserialize(children: Seq[SerializableAstNode], srcResource: WeaveResource, srcIdentifier: NameIdentifier): T
}

object AstNodeSerializer {
  def serialize[T <: AstNode](value: T)(implicit serializer: AstNodeSerializer[T]): SerializableAstNode = {
    serializer.serialize(value)
  }

  def deserialize[T <: AstNode](value: SerializableAstNode, srcResource: WeaveResource, srcIdentifier: NameIdentifier)(implicit serializer: AstNodeSerializer[T]): T = {
    serializer.deserialize(value, srcResource, srcIdentifier)
  }
}