package org.mule.weave.v2.runtime

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.api.tooling.annotation.DWAnnotationProcessor
import org.mule.weave.v2.core.io.FileHelper
import org.mule.weave.v2.core.io.service.CustomWorkingDirectoryService
import org.mule.weave.v2.core.io.service.WorkingDirectoryService
import org.mule.weave.v2.core.util.TypeAliases
import org.mule.weave.v2.interpreted.ConfigurationHelper
import org.mule.weave.v2.interpreted.debugger.server.DefaultWeaveDebuggingSession
import org.mule.weave.v2.interpreted.debugger.server.WeaveDebuggerExecutor
import org.mule.weave.v2.interpreted.debugger.server.tcp.TcpServerProtocol
import org.mule.weave.v2.interpreted.extension.ParsingContextCreator
import org.mule.weave.v2.interpreted.extension.WeaveBasedDataFormatExtensionLoaderService
import org.mule.weave.v2.interpreted.listener.WeaveExecutionListener
import org.mule.weave.v2.interpreted.profiler.ExecutionTelemetryListener
import org.mule.weave.v2.model.EmptyWeaveServicesProvider
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.SPIWeaveServicesProvider
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.WeaveServicesProvider
import org.mule.weave.v2.model.service.CpuLimitService
import org.mule.weave.v2.model.service.CpuLimitedPatternService
import org.mule.weave.v2.model.service.DefaultCpuLimitService
import org.mule.weave.v2.model.service.DefaultLanguageLevelService
import org.mule.weave.v2.model.service.DefaultLoggingConfigurationService
import org.mule.weave.v2.model.service.DefaultPatternService
import org.mule.weave.v2.model.service.LanguageLevelService
import org.mule.weave.v2.model.service.LoggingConfigurationService
import org.mule.weave.v2.model.service.LoggingService
import org.mule.weave.v2.model.service.NoOpCpuLimitService
import org.mule.weave.v2.model.service.PatternService
import org.mule.weave.v2.model.service.PropertiesSettings
import org.mule.weave.v2.model.service.RuntimeSettings
import org.mule.weave.v2.model.service.SettingsService
import org.mule.weave.v2.model.service.SimpleSettingsService
import org.mule.weave.v2.model.service.WeaveLanguageLevelService
import org.mule.weave.v2.model.service.api.LoggingConfigurationSourceProvider
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.CompositeDataFormatExtensionsLoaderService
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.DataFormatExtensionsLoaderService
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.DefaultDataFormatExtensionsLoaderService
import org.mule.weave.v2.module.reader.ConfigurableStreaming
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.DocumentParser
import org.mule.weave.v2.parser.MappingParser
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.annotation.StreamingCapableVariableAnnotation
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.module.MimeType
import org.mule.weave.v2.parser.phase.ModuleParsingPhasesManager
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.WatchdogParsingListener
import org.mule.weave.v2.parser.phase.metrics.CompilationMetrics
import org.mule.weave.v2.parser.phase.metrics.ParsingMetricsCollectorListener
import org.mule.weave.v2.runtime.utils.FutureValue
import org.mule.weave.v2.sdk.{ BinaryWeaveResource, DefaultWeaveResource, WeaveResource, WeaveResourceFactory }
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.utils.DataWeaveVersion

import java.io.File
import java.io.InputStream
import java.net.URL
import java.nio.charset.Charset
import java.nio.file.Files
import java.util.Properties
import java.util.concurrent.Executors
import scala.collection.mutable
import scala.io.Source

/**
  * Simple weave execution api that is also compatible with java.
  * All compile methods throws CompilationException when the script can not be compiled
  */
@WeaveApi(Seq("Bat", "Studio", "data-weave-agent"))
class DataWeaveScriptingEngine(componentsFactory: ModuleComponentsFactory, parsingConfiguration: ParserConfiguration, val configuration: Properties) {

  private var profileParsing: Boolean = false
  private var commonSubExpressionElimination: Boolean = true
  private var debuggerExecutor: WeaveDebuggerExecutor = _
  private var debuggerPort = TcpServerProtocol.DEFAULT_PORT
  private var debug = false
  private var myWorkingDirectory: File = _
  private var myLoggingService: Option[LoggingService] = None
  private var myLoggingConfigurationSourceProvider: Option[LoggingConfigurationSourceProvider] = None
  private var parsingMetricsListener: ParsingMetricsCollectorListener = _

  def this() {
    this(DynamicModuleComponentFactory(), ParserConfiguration(), System.getProperties)
  }

  def this(componentsFactory: ModuleComponentsFactory, parsingConfiguration: ParserConfiguration) {
    this(componentsFactory, parsingConfiguration, System.getProperties)
  }

  def this(componentsFactory: ModuleComponentsFactory) {
    this(componentsFactory, new ParserConfiguration(), System.getProperties)
  }

  def debugPort(portNumber: Int): Unit = {
    this.debuggerPort = portNumber
  }

  private def startDebugSession(): Unit = {
    val session: DefaultWeaveDebuggingSession = new DefaultWeaveDebuggingSession(TcpServerProtocol(debuggerPort))
    debuggerExecutor = new WeaveDebuggerExecutor(session)
    session.start(debuggerExecutor)
  }

  def enableDebug(): Unit = {
    debug = true
  }

  def isDebugEnable(): Boolean = {
    debug
  }

  /**
    * Validates an script it can be either a module or a mapping file and returns a Result object that reflect
    *
    * @param weaveFile The file to be validates
    * @param config    The configuration where the scope and implicit inputs can be specified
    * @return The Validation result with the status and the messages
    */
  def validate(weaveFile: WeaveFile, config: ValidationConfiguration): ValidationResult = {
    val documentParser: DocumentParser = new DocumentParser()
    val components: ModuleComponents = componentsFactory.createComponents()
    val resource: WeaveResource = DefaultWeaveResource(weaveFile.url, weaveFile.content)
    val parsingContext: ParsingContext = createParsingContext(weaveFile.nameIdentifier, components.parser, dataWeaveVersion = config.version)
    config.implicitInputs.foreach((input) => {
      parsingContext.addImplicitInput(input.name, input.weaveType)
    })
    val result = {
      if (config.phase == ValidationPhase.PARSE) {
        documentParser.parse(resource, parsingContext)
      } else if (config.phase == ValidationPhase.SCOPE) {
        documentParser.runScopePhases(resource, parsingContext)
      } else {
        documentParser.runAllPhases(resource, parsingContext)
      }
    }
    val successful = result.hasResult()
    val errors = result.errorMessages().map((pair) => ValidationMessage(pair._1, pair._2)).toArray
    val warnings = result.warningMessages().map((pair) => ValidationMessage(pair._1, pair._2)).toArray
    ValidationResult(successful, errors, warnings)
  }

  /**
    * Specifies the directory where all compiled scripts are going to be used as their parent folder for their working directory
    *
    * @param workingDirectory The Engine Working Directory
    * @return The Engine with the working directory modified
    */
  def withWorkingDirectory(workingDirectory: File): DataWeaveScriptingEngine = {
    this.myWorkingDirectory = workingDirectory
    this
  }

  def workingDirectory(): File = {
    this.myWorkingDirectory
  }

  def debugExecutor(): WeaveDebuggerExecutor = {
    if (debuggerExecutor == null) {
      startDebugSession()
    }
    debuggerExecutor
  }

  /**
    * Disables the debugging. For all scripts
    */
  def disableDebug(): DataWeaveScriptingEngine = {
    debug = false
    this
  }

  /**
    * Configures the logging service to be used
    *
    * @param loggingService The Logging Service to be used
    * @return This instance with the logging service modified
    */
  def withLoggingService(loggingService: LoggingService): DataWeaveScriptingEngine = {
    this.myLoggingService = Option(loggingService)
    this
  }

  /**
    * Configures the source provider to be used by the logging configuration service.
    * The default source provider looks at "classpath://log-config.dwl"
    *
    * @param loggingConfigurationSourceProvider
    * @return
    */
  def withLoggingConfigurationSourceProvider(loggingConfigurationSourceProvider: LoggingConfigurationSourceProvider): DataWeaveScriptingEngine = {
    this.myLoggingConfigurationSourceProvider = Option(loggingConfigurationSourceProvider)
    this
  }

  /**
    * The current logging service
    */
  def loggingService(): Option[LoggingService] = {
    this.myLoggingService
  }

  /**
    * Enable Parsing Profile information to be printed
    *
    * @return
    */
  def enableProfileParsing(): DataWeaveScriptingEngine = {
    profileParsing = true
    this
  }

  /**
    * Disable Parsing Profile information to be printed
    *
    * @return
    */
  def disableProfileParsing(): DataWeaveScriptingEngine = {
    profileParsing = false
    this
  }

  /**
    * Disable Common SubExpression Elimination
    */
  def disableCommonSubExpressionElimination(): DataWeaveScriptingEngine = {
    this.commonSubExpressionElimination = false
    this
  }

  /**
    * Enable Common SubExpression Elimination
    */
  def enableCommonSubExpressionElimination(): DataWeaveScriptingEngine = {
    this.commonSubExpressionElimination = true
    this
  }

  private def compile(
    source: WeaveResource,
    identifier: NameIdentifier,
    implicitInputs: Array[InputType],
    defaultOutputMimeType: String,
    defaultWriterProperties: Map[String, Any],
    maxTime: Long,
    dataWeaveVersion: Option[DataWeaveVersion]): DataWeaveScript = {
    val components = componentsFactory.createComponents()
    val parsingContext: ParsingContext = createParsingContext(identifier, components.parser, maxTime, dataWeaveVersion = dataWeaveVersion)
    implicitInputs.foreach((input) => {
      parsingContext.addImplicitInput(input.name, input.weaveType)
    })

    parsingConfiguration.implicitImports.foreach((moduleName) => {
      parsingContext.addImplicitImport(moduleName)
    })

    parsingConfiguration.parsingAnnotationProcessors.foreach(annotationRegistry => {
      parsingContext.registerAnnotationProcessor(annotationRegistry._1, annotationRegistry._2)
    })

    if (!commonSubExpressionElimination) {
      parsingContext.disableCommonSubExpressionElimination()
    }

    val value: PhaseResult[CompilationResult[DocumentNode]] = {
      source match {
        case binaryWeaveResource: BinaryWeaveResource => WeaveCompiler.compileBinary(binaryWeaveResource, parsingContext, components.compiler)
        case _ =>
          if (parsingConfiguration.skipVerifications) {
            parsingContext.skipVerification = parsingConfiguration.skipVerifications
            WeaveCompiler.compileWithNoCheck(source, parsingContext, components.compiler)
          } else {
            WeaveCompiler.compile(source, parsingContext, components.compiler)
          }
      }
    }

    value
      .warningMessages()
      .foreach((warn) => {
        myLoggingService.map(_.logWarn(Message.toMessageString(warn._1, warn._2)))
      })
    new DataWeaveScript(
      value.getResult().executable,
      identifier,
      this,
      components,
      myLoggingService,
      myLoggingConfigurationSourceProvider,
      Option(myWorkingDirectory),
      defaultOutputMimeType,
      defaultWriterProperties,
      dataWeaveVersion).maxTime(maxTime)
  }

  private def createParsingContext(identifier: NameIdentifier, parserManager: ModuleParsingPhasesManager, maxTime: Long = -1L, dataWeaveVersion: Option[DataWeaveVersion] = None): ParsingContext = {
    val parsingContext = ParsingContext(identifier, parserManager, errorTrace = 1, attachDocumentation = false)
    if (profileParsing) {
      parsingMetricsListener = new ParsingMetricsCollectorListener
      parsingContext.notificationManager.addListener(parsingMetricsListener)
    }
    if (maxTime > -1L) {
      parsingContext.notificationManager.addListener(new WatchdogParsingListener(maxTime))
    }

    if (dataWeaveVersion.isDefined) {
      parsingContext.withLanguageLevel(dataWeaveVersion.get.asSVersion())
    }
    parsingContext
  }

  /**
    * Returns the Parsing Metrics information. It provides detailed information about the performance of parsing the script
    *
    * @return The Parsing Metrics or null if enableProfileParsing was not activated
    */
  def compilationMetrics(): CompilationMetrics = {
    if (parsingMetricsListener != null) {
      parsingMetricsListener.metrics()
    } else {
      null
    }
  }

  /**
    * Returns the inferred type of the given script
    *
    * @param script The script text
    * @return The result type of the result of the script if any
    */
  def inferTypeOf(script: String): Option[WeaveType] = {
    val components = componentsFactory.createComponents()
    val parsingContext: ParsingContext = createParsingContext(NameIdentifier.ANONYMOUS_NAME, components.parser)
    val mayBeParserResult = MappingParser.parse(MappingParser.typeCheckPhase(), WeaveResourceFactory.fromContent(script), parsingContext)
    mayBeParserResult.mayBeResult.flatMap((result) => {
      result.typeGraph.findNode(result.astNode.root).flatMap(_.resultType())
    })
  }

  /**
    * Creates a new CompilationConfig
    *
    * @return A new compilation configuration
    */
  def newConfig() = new CompilationConfig(componentsFactory)

  /**
    * Compile with the given Compilation Config
    *
    * @param config The config data to be used to compile
    * @return The compiled Script
    */
  def compileWith(config: CompilationConfig): DataWeaveScript = {
    compile(
      config.getResource(),
      config.getIdentifier(),
      config.getInputs(),
      config.getDefaultOutputMimeType(),
      config.getDefaultWriterProperties(),
      config.getMaxTime(),
      config.getVersion())
  }

  /**
    * Compile the specified file
    *
    * @param file The source to be used
    * @return The compiled result or an exception
    */
  def compile(file: File): DataWeaveScript = {
    val config = newConfig().withFile(file)
    compileWith(config)
  }

  /**
    * Compile the specified file
    *
    * @param file           The source to be used
    * @param implicitInputs The implicit inputs to be used
    * @return The compiled result or an exception
    */
  def compile(file: File, implicitInputs: Array[InputType]): DataWeaveScript = {
    val config = newConfig().withFile(file).withInputs(implicitInputs)
    compileWith(config)
  }

  /**
    * Compile the script with the given NameIdentifier. The NameIdentifier will be used to lookup the Underlying resource
    * in the resource resolver specified in the ComponentsFactory
    *
    * @param nameIdentifier The source to be used
    * @param implicitInputs The implicit inputs to be used
    * @return The compiled result or an exception
    */
  def compile(nameIdentifier: NameIdentifier, implicitInputs: Array[InputType]): DataWeaveScript = {
    val config = newConfig()
      .withNameIdentifier(nameIdentifier)
      .withInputs(implicitInputs)
    compileWith(config)
  }

  /**
    * Compile the script with the given NameIdentifier. The NameIdentifier will be used to lookup the Underlying resource
    * in the resource resolver specified in the ComponentsFactory
    *
    * @param nameIdentifier The name Identifier of the Script to be loaded
    * @return The CompiledScript
    */
  def compile(nameIdentifier: NameIdentifier): DataWeaveScript = {
    val config = newConfig().withNameIdentifier(nameIdentifier)
    compileWith(config)
  }

  /**
    * Compile the specified file
    *
    * @param script         The source of the script
    * @param nameIdentifier The logical name of the script
    * @param implicitInputs The implicit inputs to be used
    * @return The compiled result or an exception
    */
  def compile(script: String, nameIdentifier: NameIdentifier, implicitInputs: Array[InputType]): DataWeaveScript = {
    val config = newConfig()
      .withScript(script)
      .withNameIdentifier(nameIdentifier)
      .withInputs(implicitInputs)
    compileWith(config)
  }

  /**
    * Compile the specified file
    *
    * @param script         The source of the script
    * @param nameIdentifier The logical name of the script
    * @param implicitInputs The implicit inputs to be used
    * @return The compiled result or an exception
    */
  def compile(script: String, nameIdentifier: NameIdentifier, implicitInputs: Array[InputType], defaultOutputMimeType: String): DataWeaveScript = {
    val config = newConfig()
      .withScript(script)
      .withNameIdentifier(nameIdentifier)
      .withInputs(implicitInputs)
      .withDefaultOutputType(defaultOutputMimeType)
    compileWith(config)
  }

  /**
    * Compile the specified file
    *
    * @param script         The source of the script
    * @param nameIdentifier The logical name of the script
    * @param implicitInputs The implicit inputs to be used
    * @return The compiled result or an exception
    */
  def compile(script: String, nameIdentifier: NameIdentifier, implicitInputs: Array[InputType], defaultOutputMimeType: String, defaultWriterProperties: Map[String, Any]): DataWeaveScript = {
    val config = newConfig()
      .withScript(script)
      .withNameIdentifier(nameIdentifier)
      .withInputs(implicitInputs)
      .withDefaultOutputType(defaultOutputMimeType)
      .withDefaultWriterProperties(defaultWriterProperties)
    compileWith(config)
  }

  /**
    * Compile the specified file
    *
    * @param file           The source to be used
    * @param implicitInputs The implicit inputs to be used
    * @return The compiled result or an exception
    */
  def compile(file: File, implicitInputs: Array[String]): DataWeaveScript = {
    val config = newConfig()
      .withFile(file)
      .withInputs(implicitInputs.map(new InputType(_, None)))
    compileWith(config)
  }

  /**
    * Compile the specified file
    *
    * @param url            The source to be used
    * @param implicitInputs The implicit inputs to be used
    * @return The compiled result or an exception
    */
  def compile(url: URL, implicitInputs: Array[String]): DataWeaveScript = {
    val config = newConfig()
      .withUrl(url)
      .withInputs(implicitInputs)
    compileWith(config)
  }

  /**
    * Compile the specified script and the specified NameIdentifier
    *
    * @param script Compiles the specified script
    * @return The compiled result or an exception
    */
  def compile(script: String, nameIdentifier: String): DataWeaveScript = {
    val config = newConfig()
      .withScript(script)
      .withNameIdentifier(nameIdentifier)
    compileWith(config)
  }

  /**
    * Compile the specified script and the specified NameIdentifier
    *
    * @param script         Compiles the specified script
    * @param nameIdentifier The NameIdentifier of the script to be compiled
    * @return The compiled result or an exception
    */
  def compile(script: String, nameIdentifier: NameIdentifier): DataWeaveScript = {
    val config = newConfig()
      .withScript(script)
      .withNameIdentifier(nameIdentifier)
    compileWith(config)
  }

  /**
    * Compile the specified script and the specified NameIdentifier
    *
    * @param script         Compiles the specified script
    * @param nameIdentifier The NameIdentifier of the script to be compiled
    * @return The compiled result or an exception
    */
  def compile(script: String, nameIdentifier: NameIdentifier, languageLevel: DataWeaveVersion): DataWeaveScript = {
    val config = newConfig()
      .withScript(script)
      .withNameIdentifier(nameIdentifier)
      .withLanguageVersion(languageLevel)
    compileWith(config)
  }

  /**
    * Compile the specified file
    *
    * @param script Compiles the specified script
    * @return The compiled result or an exception
    */
  def compile(script: String): DataWeaveScript = {
    val config = newConfig().withScript(script)
    compileWith(config)
  }

  /**
    * Compile the given script
    *
    * @param script         the script text
    * @param implicitInputs The implicit inputs
    * @return
    */
  def compile(script: String, implicitInputs: Array[String]): DataWeaveScript = {
    val config = newConfig()
      .withScript(script)
      .withInputs(implicitInputs)
    compileWith(config)
  }

}

object DataWeaveScriptingEngine {

  def apply(componentsFactory: ModuleComponentsFactory, parsingConfiguration: ParserConfiguration): DataWeaveScriptingEngine = {
    new DataWeaveScriptingEngine(componentsFactory, parsingConfiguration)
  }

  def apply(componentsFactory: ModuleComponentsFactory): DataWeaveScriptingEngine = {
    new DataWeaveScriptingEngine(componentsFactory, ParserConfiguration())
  }

  /**
    * Creates a new instance of a DataWeaveScriptingEngine
    *
    * @return
    */
  def apply(): DataWeaveScriptingEngine = {
    new DataWeaveScriptingEngine()
  }

  def write(script: String, bindings: ScriptingBindings): DataWeaveResult = {
    val engine = DataWeaveScriptingEngine()
    val weaveScript = engine.compile(script, bindings.entries().toArray)
    weaveScript.write(bindings)
  }

}

/**
  * Represents a file with a logical name (NameIdentifier) a physical name (url) and a content
  *
  * @param nameIdentifier The logical name of the script
  * @param url            The url of the file
  * @param content        The content
  */
case class WeaveFile(nameIdentifier: NameIdentifier, url: String, content: String)

/**
  * The result of a validation process
  *
  * @param successful If the validation was ok
  * @param errors     The errors
  * @param warnings   The warnings
  */
case class ValidationResult(successful: Boolean, errors: Array[ValidationMessage], warnings: Array[ValidationMessage])

/**
  * The configuration for the validation process
  *
  * @param phase          1 for parse 2 for scope 3 for typecheck
  * @param implicitInputs Any implicit input defined for a mapping
  */
case class ValidationConfiguration(phase: Int = ValidationPhase.TYPECHECK, implicitInputs: Array[InputType] = Array.empty, version: Option[DataWeaveVersion] = None)

/**
  * A validation message that has the message and the location
  *
  * @param weaveLocation The location of the message
  * @param message       The message
  */
case class ValidationMessage(weaveLocation: WeaveLocation, message: Message)

object ValidationPhase {
  val PARSE = 1
  val SCOPE = 2
  val TYPECHECK = 3
}

/**
  * Defines an Script Input
  *
  * @param name      The name of the input
  * @param weaveType The type of this input
  */
class InputType(val name: String, val weaveType: Option[WeaveType]) {}

object InputType {
  def apply(name: String, weaveType: Option[WeaveType]): InputType = new InputType(name, weaveType)
}

/**
  * The result of a scripting execution
  */
@WeaveApi(Seq("data-weave-agent"))
class DataWeaveResult(content: Any, charset: Charset, binary: Boolean, mimeType: String, extension: String) {

  def getContent(): Any = content

  def getCharset(): Charset = charset

  def isBinary(): Boolean = binary

  def getMimeType(): String = mimeType

  def getExtension(): String = extension

  def getContentAsString(): String = {
    content match {
      case cp: InputStream =>
        val source = Source.fromInputStream(cp, charset.name())
        try {
          source.mkString
        } finally {
          source.close()
        }
      case resultValue => String.valueOf(resultValue)
    }
  }
}

/**
  * Represents a compiled script that can be executed.
  */
@WeaveApi(Seq("data-weave-agent"))
class DataWeaveScript(
  executable: ExecutableWeave[DocumentNode],
  nameIdentifier: NameIdentifier,
  se: DataWeaveScriptingEngine,
  components: ModuleComponents,
  loggingService: Option[LoggingService],
  loggingConfigurationSourceProvider: Option[LoggingConfigurationSourceProvider],
  containerWorkingDirectory: Option[File] = None,
  defaultOutputMimeType: String = "application/dw",
  defaultWriterProperties: Map[String, Any] = Map(),
  dataWeaveVersion: Option[DataWeaveVersion]) {

  private val configuration: Properties = new Properties(se.configuration)
  private val settingsService: SimpleSettingsService = SimpleSettingsService(PropertiesSettings(configuration), resolveLanguageLevelService)
  private var debugEnabled: Boolean = false
  private var wd: Option[File] = None
  private var maxTimeValue: Long = 0

  private val loaderService: WeaveBasedDataFormatExtensionLoaderService = WeaveBasedDataFormatExtensionLoaderService(ParsingContextCreator(components.parser), components.resourceResolver, components.compiler)

  def enableDebug(): DataWeaveScript = {
    debugEnabled = true
    this
  }

  def enableTelemetry(): DataWeaveScript = {
    configuration.setProperty(RuntimeSettings.ENABLE_TELEMETRY, true.toString)
    this
  }

  def enableMemoryTelemetry(): DataWeaveScript = {
    configuration.setProperty(RuntimeSettings.ENABLE_TELEMETRY_MEMORY, true.toString)
    this
  }

  def disableMemoryTelemetry(): DataWeaveScript = {
    configuration.setProperty(RuntimeSettings.ENABLE_TELEMETRY_MEMORY, false.toString)
    this
  }

  def disableTelemetry(): DataWeaveScript = {
    configuration.setProperty(RuntimeSettings.ENABLE_TELEMETRY, false.toString)
    this
  }

  def setProperty(property: String, value: Boolean): DataWeaveScript = {
    configuration.setProperty(property, value.toString)
    this
  }

  def setProperty(property: String, value: Number): DataWeaveScript = {
    configuration.setProperty(property, value.toString)
    this
  }

  def setProperty(property: String, value: String): DataWeaveScript = {
    configuration.setProperty(property, value)
    this
  }

  def withWorkingDirectory(wd: File): DataWeaveScript = {
    this.wd = Some(wd)
    this
  }

  private def resolveWorkingDirectory(): File = {
    if (this.wd.isDefined) {
      Files.createDirectories(this.wd.get.toPath).toFile
    } else if (containerWorkingDirectory.isDefined) {
      Files.createDirectories(this.containerWorkingDirectory.get.toPath).toFile
    } else {
      FileHelper.tmpDir()
    }
  }

  def workingDirectory(): Option[File] = {
    this.wd
  }

  def disableDebug(): DataWeaveScript = {
    debugEnabled = false
    this
  }

  def declaredOutputMimeType(): Option[String] = {
    executable.declaredOutputMimeType
  }

  def debugPort(portNumber: Int): DataWeaveScript = {
    se.debugPort(portNumber)
    this
  }

  def maxTime(maxTime: Long): DataWeaveScript = {
    this.maxTimeValue = maxTime
    this.executable.withMaxTime(maxTime)
    this
  }

  def addExecutionListener(listener: WeaveExecutionListener): DataWeaveScript = {
    executable.addExecutionListener(listener)
    this
  }

  def materializeValues(force: Boolean): DataWeaveScript = {
    executable.materializedValuesExecution(force)
    this
  }

  def write(bindings: ScriptingBindings, serviceManager: ServiceManager): DataWeaveResult = {
    doWrite(bindings, serviceManager)
  }

  def write(bindings: ScriptingBindings): DataWeaveResult = {
    doWrite(bindings, ServiceManager(EmptyWeaveServicesProvider))
  }

  def write(bindings: ScriptingBindings, serviceManager: ServiceManager, outputMimeType: String): DataWeaveResult = {
    doWrite(bindings, serviceManager, Option(outputMimeType))
  }

  def write(bindings: ScriptingBindings, serviceManager: ServiceManager, outputMimeType: String, properties: Map[String, Any]): DataWeaveResult = {
    doWrite(bindings, serviceManager, Option(outputMimeType), properties)
  }

  def write(bindings: ScriptingBindings, serviceManager: ServiceManager, outputMimeType: String, target: Option[Any]): DataWeaveResult = {
    doWrite(bindings, serviceManager, Option(outputMimeType), Map(), target)
  }

  def write(bindings: ScriptingBindings, serviceManager: ServiceManager, outputMimeType: String, properties: Map[String, Any], target: Option[Any]): DataWeaveResult = {
    doWrite(bindings, serviceManager, Option(outputMimeType), properties, target)
  }

  def write(bindings: ScriptingBindings, serviceManager: ServiceManager, target: Option[Any]): DataWeaveResult = {
    doWrite(bindings, serviceManager, None, Map(), target)
  }

  private def doWrite(bindings: ScriptingBindings, sm: ServiceManager, outputType: Option[String] = None, writerProperties: Map[String, Any] = Map(), target: Option[Any] = None): DataWeaveResult = {

    val newServiceManager: ServiceManager = createResolvedServiceManager(sm)
    implicit val ctx: EvaluationContext = EvaluationContext(newServiceManager)

    val writer: Writer = outputType
      .flatMap((mimeType) => {
        DataFormatManager
          .byContentType(mimeType)
          .map(_.writer(target))
          .map((writer) => {
            ConfigurationHelper.configure(writer, writerProperties)
          })
      })
      .orElse({
        if (executable.declaredOutput().isDefined) {
          executable.implicitWriterOption(target)
        } else {
          None
        }
      })
      .orElse({
        DataFormatManager
          .byContentType(defaultOutputMimeType)
          .map(_.writer(target))
          .map((writer) => {
            ConfigurationHelper.configure(writer, defaultWriterProperties)
          })
      })
      .getOrElse(throw new RuntimeException("Writer was not specified. Please declare: output <mimeType> in your script file."))

    val readers: mutable.Map[String, Reader] = createReaders(bindings, writer.supportsStreaming)

    val telemetryEnabled = settingsService.telemetry().enabled
    if (telemetryEnabled) {
      executable.addExecutionListener(new ExecutionTelemetryListener)
    }
    try {
      if (shouldDebug) {
        val executor: WeaveDebuggerExecutor = se.debugExecutor()
        //Register debugger to this executable!
        executable.addExecutionListener(executor)
        executable.materializedValuesExecution(true)

        val promise = new FutureValue[DataWeaveResult]()
        //Register to this session it will wait for the client or if it is already connected it will be
        //called immediately
        executor.session.addSessionListener(() => {
          val executorService = Executors.newSingleThreadExecutor();
          executorService.submit(new Runnable {
            override def run(): Unit = {
              val result: (Any, TypeAliases#JCharset) = executable.writeWith(writer, readers.toMap, bindings.literalValues.toMap)
              val writerDataFormat: Option[DataFormat[_, _]] = writer.dataFormat
              val mimeType: String = outputType.orElse(writerDataFormat.map(_.defaultMimeType.toString)).getOrElse(defaultOutputMimeType)
              val extension: String = writerDataFormat.flatMap(_.fileExtensions.headOption).getOrElse(".txt")
              promise.set(Some(new DataWeaveResult(result._1, result._2, writerDataFormat.exists(_.binaryFormat), mimeType, extension)))
            }
          })
        })
        promise.get.get
      } else {
        val result = executable.writeWith(writer, readers.toMap, bindings.literalValues.toMap)
        val writerDataFormat = writer.dataFormat
        val mimeType = outputType.orElse(writerDataFormat.map(_.defaultMimeType.toString)).getOrElse(defaultOutputMimeType)
        val extension: String = writerDataFormat.flatMap(_.fileExtensions.headOption).getOrElse(".txt")
        new DataWeaveResult(result._1, result._2, writerDataFormat.exists(_.binaryFormat), mimeType, extension)
      }
    } finally {
      if (telemetryEnabled) {
        ctx.serviceManager.telemetryService.foreach(_.close())
      }
    }
  }

  private def createResolvedServiceManager(sm: ServiceManager): ServiceManager = {
    val customServices: Map[Class[_], Any] = Map(
      (classOf[DataFormatExtensionsLoaderService], CompositeDataFormatExtensionsLoaderService(DefaultDataFormatExtensionsLoaderService, loaderService)),
      (classOf[WorkingDirectoryService], CustomWorkingDirectoryService(resolveWorkingDirectory(), settingsService)),
      (classOf[SettingsService], settingsService),
      (classOf[LanguageLevelService], resolveLanguageLevelService),
      (classOf[LoggingConfigurationService], loggingConfigurationSourceProvider.map(new DefaultLoggingConfigurationService(_)).getOrElse(DefaultLoggingConfigurationService)),
      (classOf[CpuLimitService], if (maxTimeValue > 0) new DefaultCpuLimitService(settingsService) else NoOpCpuLimitService),
      (classOf[PatternService], (if (maxTimeValue > 0) new CpuLimitedPatternService() else DefaultPatternService)) //Pattern Service to limit Pattern Matching
    )
    //This creates a Delegation chain of lookup services
    //First it will look the user provided services
    //Then it will look at the custom services
    // And last it will look at the SPI Based Service
    val newServiceManager = ServiceManager(
      loggingService.getOrElse(sm.logger),
      sm.resource,
      WeaveServicesProvider(
        WeaveServicesProvider(
          sm.serviceProvider,
          WeaveServicesProvider(customServices) /**/
        ), /**/
        new SPIWeaveServicesProvider() /**/
      ))
    newServiceManager
  }

  private def resolveLanguageLevelService = {
    dataWeaveVersion.map(WeaveLanguageLevelService(_)).getOrElse(DefaultLanguageLevelService)
  }

  private def shouldDebug: Boolean = {
    debugEnabled || se.isDebugEnable
  }

  def exec(bindings: ScriptingBindings, serviceManager: ServiceManager): ExecuteResult = {
    doExec(bindings, serviceManager)
  }

  def exec(bindings: ScriptingBindings): ExecuteResult = {
    doExec(bindings, ServiceManager(EmptyWeaveServicesProvider))
  }

  private def doExec(bindings: ScriptingBindings, sm: ServiceManager): ExecuteResult = {
    val newServiceManager: ServiceManager = createResolvedServiceManager(sm)
    implicit val ctx: EvaluationContext = EvaluationContext(newServiceManager)
    ctx.newAsyncExecution()

    try {
      val readers = createReaders(bindings, writerSupportsStreaming = false)
      val listener: ExecutionTelemetryListener = new ExecutionTelemetryListener
      if (settingsService.telemetry().enabled) {
        executable.addExecutionListener(listener)
      }

      if (shouldDebug) {
        val executor: WeaveDebuggerExecutor = se.debugExecutor()
        //Register debugger to this executable!
        executable.addExecutionListener(executor)
        executable.materializedValuesExecution(true)

        val promise = new FutureValue[Value[_]]()
        //Register to this session it will wait for the client or if it is already connected it will be
        //called immediately
        executor.session.addSessionListener(() => {
          val executorService = Executors.newSingleThreadExecutor();
          executorService.submit(new Runnable {
            override def run(): Unit = {
              val result = executable.execute(readers.toMap, bindings.literalValues.toMap)
              promise.set(Some(result))
            }
          })
        })
        //this close won't actually close the context, just decrement de async count, it'll be up to the caller to close it
        val value = promise.get.get
        ExecuteResult(value, ctx)
      } else {

        ExecuteResult(executable.execute(readers.toMap, bindings.literalValues.toMap), ctx)
      }
    } finally {
      // this close won't actually close the context, just decrement de async count,
      // it'll be up to the caller to close it
      ctx.close()
    }
  }

  private def createReaders(bindings: ScriptingBindings, writerSupportsStreaming: Boolean)(implicit ctx: EvaluationContext): mutable.Map[String, Reader] = {
    val readers = bindings.bindings.map((bind) => {
      val bvalue: BindingValue = bind._2
      val bname: String = bind._1
      val maybeInput: Option[DataFormat[_, _]] = executable.declaredInputs().get(bname)
      val mimeType: String = maybeInput
        .map(_.defaultMimeType.toString)
        .orElse(bvalue.mimeType)
        .getOrElse(throw new ScriptingEngineSetupException(s"Unable to detect reader type for `${bname}`, as no MimeType was set. Please declare the input directive i.e. input `${bname}` application/xml"))

      val maybeFormat = DataFormatManager.byContentType(mimeType)
      val reader = maybeFormat match {
        case Some(dataFormat) => {
          val reader = dataFormat.reader(SourceProvider(bvalue.value, bvalue.charset, Some(MimeType.fromSimpleString(mimeType))))
          bvalue.properties.foreach((property) => {
            reader.setOption(UnknownLocation, property._1, property._2)
          })

          reader.settings match {
            case _: ConfigurableStreaming if (writerSupportsStreaming && !shouldDebug) => { //When debugging we should not turn in auto stream
              val maybeInputDirective = AstNodeHelper.getInputs(executable.astDocument()).find(_.variable.name.equals(bind._1))
              val mayBeAnnotation = maybeInputDirective.flatMap(_.variable.annotation(classOf[StreamingCapableVariableAnnotation]))
              mayBeAnnotation match {
                case Some(streamingAnnotation) => {
                  if (streamingAnnotation.canStream) {
                    reader.setOption(UnknownLocation, ConfigurableStreaming.STREAMING_PROP_NAME, true)
                  }
                }
                case _ =>
              }
            }
            case _ =>
          }

          reader
        }
        case None => {
          throw new ScriptingEngineSetupException(s"Data Format not supported ${mimeType} for input ${bname}.")
        }
      }

      (bname, reader)
    })
    readers
  }
}

import scala.collection.JavaConverters._

@WeaveApi(Seq("data-weave-agent"))
class ScriptingBindings {

  val literalValues: mutable.Map[String, Value[_]] = mutable.Map[String, Value[_]]()

  val bindings: mutable.Map[String, BindingValue] = mutable.Map[String, BindingValue]()

  def entries(): collection.Set[String] = {
    bindings.keySet ++ literalValues.keySet
  }

  /**
    * Return the list of names of all the bindings
    *
    * @return
    */
  def bindingNames(): Array[String] = {
    entries().toArray
  }

  /**
    * Adds a binding with a given name
    *
    * @param name       The name of the input binding
    * @param value      The value
    * @param mimeType   The mimeType
    * @param properties The reader properties to be used
    * @return The Binding
    */
  def addBinding(name: String, value: AnyRef, mimeType: String, properties: java.util.Map[String, AnyRef]): ScriptingBindings = {
    bindings.+=((name, BindingValue(value, mimeType, properties.asScala.toMap)))
    this
  }

  /**
    * Adds a binding with the given name
    *
    * @param name     The name of the binding
    * @param value    The value of the binding
    * @param mimeType The MimeType of the binding
    * @return
    */
  def addBinding(name: String, value: AnyRef, mimeType: String): ScriptingBindings = {
    bindings.+=((name, BindingValue(value, mimeType, Map.empty)))
    this
  }

  /**
    * Adds a binding with the given name
    *
    * @param name     The name of the binding
    * @param value    The value of the binding
    * @param mimeType The MimeType of the binding
    * @return
    */
  def addBinding(name: String, value: AnyRef, mimeType: Option[String]): ScriptingBindings = {
    bindings.+=((name, new BindingValue(value, mimeType, Map.empty, Charset.forName("UTF-8"))))
    this
  }

  /**
    * Adds a binding with the given name
    *
    * @param name  The name of the binding
    * @param value The value of the binding
    * @return
    */
  def addBinding(name: String, value: AnyRef): ScriptingBindings = {
    bindings.+=((name, new BindingValue(value, None, Map.empty, Charset.forName("UTF-8"))))
    this
  }

  /**
    * Adds a new binding
    *
    * @param name  the name of the binding
    * @param value The bind
    * @return return this to keep chaining
    */
  def addBinding(name: String, value: BindingValue): ScriptingBindings = {
    bindings.+=((name, value))
    this
  }

  def addBinding(name: String, value: Value[_]): ScriptingBindings = {
    literalValues.+=((name, value))
    this
  }

}

/**
  * Represents the result of a script evaluation
  *
  * @param value             The value of the result
  * @param evaluationContext The evaluation context
  */
class ExecuteResult(value: Value[_], evaluationContext: EvaluationContext) extends AutoCloseable {

  private var materializedValue: Value[_] = _

  /**
    * Returns the result
    *
    * @return the result
    */
  def getResult(): Value[_] = {
    value
  }

  /**
    * The result as an Outline Navigator
    *
    * @return
    */
  def asDWValue(): DataWeaveValue = {
    DataWeaveValue(getResultMaterialized())(evaluationContext)
  }

  /**
    * The result of the value but materialized
    *
    * @return
    */
  def getResultMaterialized(): Value[_] = {
    if (materializedValue == null) {
      //We should only materialize once
      materializedValue = value.materialize(evaluationContext)
    }
    materializedValue
  }

  /**
    * Close the context values.
    */
  override def close(): Unit = {
    evaluationContext.close()
  }

}

object ExecuteResult {
  def apply(value: Value[_], evaluationContext: EvaluationContext): ExecuteResult = new ExecuteResult(value, evaluationContext)
}

object ScriptingBindings {
  def apply(): ScriptingBindings = new ScriptingBindings()
}

class BindingValue(val value: AnyRef, val mimeType: Option[String], val properties: Map[String, Any], val charset: Charset)

object BindingValue {
  def apply(value: AnyRef, mimeType: String): BindingValue = new BindingValue(value, Some(mimeType), Map(), Charset.forName("UTF-8"))

  def apply(value: AnyRef, mimeType: String, properties: Map[String, Any]): BindingValue = new BindingValue(value, Some(mimeType), properties, Charset.forName("UTF-8"))

  def apply(value: AnyRef, mimeType: String, properties: Map[String, Any], charset: Charset): BindingValue = new BindingValue(value, Some(mimeType), properties, charset)

  def apply(value: AnyRef, mimeType: Option[String], properties: Map[String, Any], charset: Charset): BindingValue = new BindingValue(value, mimeType, properties, charset)

}

case class ParserConfiguration(
  implicitImports: Seq[String] = Seq(),
  parsingAnnotationProcessors: mutable.Map[NameIdentifier, DWAnnotationProcessor] = mutable.HashMap(),
  skipVerifications: Boolean = false)

class ParserConfigurationBuilder {
  private val imports = new mutable.ArrayBuffer[String]()
  private val annotationProcessors = new mutable.HashMap[NameIdentifier, DWAnnotationProcessor]()

  def addImplicitImport(moduleName: String): ParserConfigurationBuilder = {
    imports.+=(moduleName)
    this
  }

  def addAnnotationProcessor(annotationName: NameIdentifier, processor: DWAnnotationProcessor): ParserConfigurationBuilder = {
    annotationProcessors.put(annotationName, processor)
    this
  }

  def build(): ParserConfiguration = {
    ParserConfiguration(imports, annotationProcessors)
  }
}

class ScriptingEngineSetupException(message: String) extends Exception(message) {}
