package org.mule.weave.v2.interpreted

import org.mule.weave.v2.compilation.exception.WeaveBinaryException
import org.mule.weave.v2.interpreted.node.ModuleNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ModuleParsingPhasesManager
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.sdk.ClassLoaderWeaveResourceResolver
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.mule.weave.v2.utils.LockFactory

import java.util.concurrent.ConcurrentHashMap

/**
  * Handles the transformation to a Executable ModuleNode.
  * It goes from an fully qualified name, look it up and executes the corresponding compilation phases.
  */
trait RuntimeModuleNodeCompiler {
  /**
    * Compiles the module with the specified NameIdentifier
    *
    * @param moduleName     The name of the module to be compiled
    * @param parsingContext The parsing context
    * @return The runtime module node if it was able to successfully compile it.
    */
  def compile(moduleName: NameIdentifier, parsingContext: ParsingContext): Option[ModuleNode]

  /**
    * Compiles the module with the specified NameIdentifier
    *
    * @param moduleName     The name of the module to be compiled
    * @param parsingContext The parsing context
    * @param rootCompiler   The root compiler in the chain of compilers.
    * @return The runtime module node if it was able to successfully compile it.
    */
  def compile(moduleName: NameIdentifier, parsingContext: ParsingContext, rootCompiler: RuntimeModuleNodeCompiler): Option[ModuleNode]

  /**
    * Invalidate a cached entry with the given NameIdentifier
    *
    * @param nameIdentifier The name identifier of the module to invalidate.
    */
  def invalidate(nameIdentifier: NameIdentifier): Unit
}

/**
  * A runtime module compiler
  *
  * @param parserManager If a parser manager is specified then it will use this one to look for resources. Else it will use the one in the parsing context
  */
class DefaultRuntimeModuleNodeCompiler(parserManager: Option[ModuleParsingPhasesManager], weaveResourceResolver: WeaveResourceResolver) extends RuntimeModuleNodeCompiler {

  private val modules = new ConcurrentHashMap[NameIdentifier, ModuleNode]()
  private val modulesLock = LockFactory.createLock()

  def this() = {
    this(None, ClassLoaderWeaveResourceResolver.apply())
  }

  override def compile(moduleName: NameIdentifier, parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): Option[ModuleNode] = {
    def compileFromDwl() = {
      val parsingPhasesManager = parserManager.getOrElse(parsingContext.moduleParserManager)
      val maybeResult = parsingPhasesManager.scopeCheckModule(moduleName, parsingContext)
      maybeResult.map(result => {
        val childCtx = parsingContext.child(moduleName)
        val compilationResult = WeaveCompiler.runtimeModuleCompilation(result.getResult(), childCtx, moduleNodeLoader)
        val engine = compilationResult.getResult().executable
        val moduleNode = engine.asInstanceOf[InterpretedModuleExecutableWeave].executableDocument
        modules.put(moduleName, moduleNode)
        parsingContext.notificationManager.endCompilation(moduleName, cached = false, fromBinary = false, Some(moduleNode))
        moduleNode
      })
    }

    parsingContext.notificationManager.startCompilation(moduleName)
    val maybeModule = modules.get(moduleName)

    if (maybeModule != null) {
      parsingContext.notificationManager.endCompilation(moduleName, cached = true, fromBinary = false, Some(maybeModule))
      Some(maybeModule)
    } else {
      modulesLock.lock(
        moduleName, {
        val maybeModule = modules.get(moduleName)
        if (maybeModule != null) {
          Some(maybeModule)
        } else {
          val maybeBinaryWeaveResource = weaveResourceResolver.resolveBinary(moduleName)
          if (maybeBinaryWeaveResource.isDefined) {
            val childCtx = parsingContext.child(moduleName)
            val compilationResult = WeaveCompiler.compileModuleBinary(moduleName, maybeBinaryWeaveResource.get, childCtx, moduleNodeLoader)
            if (compilationResult.hasResult() && compilationResult.noErrors()) {
              val engine = compilationResult.getResult().executable
              val moduleNode = engine.asInstanceOf[InterpretedModuleExecutableWeave].executableDocument
              modules.put(moduleName, moduleNode)
              parsingContext.notificationManager.endCompilation(moduleName, cached = false, fromBinary = true, Some(moduleNode))
              Some(moduleNode)
            } else {
              //TODO: do we want to skip slow path if binary compilation raised an error not related to deserialization?
              //      in that case that error should happen again here.
              val binaryCompilationErrors = childCtx.messageCollector.errorMessages.filter(e => e._2.isInstanceOf[WeaveBinaryException])
              binaryCompilationErrors.foreach(e => parsingContext.messageCollector.warning(e._2, e._1))
              compileFromDwl()
            }
          } else {
            compileFromDwl()
          }
        }
      })
    }
  }

  def compile(moduleName: NameIdentifier, parsingContext: ParsingContext): Option[ModuleNode] = {
    compile(moduleName, parsingContext, this)
  }

  override def invalidate(nameIdentifier: NameIdentifier): Unit = {
    modules.remove(nameIdentifier)
  }
}

class CompositeRuntimeModuleNodeCompiler(parent: RuntimeModuleNodeCompiler, child: RuntimeModuleNodeCompiler) extends RuntimeModuleNodeCompiler {

  override def compile(moduleName: NameIdentifier, parsingContext: ParsingContext): Option[ModuleNode] = {
    compile(moduleName, parsingContext, this)
  }

  override def compile(moduleName: NameIdentifier, parsingContext: ParsingContext, rootCompiler: RuntimeModuleNodeCompiler): Option[ModuleNode] = {
    parent.compile(moduleName, parsingContext, rootCompiler).orElse(child.compile(moduleName, parsingContext, rootCompiler))
  }

  override def invalidate(nameIdentifier: NameIdentifier): Unit = {
    parent.invalidate(nameIdentifier)
    child.invalidate(nameIdentifier)
  }

}

object CompositeRuntimeModuleNodeCompiler {
  def apply(parent: RuntimeModuleNodeCompiler, child: RuntimeModuleNodeCompiler): CompositeRuntimeModuleNodeCompiler = {
    new CompositeRuntimeModuleNodeCompiler(parent, child)
  }
}

object RuntimeModuleNodeCompiler {

  def apply(): RuntimeModuleNodeCompiler = new DefaultRuntimeModuleNodeCompiler()

  def apply(parserManager: ModuleParsingPhasesManager, weaveResourceResolver: WeaveResourceResolver, parent: Option[RuntimeModuleNodeCompiler] = None): RuntimeModuleNodeCompiler = {
    val compiler = new DefaultRuntimeModuleNodeCompiler(Some(parserManager), weaveResourceResolver)
    parent match {
      case Some(parent) => {
        new CompositeRuntimeModuleNodeCompiler(parent, compiler)
      }
      case _ => {
        compiler
      }
    }
  }

  def parentFirst(parserManager: ModuleParsingPhasesManager, weaveResourceResolver: WeaveResourceResolver, parent: RuntimeModuleNodeCompiler): RuntimeModuleNodeCompiler = {
    RuntimeModuleNodeCompiler(parserManager = parserManager, weaveResourceResolver = weaveResourceResolver, parent = Some(parent))
  }

  def parentLast(parserManager: ModuleParsingPhasesManager, weaveResourceResolver: WeaveResourceResolver, parent: RuntimeModuleNodeCompiler): RuntimeModuleNodeCompiler = {
    val compiler = new DefaultRuntimeModuleNodeCompiler(Some(parserManager), weaveResourceResolver)
    new CompositeRuntimeModuleNodeCompiler(compiler, parent)
  }

  def chain(parserManager: ModuleParsingPhasesManager, weaveResourceResolver: WeaveResourceResolver, parent: RuntimeModuleNodeCompiler, parentLast: Boolean): RuntimeModuleNodeCompiler = {
    if (parentLast) {
      RuntimeModuleNodeCompiler.parentLast(parserManager, weaveResourceResolver, parent)
    } else {
      parentFirst(parserManager, weaveResourceResolver, parent)
    }
  }
}