package org.mule.weave.v2.interpreted.node.updater

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.ExecutionNode
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.structure.function.DynamicFunctionNode
import org.mule.weave.v2.model.capabilities.AttributesCapable
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.NameSeq
import org.mule.weave.v2.model.structure.NameValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.BooleanType
import org.mule.weave.v2.model.types.NameType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.AttributeDelegateValue
import org.mule.weave.v2.model.values.AttributesValue
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.NameValue
import org.mule.weave.v2.model.values.NullValue
import org.mule.weave.v2.model.values.NumberValue
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.coercion.ArrayCoercer
import org.mule.weave.v2.model.values.math
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.parser.ast.WeaveLocationCapable
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.runtime.exception.AmbiguousBranchTypeFoundException

import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

/**
  * This node runs the update the updater expressions are dynamic so it needs to be evaluated each time
  */
class DynamicUpdaterValueNode(value: ValueNode[Any], updater: UpdaterCasesNode) extends ValueNode[Any] with Product2[ValueNode[_], UpdaterCasesNode] {

  override protected def doExecute(implicit ctx: ExecutionContext): Value[Any] = {
    val valueToUpdate: Value[Any] = value.execute
    val valueUpdater: ValueUpdater = updater.toValueUpdater
    valueUpdater.update(valueToUpdate, ctx.executionStack().activeFrame(), 0)
  }

  override def _1: ValueNode[_] = value

  override def _2: UpdaterCasesNode = updater
}

/**
  * This node runs the update all the selector expressions are static so we can cache the updater tree
  */
class StaticUpdaterValueNode(value: ValueNode[Any], updater: UpdaterCasesNode) extends ValueNode[Any] with Product2[ValueNode[_], UpdaterCasesNode] {

  var valueUpdater: ValueUpdater = _

  override protected def doExecute(implicit ctx: ExecutionContext): Value[Any] = {
    val valueToUpdate: Value[Any] = value.execute
    if (valueUpdater == null) {
      valueUpdater = updater.toValueUpdater
    }
    valueUpdater.update(valueToUpdate, ctx.executionStack().activeFrame(), 0)
  }

  override def _1: ValueNode[_] = value

  override def _2: UpdaterCasesNode = updater
}

class UpdaterCasesNode(expressions: Seq[UpdaterCaseNode]) extends ExecutionNode with Product {
  def toValueUpdater(implicit ctx: ExecutionContext): ValueUpdater = {
    val updater = new RootUpdater(this)
    expressions.foreach((uen) => {
      uen.toValueUpdater(updater)
    })
    updater
  }

  override def productElement(n: Int): Any = expressions(n)

  override def productArity: Int = expressions.length

  override def canEqual(that: Any): Boolean = that.isInstanceOf[this.type]
}

sealed trait UpdaterKind {}

object IndexKind extends UpdaterKind

object FieldKind extends UpdaterKind

object MultiFieldKind extends UpdaterKind

object AttributeKind extends UpdaterKind

case class UpdaterCaseNode(valueNode: UpdaterExpressionElementNode, updater: DynamicFunctionNode, maybeConditionFunction: Option[DynamicFunctionNode], forceCreate: Boolean) extends ExecutionNode with Product2[UpdaterExpressionElementNode, DynamicFunctionNode] {
  override def _1: UpdaterExpressionElementNode = valueNode

  override def _2: DynamicFunctionNode = updater

  def toValueUpdater(parent: ValueUpdater)(implicit ctx: ExecutionContext): ValueUpdater = {

    valueNode
      .toValueUpdater(parent, forceCreate)
      .addTerminalUpdater(updater, maybeConditionFunction, updater)
      .markForceCreateIfRequired(forceCreate)
  }
}

case class UpdaterExpressionElementNode(valueNode: ValueNode[_], updaterKind: UpdaterKind, childNode: Option[UpdaterExpressionElementNode]) extends ExecutionNode with Product3[ValueNode[_], UpdaterKind, Option[UpdaterExpressionElementNode]] {
  override def _1: ValueNode[_] = valueNode

  override def _2: UpdaterKind = updaterKind

  override def _3: Option[UpdaterExpressionElementNode] = childNode

  def toValueUpdater(parent: ValueUpdater, forceCreate: Boolean)(implicit ctx: ExecutionContext): ValueUpdater = {
    val updater = updaterKind match {
      case IndexKind => {
        val index = NumberType.coerce(valueNode.execute, this).evaluate
        parent.addIndexUpdater(index, this)
      }
      case FieldKind => {
        val name = NameType.coerce(valueNode.execute, this).evaluate
        parent.addFieldUpdater(name, this)
      }
      case MultiFieldKind => {
        val name = NameType.coerce(valueNode.execute, this).evaluate
        parent.addMultiFieldUpdater(name, this)
      }
      case AttributeKind => {
        val name = NameType.coerce(valueNode.execute, this).evaluate
        parent.addAttributeUpdater(name, this)
      }
    }

    updater.markForceCreateIfRequired(forceCreate)

    childNode match {
      case Some(value) => {
        value.toValueUpdater(updater, forceCreate)
      }
      case None => {
        updater
      }
    }
  }

}

sealed trait ValueUpdater extends WeaveLocationCapable {

  var forceCreate: Boolean = false

  val children: mutable.ArrayBuffer[ValueUpdater] = mutable.ArrayBuffer()

  private lazy val terminalUpdaters: ArrayBuffer[TerminalUpdater] = {
    children.collect({
      case f: TerminalUpdater => f
    })
  }

  private lazy val terminalUpdaterForceCreate: ArrayBuffer[TerminalUpdater] = terminalUpdaters.filter(_.forceCreate)

  private lazy val indexUpdaters: ArrayBuffer[ArrayIndexUpdater] = children
    .collect({
      case ai: ArrayIndexUpdater => ai
    })
    .sortBy(_.index(Long.MaxValue).intValue())

  private lazy val indexUpdatersForceCreate: Seq[ArrayIndexUpdater] = indexUpdaters.filter(_.forceCreate)

  private lazy val fieldUpdaters: Seq[(FieldUpdater, Int)] = children
    .collect({
      case fi: FieldUpdater => fi
    })
    .zipWithIndex

  private lazy val fieldUpdatersForceCreate: Seq[(FieldUpdater, Int)] = fieldUpdaters.filter(_._1.forceCreate)

  private lazy val attributeUpdaters: Seq[(AttributeUpdater, Int)] = children
    .collect({
      case fi: AttributeUpdater => fi
    })
    .zipWithIndex

  private lazy val attributeUpdatersForceCreate: Seq[(AttributeUpdater, Int)] = attributeUpdaters.filter(_._1.forceCreate)

  //  def copy()(implicit ctx: ExecutionContext): ValueUpdater

  def markForceCreateIfRequired(forceCreate: Boolean): ValueUpdater = {
    if (forceCreate)
      this.forceCreate = forceCreate
    this
  }

  def addTerminalUpdater(binaryFunctionValue: DynamicFunctionNode, condition: Option[DynamicFunctionNode], delegate: WeaveLocationCapable): ValueUpdater = {
    addChild(new TerminalUpdater(binaryFunctionValue, condition, delegate))
  }

  def addMultiFieldUpdater(name: QualifiedName, delegate: WeaveLocationCapable): ValueUpdater = {
    children
      .find({
        case m: FieldUpdater if (m.name.matches(name)) => {
          true
        }
        case _ => false
      })
      .getOrElse({
        addChild(new FieldUpdater(name, true, delegate))
      })
  }

  def addAttributeUpdater(name: QualifiedName, delegate: WeaveLocationCapable): ValueUpdater = {
    children
      .find({
        case m: AttributeUpdater if (m.name.matches(name)) => {
          true
        }
        case _ => false
      })
      .getOrElse({
        addChild(new AttributeUpdater(name, false, delegate))
      })
  }

  def addFieldUpdater(name: QualifiedName, delegate: WeaveLocationCapable): ValueUpdater = {
    children
      .find({
        case m: FieldUpdater if (m.name.matches(name)) => {
          true
        }
        case _ => false
      })
      .getOrElse({
        addChild(new FieldUpdater(name, false, delegate))
      })
  }

  def addIndexUpdater(index: math.Number, delegate: WeaveLocationCapable): ValueUpdater = {
    children
      .find({
        case m: ArrayIndexUpdater if (m.indexValue.equals(index)) => {
          true
        }
        case _ => false
      })
      .getOrElse({
        addChild(new ArrayIndexUpdater(index, delegate))
      })
  }

  def addChild[T <: ValueUpdater](valueUpdater: T): T = {
    children.+=(valueUpdater)
    valueUpdater
  }

  protected def createAttributes(ownerFrame: Frame)(implicit ctx: ExecutionContext): Option[Value[NameSeq]] = {
    val nameValuePairs = attributeUpdatersForceCreate.flatMap(_._1.createAttribute(ownerFrame))
    if (nameValuePairs.isEmpty) {
      None
    } else {
      Some(AttributesValue(nameValuePairs))
    }
  }

  def createValue(ownerFrame: Frame)(implicit ctx: ExecutionContext): Option[Value[_]] = {
    val maybeTerminalValue = if (terminalUpdaterForceCreate.nonEmpty) {
      terminalUpdaterForceCreate.toStream //
        .find((tu) => tu.acceptsCondition(ownerFrame, NullValue, NumberValue(-1)))
        .map((tu) => tu.updater(ownerFrame).call(NullValue, NumberValue(-1))) //
    } else {
      None
    }

    if (maybeTerminalValue.isDefined) {
      maybeTerminalValue
    } else {
      val keyValuePairs: Seq[KeyValuePair] = fieldUpdatersForceCreate.flatMap(_._1.createKeyValuePair(ownerFrame))
      val indexes = indexUpdatersForceCreate.flatMap(_.createValue(ownerFrame))
      if (indexes.nonEmpty && keyValuePairs.nonEmpty) {
        throw new AmbiguousBranchTypeFoundException(indexUpdatersForceCreate.head.location())
      } else if (indexes.nonEmpty) {
        Some(ArrayValue(indexes))
      } else if (keyValuePairs.nonEmpty) {
        Some(ObjectValue(ObjectSeq(keyValuePairs)))
      } else {
        None
      }
    }
  }

  def update(value: Value[_], ownerFrame: Frame, number: Int)(implicit ctx: ExecutionContext): Value[_] = {
    if (terminalUpdaters.isEmpty) {
      doUpdate(value, ownerFrame, number)
    } else {
      val indexNumber = NumberValue(number)
      val maybeUpdater = terminalUpdaters.find((tu) => tu.acceptsCondition(ownerFrame, value, indexNumber))
      if (maybeUpdater.isDefined) {
        maybeUpdater.get.updater(ownerFrame).call(value, indexNumber) //Terminal node
      } else {
        doUpdate(value, ownerFrame, number)
      }
    }
  }

  private def doUpdate(value: Value[_], ownerFrame: Frame, number: Int)(implicit ctx: ExecutionContext): Value[_] = {
    value match {
      case arrV if (ArrayType.accepts(value)) => {
        if (indexUpdaters.isEmpty) {
          value
        } else {
          var updaterIndex = 0
          val arraySeq: ArraySeq = ArrayCoercer.coerceToArraySeq(arrV)

          if (indexUpdaters.last.indexValue < 0 || indexUpdatersForceCreate.nonEmpty) {
            //If negative id then we need to be Seq otherwise use Iterator for better performance
            val arrayValues = arraySeq.materialize().toSeq()
            val updatedValues = arrayValues.zipWithIndex.map((vi) => {
              if (updaterIndex < indexUpdaters.size && indexUpdaters(updaterIndex).index(arrayValues.size).intValue() == vi._2) {
                val updater = indexUpdaters(updaterIndex)
                updaterIndex = updaterIndex + 1
                updater.update(vi._1, ownerFrame, vi._2)
              } else {
                vi._1
              }
            })

            if (updaterIndex != indexUpdaters.size && indexUpdatersForceCreate.nonEmpty) {
              //If not all updaters have been executed and there are some ForceCreate
              val newValues: ArrayBuffer[Value[_]] = indexUpdaters //
                .slice(updaterIndex, indexUpdaters.size) //
                .filter(_.forceCreate) //
                .flatMap(_.createValue(ownerFrame))
              ArrayValue(updatedValues ++ newValues, this)
            } else {
              ArrayValue(updatedValues, this)
            }
          } else {
            val arrayValues = arraySeq.toSeq()
            val iterator = arrayValues.zipWithIndex.map((vi) => {
              //As we know they are all positive values we don't need to the index() function of the updaters and we can directly
              //access the index numbers
              if (updaterIndex < indexUpdaters.size && indexUpdaters(updaterIndex).indexValue.toInt == vi._2) {
                val updater = indexUpdaters(updaterIndex)
                updaterIndex = updaterIndex + 1
                updater.update(vi._1, ownerFrame, vi._2)
              } else {
                vi._1
              }
            })
            ArrayValue(iterator, this)
          }
        }
      }
      case objV if (ObjectType.accepts(value)) => {

        val objectSeq = ObjectType.coerce(objV).evaluate
        if (fieldUpdatersForceCreate.nonEmpty) {
          val alreadyUpdated = new Array[UpdaterOccurrence[FieldUpdater]](fieldUpdaters.size)
          val keyValuePairs: ArrayBuffer[KeyValuePair] = new ArrayBuffer[KeyValuePair]()
          //We can not use map because is lazy and we do side effect with alreadyUpdated :(
          objectSeq
            .materialize()
            .toIterator()
            .foreach((kvp) => {
              keyValuePairs += updateKeyValuePair(kvp, ownerFrame, alreadyUpdated)
            })
          val needToBeCreated = fieldUpdatersForceCreate.filter((tuple) => alreadyUpdated(tuple._2) == null)
          val missingKVP = needToBeCreated.flatMap((tuple) => {
            tuple._1.createKeyValuePair(ownerFrame)
          })
          ObjectValue(ObjectSeq(keyValuePairs ++ missingKVP), this)
        } else {
          val alreadyUpdated = new Array[UpdaterOccurrence[FieldUpdater]](fieldUpdaters.size)
          val mappedObject = objectSeq
            .toIterator()
            .map((kvp) => {
              updateKeyValuePair(kvp, ownerFrame, alreadyUpdated)
            })
          ObjectValue(mappedObject, this)
        }
      }
      case _ => value
    }
  }

  def updateAttributes(value: Value[QualifiedName], ownerFrame: Frame)(implicit ctx: ExecutionContext): Value[QualifiedName] = {
    if (attributeUpdaters.isEmpty) {
      value
    } else {
      val alreadyUpdated = new Array[UpdaterOccurrence[AttributeUpdater]](attributeUpdaters.size)
      val updatedAttrs = value match {
        case key: AttributesCapable => {
          key.attributes
            .map((attrVal) => {
              val attrs: NameSeq = attrVal.evaluate
              val updatedAttrs: ArrayBuffer[NameValuePair] = new ArrayBuffer[NameValuePair]()
              val attributes = attrs.toStream()
              attributes
                .foreach((nvp) => {
                  val attrNameValue = nvp._1.materialize
                  val attrName = attrNameValue.evaluate
                  val possibles = attributeUpdaters.find((tuple) => {
                    if (!tuple._1.repeated) {
                      if (alreadyUpdated(tuple._2) == null) {
                        tuple._1.name.matches(attrName)
                      } else {
                        false
                      }
                    } else {
                      tuple._1.name.matches(attrName)
                    }
                  })
                  val nameValuePair = possibles match {
                    case Some((node, index)) => {
                      if (alreadyUpdated(index) == null) {
                        alreadyUpdated.update(index, UpdaterOccurrence(node))
                      } else {
                        alreadyUpdated(index).occurrence = alreadyUpdated(index).occurrence + 1
                      }
                      NameValuePair(attrNameValue, node.update(nvp._2, ownerFrame, 0))
                    }
                    case None => NameValuePair(attrNameValue, nvp._2)
                  }
                  updatedAttrs += nameValuePair
                })

              val needToBeCreated = attributeUpdatersForceCreate.filter((tuple) => alreadyUpdated(tuple._2) == null)
              val missingKVP: Seq[NameValuePair] = needToBeCreated.flatMap((tuple) => {
                tuple._1.createAttribute(ownerFrame)
              })

              AttributesValue(NameSeq(updatedAttrs ++ missingKVP))
            })
            .orElse({
              //If no attributes are present just create the ones that are force
              val needToBeCreated = attributeUpdatersForceCreate.filter((tuple) => alreadyUpdated(tuple._2) == null)
              val missingKVP: Seq[NameValuePair] = needToBeCreated.flatMap((tuple) => {
                tuple._1.createAttribute(ownerFrame)
              })
              Some(AttributesValue(NameSeq(missingKVP)))
            })
        }
        case _ => None
      }
      KeyValue(value.evaluate, updatedAttrs)
    }
  }

  private def updateKeyValuePair(kvp: KeyValuePair, ownerFrame: Frame, alreadyUpdated: Array[UpdaterOccurrence[FieldUpdater]])(implicit ctx: ExecutionContext): KeyValuePair = {
    val materializedKey = kvp._1.materialize
    val kvpName: QualifiedName = materializedKey.evaluate
    val possibles = fieldUpdaters.find((tuple) => {
      if (!tuple._1.repeated) {
        if (alreadyUpdated(tuple._2) == null) {
          tuple._1.name.matches(kvpName)
        } else {
          false
        }
      } else {
        tuple._1.name.matches(kvpName)
      }
    })

    possibles match {
      case Some((node, index)) => {
        if (alreadyUpdated(index) == null) {
          alreadyUpdated.update(index, UpdaterOccurrence(node))
        } else {
          alreadyUpdated(index).occurrence = alreadyUpdated(index).occurrence + 1
        }
        node.updateKeyValuePair(materializedKey, ownerFrame, kvp._2, alreadyUpdated(index).occurrence)
      }
      case _ => KeyValuePair(materializedKey, kvp._2)
    }
  }
}

case class UpdaterOccurrence[T <: ValueUpdater](updater: T, var occurrence: Int = 0)

class RootUpdater(val delegate: WeaveLocationCapable) extends ValueUpdater {
  override def location(): WeaveLocation = delegate.location()
}

class TerminalUpdater(val updaterFactory: DynamicFunctionNode, val condition: Option[DynamicFunctionNode], val delegate: WeaveLocationCapable) extends ValueUpdater {

  def updater(ownerFrame: Frame)(implicit ctx: ExecutionContext): FunctionValue = {
    ctx.runInFrame(ownerFrame, {
      updaterFactory.execute.asInstanceOf[FunctionValue]
    })
  }

  def hasCondition: Boolean = condition.isDefined

  def acceptsCondition(ownerFrame: Frame, arg1: Value[_], arg2: Value[_])(implicit ctx: ExecutionContext): Boolean = {
    if (hasCondition) {
      ctx.runInFrame(ownerFrame, {
        BooleanType.coerce(condition.get.execute.asInstanceOf[FunctionValue].call(arg1, arg2), this).evaluate
      })
    } else {
      true
    }
  }

}

class ArrayIndexUpdater(val indexValue: Number, val delegate: WeaveLocationCapable) extends ValueUpdater {
  override def location(): WeaveLocation = delegate.location()

  def index(size: Long): Long = {
    val indexLong = indexValue.toLong
    if (indexLong >= 0) {
      indexLong
    } else {
      size + indexLong
    }
  }
}

class AttributeUpdater(val name: QualifiedName, val repeated: Boolean, val delegate: WeaveLocationCapable) extends ValueUpdater {
  def createAttribute(ownerFrame: Frame)(implicit ctx: ExecutionContext): Option[NameValuePair] = {
    createValue(ownerFrame).map((value) => NameValuePair(NameValue(name), value))
  }

  override def location(): WeaveLocation = delegate.location()
}

class FieldUpdater(val name: QualifiedName, val repeated: Boolean, val delegate: WeaveLocationCapable) extends ValueUpdater {

  override def location(): WeaveLocation = delegate.location()

  def createKeyValuePair(ownerFrame: Frame)(implicit ctx: ExecutionContext): Option[KeyValuePair] = {
    val maybeValue = createValue(ownerFrame)
    val maybeAttributes = createAttributes(ownerFrame)
    if (maybeValue.isDefined) {
      Some(KeyValuePair(KeyValue(name, maybeAttributes), maybeValue.get))
    } else if (maybeAttributes.isDefined) {
      Some(KeyValuePair(KeyValue(name, maybeAttributes), NullValue))
    } else {
      None
    }

  }

  def updateKeyValuePair(key: Value[QualifiedName], ownerFrame: Frame, value: Value[_], occurrence: Int)(implicit ctx: ExecutionContext): KeyValuePair = {
    KeyValuePair(updateAttributes(key, ownerFrame), update(AttributeDelegateValue(value, key), ownerFrame, occurrence))
  }
}
