package org.mule.weave.v2.runtime.tools

import org.mule.weave.v2.core.RuntimeConfigProperties
import org.mule.weave.v2.core.versioning.CompatibilityFlag
import org.mule.weave.v2.core.versioning.Setting
import org.mule.weave.v2.core.versioning.SystemSetting
import org.mule.weave.v2.model.service.DefaultSettingsService
import org.mule.weave.v2.utils.DataWeaveVersion

import java.io.BufferedWriter
import java.io.File
import java.io.FileWriter

object SystemPropertyCompatibilityTableDocGenerator {
  def main(args: Array[String]): Unit = {
    if (args.length == 0) {
      println("Missing target directory param or DW version")
      return
    }
    val folder = args(0)
    val targetFolder = new File(folder)
    if (targetFolder.isFile) {
      println("Target should be a directory")
      return
    }

    targetFolder.mkdirs()

    val settings = DefaultSettingsService.settings()
    val systemSettings = settings.flatMap(_.settings()) ++ RuntimeConfigProperties.runtimeConfigProperties
    val targetFile = new File(targetFolder, "dataweave-system-properties-cross-table.adoc")

    println(s"Creating system properties compatibility cross table at ${targetFile.getAbsolutePath}")

    val allSettings = systemSettings.filter(!_.internal).sortBy(_.fullName)
    val summaryTable = crossTableInfo(allSettings)

    val writer = new BufferedWriter(new FileWriter(targetFile))
    try {
      writer.write(summaryTable)
    } finally {
      writer.close()
    }
  }

  private def crossTableInfo(settings: Seq[Setting[_]]): String = {
    def tableRow(setting: Setting[_], range: Seq[DataWeaveVersion]): String = {
      val presentByVersion: String = setting match {
        case flag: CompatibilityFlag =>
          range.map(version => {
            if (flag.creationVersion <= version && flag.deletionVersion > version) {
              "✅"
            } else {
              "❌"
            }
          }).mkString("|")
        case _: SystemSetting[_] => range.map(_ => "✅").mkString("|")
      }

      s"| ${setting.fullName} | $presentByVersion"
    }

    var currentVersion: DataWeaveVersion = DataWeaveVersion(2, 3)
    var range: Seq[DataWeaveVersion] = Seq()

    while (currentVersion <= DataWeaveVersion()) {
      range = range :+ currentVersion
      currentVersion = DataWeaveVersion(currentVersion.major, currentVersion.minor + 1)
    }

    s"""[cols="1,${range.map(_ => "^1").mkString(",")},options="header"]
       ||=====
       |||${range.map(_.toString).mkString("|")}
       |${settings.map(tableRow(_, range)).mkString("\n|")}
       ||=====
       |""".stripMargin
  }

}
