package org.mule.weave.v2.compilation.mapper

import org.mule.weave.v2.compilation.ArraySerializableAstNode
import org.mule.weave.v2.compilation.BooleanSerializableValueAstNode
import org.mule.weave.v2.compilation.IntSerializableValueAstNode
import org.mule.weave.v2.compilation.NoneSerializableValueAstNode
import org.mule.weave.v2.compilation.SerializableAstNode
import org.mule.weave.v2.compilation.StringSerializableValueAstNode
import org.mule.weave.v2.parser.SafeStringBasedParserInput
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.{ ParserPosition, Position }
import org.mule.weave.v2.sdk.WeaveResource

trait DWSerializer[T] {
  def serialize(value: T): SerializableAstNode
  def deserialize(node: SerializableAstNode, context: SerializerContext): T

  def deserializeMaybe(node: SerializableAstNode, context: SerializerContext): Option[T] = {
    node match {
      case NoneSerializableValueAstNode() => None
      case _                              => Some(deserialize(node, context))
    }
  }

  def deserializeSeq(node: SerializableAstNode, context: SerializerContext): Seq[T] = {
    node.asInstanceOf[ArraySerializableAstNode].children.map(deserialize(_, context)).toList
  }

  def deserializeMaybeSeq(node: SerializableAstNode, context: SerializerContext): Option[Seq[T]] = {
    node match {
      case NoneSerializableValueAstNode() => None
      case _                              => Some(deserializeSeq(node, context))
    }
  }

  def deserializeString(node: SerializableAstNode): String = {
    node.asInstanceOf[StringSerializableValueAstNode].value
  }

  def deserializeBoolean(node: SerializableAstNode): Boolean = {
    node.asInstanceOf[BooleanSerializableValueAstNode].value
  }

  def deserializeInt(node: SerializableAstNode): Int = {
    node.asInstanceOf[IntSerializableValueAstNode].value
  }

  def deserializeMaybeString(node: SerializableAstNode): Option[String] = {
    node match {
      case NoneSerializableValueAstNode()        => None
      case StringSerializableValueAstNode(value) => Some(value)
      case _                                     => None //TODO: throw?
    }
  }

  def serializeMaybe(maybeValue: Option[T]): SerializableAstNode = {
    maybeValue match {
      case Some(value) => serialize(value)
      case None        => NoneSerializableValueAstNode()
    }
  }

  def serializeMaybeSeq(maybeValue: Option[Seq[T]]): SerializableAstNode = {
    maybeValue match {
      case Some(value) => serializeSeq(value)
      case None        => NoneSerializableValueAstNode()
    }
  }

  def serializeSeq(seq: Seq[T]): ArraySerializableAstNode = {
    ArraySerializableAstNode(seq.map(serialize))
  }

  def serializeString(str: String): StringSerializableValueAstNode = StringSerializableValueAstNode(str)

  def serializeBoolean(b: Boolean): BooleanSerializableValueAstNode = BooleanSerializableValueAstNode(b)

  def serializeInt(n: Int): IntSerializableValueAstNode = IntSerializableValueAstNode(n)

  def serializerMaybeString(maybeStr: Option[String]): SerializableAstNode = {
    maybeStr match {
      case Some(str) => serializeString(str)
      case None      => NoneSerializableValueAstNode()
    }
  }
}

trait SerializerContext {
  def identifier(): NameIdentifier
  def position(index: Int): Position
}

case class ResourceBasedSerializerContext(resource: WeaveResource, identifier: NameIdentifier) extends SerializerContext {
  private val parserInput = SafeStringBasedParserInput(resource.content())
  override def position(index: Int): Position = ParserPosition(index, parserInput)
}

case class ParserInputBasedSerializerContext(parserInput: SafeStringBasedParserInput, identifier: NameIdentifier) extends SerializerContext {
  override def position(index: Int): Position = ParserPosition(index, parserInput)
}

