package org.mule.weave.v2.interpreted.api.contribution

import java.util.Optional

/**
  * Provides a set of [[JavaBasedDataWeaveFunction]].
  */
trait JavaBasedDataWeaveFunctionValueProvider {
  /**
    * @return The fully qualified name of the module identifier.
    */
  def moduleFQNIdentifier: String

  /**
    * @return an Array of [[JavaBasedDataWeaveFunction]].
    */
  def functions: Array[JavaBasedDataWeaveFunction]
}

/**
  * Representation of a Java based DataWeave function.
  */
trait JavaBasedDataWeaveFunction {
  /**
    * @return The function local function name used to match the function signature in the `.dwl` file
    *         with the corresponding [[JavaBasedDataWeaveFunction]] implementation.
    */
  def functionName: String

  /**
    * @return The function argument.
    */
  def argument: Optional[JavaBasedDataWeaveFunctionArgument]

  /**
    * The function logic to be executed.
    * @param args The array of function arguments.
    * @param servicesProvider The service provider.
    * @return The function outcome.
    */
  def call(args: Array[AnyRef], servicesProvider: ServicesProvider): JavaBasedDataWeaveFunctionResult
}

/**
  * Representation of a Java based DataWeave function without argument.
  */
trait NullaryJavaBasedDataWeaveFunction extends JavaBasedDataWeaveFunction {

  override final def argument: Optional[JavaBasedDataWeaveFunctionArgument] = Optional.empty()

  override final def call(args: Array[AnyRef], servicesProvider: ServicesProvider): JavaBasedDataWeaveFunctionResult = {
    call(servicesProvider)
  }

  /**
    * The function logic to be executed.
    *
    * @param servicesProvider The service provider.
    * @return The function outcome.
    */
  def call(servicesProvider: ServicesProvider): JavaBasedDataWeaveFunctionResult
}

/**
  * Representation of a [[JavaBasedDataWeaveFunction]] argument.
  */
trait JavaBasedDataWeaveFunctionArgument {
  /**
    * @return The mediaType to write the function argument.
    */
  def mediaType: String
}

/**
  * Representation of a [[JavaBasedDataWeaveFunction]] call.
  */
trait JavaBasedDataWeaveFunctionResult {
  /**
    * @return The mimeType to read the function outcome.
    */
  def mediaType: String

  /**
    * @return The function outcome.
    */
  def value: AnyRef
}

/**
  * Provides access to the services available in the execution content.
  */
trait ServicesProvider {
  /**
    * Search for a custom service registered in the function execution context by the service [[Class]].
    *
    * @param service The [[Class]] service instance to look for.
    * @tparam T The service [[Class]] instance.
    * @return The custom service instance or `empty` if there is no custom service for the given service [[Class]].
    */
  def lookupCustomService[T](service: Class[T]): Optional[T]
}
