package org.mule.weave.v2.interpreted.node.structure.function

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.api.contribution.JavaBasedDataWeaveFunction
import org.mule.weave.v2.interpreted.api.contribution.JavaBasedDataWeaveFunctionResult
import org.mule.weave.v2.interpreted.api.contribution.ServicesProvider
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.ConfigurableReaderWriter
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.module.writer.WriterHelper
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.module.MimeType
import org.mule.weave.v2.runtime.core.exception.JavaBasedDataWeaveFunctionExecutionException
import org.mule.weave.v2.runtime.core.exception.UnsupportedJavaBasedDataWeaveFunctionMediaTypeArgumentException
import org.mule.weave.v2.runtime.core.exception.UnsupportedJavaBasedDataWeaveFunctionMediaTypeResultException
import org.mule.weave.v2.utils.Optionals.toJavaOptional

import java.io.PrintWriter
import java.io.StringWriter
import java.nio.charset.Charset
import java.util.Optional
import scala.language.postfixOps

class JavaBasedDataWeaveFunctionBodyValueNode(
  functionFQNIdentifier: NameIdentifier,
  function: JavaBasedDataWeaveFunction,
  parameters: Array[FunctionParameterNode],
  bodyLocation: WeaveLocation)
    extends ValueNode[Any] {

  private lazy val maybeArgumentMimeType: Option[MimeType] = {
    if (function.argument.isPresent) {
      Some(MimeType.fromSimpleString(function.argument.get().mediaType))
    } else {
      None
    }
  }

  private def extractFunctionArgumentValues()(implicit ctx: ExecutionContext): Array[Value[_]] = {
    val argsValues = new Array[Value[_]](parameters.length)
    var i = 0
    while (i < parameters.length) {
      val param = parameters(i)
      val arg = if (param.variable.module.isDefined) {
        val moduleName = param.variable.module.get
        ctx.executionStack().getVariable(moduleName.slot, param.variable.slot)
      } else {
        ctx.executionStack().getVariable(param.variable.slot)
      }
      argsValues.update(i, arg)
      i = i + 1
    }
    argsValues
  }

  private def createFunctionArgumentWriter(mimeType: MimeType, mediaType: String)(implicit ctx: EvaluationContext): Writer = {
    val maybeDataFormat = DataFormatManager.byContentType(mimeType)
    if (maybeDataFormat.isDefined) {
      val writer = maybeDataFormat.get.writer(None, mimeType)
      // Configure writer options
      configure(writer, mimeType)
      writer
    } else {
      throw new UnsupportedJavaBasedDataWeaveFunctionMediaTypeArgumentException(mediaType, location())
    }
  }

  private def createFunctionResultReader(result: JavaBasedDataWeaveFunctionResult)(implicit ctx: EvaluationContext): Reader = {
    val mimeType = MimeType.fromSimpleString(result.mediaType)
    val maybeResultDataFormat = DataFormatManager.byContentType(mimeType)
    if (maybeResultDataFormat.isDefined) {
      val resultDataFormat = maybeResultDataFormat.get
      val charset = mimeType.getCharset()
        .map(Charset.forName)
        .getOrElse(ctx.serviceManager.charsetProviderService.defaultCharset())
      val reader = resultDataFormat.reader(SourceProvider(result.value, charset, Some(mimeType)))
      // Configure reader options
      configure(reader, mimeType)
      reader
    } else {
      throw new UnsupportedJavaBasedDataWeaveFunctionMediaTypeResultException(result.mediaType, location())
    }
  }

  private def configure[T <: ConfigurableReaderWriter](configAware: T, mimeType: MimeType)(implicit ctx: EvaluationContext): T = {
    // Configure the options based on the mimeType properties
    val options = configAware.settings.settingsOptions().definition.iterator
    while (options.hasNext) {
      val (name, moduleOption) = options.next()
      val maybeParam = mimeType.getParameterCaseInsensitive(name)
      if (maybeParam.isDefined) {
        configAware.setOption(location(), moduleOption.name, maybeParam.get)
      }
    }
    configAware
  }

  override protected def doExecute(implicit ctx: ExecutionContext): Value[Any] = {
    try {
      // Create function call args
      val args: Array[AnyRef] = maybeArgumentMimeType match {
        case Some(argumentMimeType) =>
          // Extract function argument from execution stack
          val functionArgs = extractFunctionArgumentValues()
          // Write every value argument before calling the function
          functionArgs.map(arg => {
            val mediaType = if (function.argument.isPresent) {
              function.argument.get().mediaType
            } else {
              "unknown-media-type"
            }
            val writer = createFunctionArgumentWriter(argumentMimeType, mediaType)
            val (result, _) = WriterHelper.writeAndGetResult(writer, arg, UnknownLocationCapable)(ctx)
            result.asInstanceOf[AnyRef]
          })
        case _ =>
          Array.empty
      }

      // Execute function call
      val result = function.call(args, new DefaultServicesProvider(ctx.serviceManager))

      // Read the function result
      val reader = createFunctionResultReader(result)
      val value = reader.read("JavaBasedDataWeaveFunctionResult")
      value
    } catch {
      case e: UnsupportedJavaBasedDataWeaveFunctionMediaTypeArgumentException =>
        throw e
      case e: UnsupportedJavaBasedDataWeaveFunctionMediaTypeResultException =>
        throw e
      case e: Exception =>
        val stringWriter = new StringWriter()
        e.printStackTrace(new PrintWriter(stringWriter))
        throw new JavaBasedDataWeaveFunctionExecutionException(functionFQNIdentifier, location(), s"${stringWriter.toString}")
    }
  }

  // This is for injected functions
  override def location(): WeaveLocation = bodyLocation
}

class DefaultServicesProvider(serviceManager: ServiceManager) extends ServicesProvider {
  override def lookupCustomService[T](service: Class[T]): Optional[T] = {
    serviceManager.lookupCustomService(service) asJava
  }
}