package org.mule.weave.v2.interpreted.transform

import org.mule.weave.v2.core.exception.InvalidNativeNameException
import org.mule.weave.v2.grammar.literals.TypeLiteral
import org.mule.weave.v2.interpreted.api.contribution.NullaryJavaBasedDataWeaveFunction
import org.mule.weave.v2.interpreted.contribution.JavaBasedDataWeaveFunctionManager
import org.mule.weave.v2.interpreted.exception.InvalidUriException
import org.mule.weave.v2.interpreted.marker.InterceptorInvalidReferenceAnnotation
import org.mule.weave.v2.interpreted.marker.InterceptorOnNativeFunctionAnnotation
import org.mule.weave.v2.interpreted.marker.InterceptorOnReferenceAnnotation
import org.mule.weave.v2.interpreted.marker.LiteralNodeAnnotation
import org.mule.weave.v2.interpreted.marker.NativeFunctionAnnotation
import org.mule.weave.v2.interpreted.marker.OverloadedFunctionNotCacheableAnnotation
import org.mule.weave.v2.interpreted.marker.RequiresMaterializationAnnotation
import org.mule.weave.v2.interpreted.marker.TypeParameterNodeAnnotation
import org.mule.weave.v2.interpreted.node.DefaultFunctionCallNode
import org.mule.weave.v2.interpreted.node.ExistsSelectorNode
import org.mule.weave.v2.interpreted.node.NullUnSafeNode
import org.mule.weave.v2.interpreted.node.TypeReferenceNode
import org.mule.weave.v2.interpreted.node.TypeReferenceWithTypeParamNode
import org.mule.weave.v2.interpreted.node.TypeSelectorReferenceNode
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.{ NullSafeNode => RNullSafeNode }
import org.mule.weave.v2.interpreted.node.{ VariableReferenceNode => RVariableReferenceNode }
import org.mule.weave.v2.interpreted.node.executors.DefaultFunctionCallExecutor
import org.mule.weave.v2.interpreted.node.structure.ArrayNode
import org.mule.weave.v2.interpreted.node.structure.ArrayTypeNode
import org.mule.weave.v2.interpreted.node.structure.AttributesNode
import org.mule.weave.v2.interpreted.node.structure.BooleanNode
import org.mule.weave.v2.interpreted.node.structure.ConditionalNode
import org.mule.weave.v2.interpreted.node.structure.DateTimeNode
import org.mule.weave.v2.interpreted.node.structure.DynamicArrayNode
import org.mule.weave.v2.interpreted.node.structure.DynamicKeyNode
import org.mule.weave.v2.interpreted.node.structure.DynamicNameNode
import org.mule.weave.v2.interpreted.node.structure.FilteredOverloadedFunctionNode
import org.mule.weave.v2.interpreted.node.structure.HeadTailArrayNode
import org.mule.weave.v2.interpreted.node.structure.HeadTailObjectNode
import org.mule.weave.v2.interpreted.node.structure.KeyNode
import org.mule.weave.v2.interpreted.node.structure.KeyValuePairNode
import org.mule.weave.v2.interpreted.node.structure.LiteralArrayNode
import org.mule.weave.v2.interpreted.node.structure.LiteralObjectNode
import org.mule.weave.v2.interpreted.node.structure.LocalDateNode
import org.mule.weave.v2.interpreted.node.structure.LocalDateTimeNode
import org.mule.weave.v2.interpreted.node.structure.LocalTimeNode
import org.mule.weave.v2.interpreted.node.structure.NameValuePairNode
import org.mule.weave.v2.interpreted.node.structure.NamespaceNode
import org.mule.weave.v2.interpreted.node.structure.NoFilter
import org.mule.weave.v2.interpreted.node.structure.NullNode
import org.mule.weave.v2.interpreted.node.structure.NumberNode
import org.mule.weave.v2.interpreted.node.structure.ObjectNode
import org.mule.weave.v2.interpreted.node.structure.OverloadedFunctionNode
import org.mule.weave.v2.interpreted.node.structure.PeriodNode
import org.mule.weave.v2.interpreted.node.structure.RFunctionParamTypeNode
import org.mule.weave.v2.interpreted.node.structure.RFunctionTypeNode
import org.mule.weave.v2.interpreted.node.structure.RKeyTypeNode
import org.mule.weave.v2.interpreted.node.structure.RKeyValuePairTypeNode
import org.mule.weave.v2.interpreted.node.structure.RLiteralTypeNode
import org.mule.weave.v2.interpreted.node.structure.RNameTypeNode
import org.mule.weave.v2.interpreted.node.structure.RNameValuePairTypeNode
import org.mule.weave.v2.interpreted.node.structure.RObjectTypeNode
import org.mule.weave.v2.interpreted.node.structure.RegexNode
import org.mule.weave.v2.interpreted.node.structure.TimeNode
import org.mule.weave.v2.interpreted.node.structure.TimeZoneNode
import org.mule.weave.v2.interpreted.node.structure.TypeNode
import org.mule.weave.v2.interpreted.node.structure.UriNode
import org.mule.weave.v2.interpreted.node.structure.VersionFilter
import org.mule.weave.v2.interpreted.node.structure.function.DynamicFunctionNode
import org.mule.weave.v2.interpreted.node.structure.function.FunctionParameterNode
import org.mule.weave.v2.interpreted.node.structure.function.InterceptedCheckFunctionDecoratorNode
import org.mule.weave.v2.interpreted.node.structure.function.JavaBasedDataWeaveFunctionBodyValueNode
import org.mule.weave.v2.interpreted.node.structure.function.LiteralFunctionValueNode
import org.mule.weave.v2.interpreted.node.structure.function.StaticFunctionNode
import org.mule.weave.v2.interpreted.node.structure.header.directives
import org.mule.weave.v2.interpreted.node.structure.schema.{ SchemaNode => RSchemaNode }
import org.mule.weave.v2.interpreted.node.structure.{ IntersectionTypeNode => RIntersectionTypeNode }
import org.mule.weave.v2.interpreted.node.structure.{ NameNode => RNameNode }
import org.mule.weave.v2.interpreted.node.structure.{ StringInterpolationNode => RStringInterpolationNode }
import org.mule.weave.v2.interpreted.node.structure.{ StringNode => RStringNode }
import org.mule.weave.v2.interpreted.node.structure.{ TypeParameterNode => RTypeParameterNode }
import org.mule.weave.v2.interpreted.node.structure.{ UnionTypeNode => RUnionTypeNode }
import org.mule.weave.v2.model.capabilities.UpdateLocationCapable
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.NativeValueManager
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.parser
import org.mule.weave.v2.parser.annotation.SinceAstNodeAnnotation
import org.mule.weave.v2.parser.ast
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.functions
import org.mule.weave.v2.parser.ast.types
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.functions.FunctionParameter
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.StringInterpolationNode
import org.mule.weave.v2.parser.ast.types.ObjectTypeNode
import org.mule.weave.v2.parser.ast.types.TypeParameterNode
import org.mule.weave.v2.parser.ast.types._
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.exception.InvalidDateLiteralException
import org.mule.weave.v2.parser.exception.ParseException
import org.mule.weave.v2.parser.exception.WeaveRuntimeException
import org.mule.weave.v2.parser.location.{ Location, UnknownLocation }
import org.mule.weave.v2.runtime.core.exception.InvalidJavaBasedDataWeaveFunctionArgumentException
import org.mule.weave.v2.runtime.core.exception.InvalidNullaryJavaBasedDataWeaveFunctionArgumentException
import org.mule.weave.v2.runtime.exception.CompilationExecutionException
import org.mule.weave.v2.utils.StringEscapeHelper

import java.time._
import java.time.format.DateTimeParseException
import scala.annotation.tailrec
import scala.util.matching.Regex

trait EngineStructureTransformations extends AstTransformation with EngineVariableTransformations {
  def transformDateTimeNode(dateTime: String)(implicit location: Location): DateTimeNode = {
    try {
      new DateTimeNode(ZonedDateTime.parse(dateTime))
    } catch {
      case dtpe: DateTimeParseException => throw new InvalidDateLiteralException(location, dateTime)
    }
  }

  def transformStringNode(v: parser.ast.structure.StringNode): RStringNode = {
    val quotedChar: Option[Char] = v.quotedBy()
    if (quotedChar.isDefined) {
      val unescaped = StringEscapeHelper.unescapeString(v.value, quotedChar.get, v.location())
      RStringNode(unescaped)
    } else {
      RStringNode(v.value)
    }
  }

  def transformTimeNode(time: String)(implicit location: Location): TimeNode = {
    try {
      new TimeNode(OffsetTime.parse(time))
    } catch {
      case dtpe: DateTimeParseException => throw new InvalidDateLiteralException(location, time)
    }
  }

  def transformStringInterpolationNode(v: Seq[AstNode]) = RStringInterpolationNode(transformSeq(v))

  def transformLocalDateTimeNode(dateTime: String)(implicit location: Location): LocalDateTimeNode = {
    try {
      new LocalDateTimeNode(LocalDateTime.parse(dateTime))
    } catch {
      case dtpe: DateTimeParseException => throw new InvalidDateLiteralException(location, dateTime)
    }
  }

  def transformUriNode(uri: String, uriNode: AstNode): UriNode = {
    try {
      UriNode(uri)
    } catch {
      case e: IllegalArgumentException => throw new InvalidUriException(uri, e.getMessage, uriNode.location())
    }
  }

  def transformConditionalNode(value: AstNode, cond: AstNode) = new ConditionalNode(transform(value), transform(cond))

  def transformPeriodNode(period: String)(implicit location: Location): PeriodNode = {
    try {
      if (period.contains('T')) {
        new PeriodNode(Duration.parse(period))
      } else {
        new PeriodNode(Period.parse(period))
      }
    } catch {
      case e: DateTimeParseException => throw new InvalidDateLiteralException(location, period)
    }
  }

  def transformRegexNode(regex: String): RegexNode = new RegexNode(new Regex(regex))

  val nativeTypeMap: Map[String, Type] = Map(
    TypeLiteral.STRING_TYPE_NAME -> StringType,
    TypeLiteral.BOOLEAN_TYPE_NAME -> BooleanType,
    TypeLiteral.NUMBER_TYPE_NAME -> NumberType,
    TypeLiteral.RANGE_TYPE_NAME -> RangeType,
    TypeLiteral.URI_TYPE_NAME -> UriType,
    TypeLiteral.DATETIME_TYPE_NAME -> DateTimeType,
    TypeLiteral.LOCALDATETIME_TYPE_NAME -> LocalDateTimeType,
    TypeLiteral.DATE_TYPE_NAME -> LocalDateType,
    TypeLiteral.LOCALTIME_TYPE_NAME -> LocalTimeType,
    TypeLiteral.TIME_TYPE_NAME -> TimeType,
    TypeLiteral.TIMEZONE_TYPE_NAME -> TimeZoneType,
    TypeLiteral.PERIOD_TYPE_NAME -> PeriodType,
    TypeLiteral.BINARY_TYPE_NAME -> BinaryType,
    TypeLiteral.NULL_TYPE_NAME -> NullType,
    TypeLiteral.ANY_TYPE_NAME -> AnyType,
    TypeLiteral.REGEX_TYPE_NAME -> RegexType,
    TypeLiteral.NOTHING_TYPE_NAME -> NothingType,
    TypeLiteral.NAME_SPACE_TYPE_NAME -> NamespaceType,
    TypeLiteral.ARRAY_TYPE_NAME -> ArrayType,
    TypeLiteral.TYPE_TYPE_NAME -> TypeType,
    TypeLiteral.OBJECT_TYPE_NAME -> ObjectType,
    TypeLiteral.KEY_TYPE_NAME -> KeyType,
    TypeLiteral.KEY_VALUE_PAIR_TYPE_NAME -> KeyValuePairType,
    TypeLiteral.ATTRIBUTES_TYPE_NAME -> AttributesType,
    TypeLiteral.NAME_TYPE_NAME -> NameType,
    TypeLiteral.NAME_VALUE_PAIR_TYPE_NAME -> NameValuePairType,
    TypeLiteral.NAME_SPACE_TYPE_NAME -> NamespaceType,
    TypeLiteral.SCHEMA_TYPE_NAME -> SchemaType,
    TypeLiteral.SCHEMA_PROPERTY_TYPE_NAME -> SchemaPropertyType,
    TypeLiteral.FUNCTION_TYPE_NAME -> FunctionType)

  def transformTypeDirective(variable: NameIdentifier, typeParametersListNode: Option[TypeParametersListNode], t: WeaveTypeNode, codeAnnotations: Seq[AnnotationNode])(implicit location: Location): directives.Directive = {
    if (typeParametersListNode.isEmpty || typeParametersListNode.get.typeParameters.isEmpty || nativeTypeMap.contains(variable.name)) {
      new directives.TypeDirective(transform(variable), transformTypeNode(t, ignoreTypeParameter = false, codeAnnotations))
    } else {
      val typeNode = TypeReferenceNode(NameIdentifier(TypeLiteral.TYPE_TYPE_NAME))
      new directives.TypeFunctionDirective(
        transform(variable),
        transformTypeNode(t, ignoreTypeParameter = false, codeAnnotations),
        typeParametersListNode.get.typeParameters
          .map(typeParam => {
            val nameSlot = transformNameSlot(typeParam.name)
            new FunctionParameterNode(nameSlot, transform(typeNode), None, false, false)
          })
          .toArray)
    }
  }

  def transformTypeNode(weaveType: WeaveTypeNode, ignoreTypeParameter: Boolean = true, typeDirectiveAnnotation: Seq[AnnotationNode] = Seq())(implicit location: Location): ValueNode[Type] = {
    val result = weaveType match {
      case ObjectTypeNode(properties, asSchema, asTypeSchema, close, _, _) => {
        if (ignoreTypeParameter) {
          //We should filter properties with type parameters in the key
          val filteredProperties = properties.collect({
            case kvt: KeyValueTypeNode if !isTypeReferenceNode(kvt.key) => kvt
          })
          new RObjectTypeNode(transformSeq(filteredProperties), !close, transformOption(asSchema), transformOption(asTypeSchema))
        } else {
          new RObjectTypeNode(transformSeq(properties), !close, transformOption(asSchema), transformOption(asTypeSchema))
        }
      }
      case KeyValueTypeNode(k, v, repeated, opt)                 => new RKeyValuePairTypeNode(transform(k), transform(v), opt, repeated)
      case NameValueTypeNode(k, v, opt)                          => new RNameValuePairTypeNode(transform(k), transform(v), opt)
      case KeyTypeNode(name, attrs, _, _, _)                     => new RKeyTypeNode(transform(name), transformSeq(attrs))
      case NameTypeNode(Some(name), nsMaybe, maybeTypeSchema, _) => new RNameTypeNode(name, transformOption(nsMaybe), transformOption(maybeTypeSchema))
      case NameTypeNode(None, None, maybeTypeSchema, _)          => new TypeNode(NameType, None, transformOption(maybeTypeSchema))
      case FunctionTypeNode(params, returnWeaveType, asSchema, asTypeSchema, _) => {
        val paramTypeNodes: Seq[RFunctionParamTypeNode] = params.map((param) => {
          val valueType: ValueNode[Type] = transform(param.valueType)
          new RFunctionParamTypeNode(valueType, param.name.map(_.name), param.optional)
        })
        val returnTypeNode: ValueNode[Type] = transform(returnWeaveType)
        new RFunctionTypeNode(paramTypeNodes, returnTypeNode, transformOption(asSchema), transformOption(asTypeSchema))
      }
      case UnionTypeNode(elems, asSchema, asTypeSchema, _) => {
        val seq = elems.map(transformTypeNode(_))
        new RUnionTypeNode(seq, transformOption(asSchema), transformOption(asTypeSchema))
      }
      case IntersectionTypeNode(elems, asSchema, asTypeSchema, _) => {
        val nodes = elems.map(transformTypeNode(_))
        new RIntersectionTypeNode(nodes, transformOption(asSchema), transformOption(asTypeSchema))
      }
      case NativeTypeNode(typeId, asSchema) => new TypeNode(nativeTypeMap(typeId), transformOption(asSchema))
      case LiteralTypeNode(valueNode, asSchema, asTypeSchema, _) => {
        valueNode match {
          case sn @ ast.structure.StringNode(strValue, _) => new TypeNode(new StringType(Some(StringEscapeHelper.unescapeString(strValue, sn.quotedBy().getOrElse('"')))), transformOption(asSchema), transformOption(asTypeSchema))
          case ast.structure.NumberNode(numValue, _)      => new TypeNode(new NumberType(Some(Number(valueNode.location(), numValue))), transformOption(asSchema), transformOption(asTypeSchema))
          case ast.structure.BooleanNode(booValue, _)     => new TypeNode(new BooleanType(Some(booValue.toBoolean)), transformOption(asSchema), transformOption(asTypeSchema)) //TODO: Check toBoolean
        }
      }
      case TypeParameterNode(_, base) => {
        base match {
          case None        => new TypeNode(AnyType, None)
          case Some(value) => new RTypeParameterNode(transformTypeNode(value))
        }
      }
      case trn: types.TypeReferenceNode => {
        transformTypeReferenceNode(trn, ignoreTypeParameter)
      }
      case trns: TypeSelectorNode => {
        transformTypeSelectorReferenceNode(trns, ignoreTypeParameter)
      }
    }
    result._location = Some(weaveType.location())
    result
  }

  def isTypeReferenceNode(weaveTypeNode: WeaveTypeNode): Boolean = {
    weaveTypeNode.isInstanceOf[types.TypeReferenceNode] && asTypeParameterRef(weaveTypeNode.asInstanceOf[types.TypeReferenceNode]).nonEmpty
  }

  def transformTypeReferenceNode(trn: ast.types.TypeReferenceNode, ignoreTypeParameter: Boolean = true)(implicit location: Location): ValueNode[Type] = {
    if (nativeTypeMap.contains(trn.variable.name)) {
      if (trn.variable.name.equals("Array") && trn.typeArguments.isDefined && trn.typeArguments.get.size == 1) {
        val option: Option[RSchemaNode] = transformOption(trn.asSchema)
        new ArrayTypeNode(transform(trn.typeArguments.get.head), option, transformOption(trn.asTypeSchema))
      } else {
        new TypeNode(nativeTypeMap(trn.variable.name), transformOption(trn.asSchema), transformOption(trn.asTypeSchema))
      }
    } else {
      val typeParameterReference: Option[TypeParameterNode] = asTypeParameterRef(trn)
      if (ignoreTypeParameter && typeParameterReference.isDefined) {
        transformTypeNode(typeParameterReference.get)
      } else if (trn.typeArguments.isEmpty || trn.typeArguments.get.isEmpty) {
        val maybeSlot = transformReference(trn.variable)
        if (maybeSlot.isDefined) {
          new TypeReferenceNode(maybeSlot.get, transformOption(trn.asSchema), transformOption(trn.asTypeSchema))
        } else {
          throw new ParseException(s"Unable to resolve type reference node ${trn.variable.name}", trn.variable.location())
        }
      } else {
        val maybeSlot = transformReference(trn.variable)
        if (maybeSlot.isDefined) {
          val functionExecutor: DefaultFunctionCallExecutor = new DefaultFunctionCallExecutor(RVariableReferenceNode(maybeSlot.get), false, trn.location())
          val node = new TypeReferenceWithTypeParamNode(
            trn.variable.name,
            new DefaultFunctionCallNode(functionExecutor, trn.typeArguments.get.map(transformTypeNode(_)).toArray),
            transformOption(trn.asSchema),
            transformOption(trn.asTypeSchema))
          node.asInstanceOf[ValueNode[Type]]
        } else {
          throw new ParseException(s"Unable to resolve type reference node ${trn.variable.name}", trn.variable.location())
        }
      }
    }
  }

  private def transformTypeSelectorReferenceNode(trns: TypeSelectorNode, ignoreTypeParameter: Boolean)(implicit location: Location): ValueNode[Type] = {
    trns.selector match {
      case nameNode @ NameNode(StringInterpolationNode(_), _, _) => throw new WeaveRuntimeException(s"Type selection using string interpolation is not allowed", nameNode.location());
      case _ => new TypeSelectorReferenceNode(transformTypeNode(trns.weaveTypeNode, ignoreTypeParameter), transform(trns.selector), transformOption(trns.asSchema), transformOption(trns.asTypeSchema))
    }
  }

  private def asTypeParameterRef(trn: types.TypeReferenceNode): Option[TypeParameterNode] = {
    val maybeTypeParameterNode = trn.variable.annotation(classOf[TypeParameterNodeAnnotation])
    maybeTypeParameterNode match {
      case Some(_) =>
        findTypeParameter(trn, trn)
      case _ =>
        None
    }
  }

  @tailrec
  private def findTypeParameter(trn: types.TypeReferenceNode, node: AstNode): Option[TypeParameterNode] = {
    val maybeFunctionNode = astNavigator().parentWithType(node, classOf[FunctionNode])
    maybeFunctionNode match {
      case Some(fn) =>
        fn.typeParameterList match {
          case Some(typeParameterList) =>
            val maybeTypeParameter = typeParameterList.typeParameters.find(tp => {
              tp.name == trn.variable
            })
            if (maybeTypeParameter.isDefined) {
              maybeTypeParameter
            } else {
              findTypeParameter(trn, fn)
            }
          case None =>
            findTypeParameter(trn, fn)
        }
      case None =>
        throw new WeaveRuntimeException(s"Unable to resolve reference of Type Parameter ${trn.variable}", trn.location())
    }
  }

  def transformTimeZoneNode(timeZone: String)(implicit location: Location): TimeZoneNode = {
    new TimeZoneNode(timeZone)
  }

  def transformLocalDateNode(value: String)(implicit location: Location): LocalDateNode = {
    try {
      val localDate: LocalDate = DateHelper.parseLocalDate(value)
      new LocalDateNode(localDate)
    } catch {
      case dtpe: DateTimeParseException => throw new InvalidDateLiteralException(location, value)
    }

  }

  def transformNullNode(): NullNode = new NullNode()

  def transformArrayNode(v: ast.structure.ArrayNode): ValueNode[ArraySeq] = {
    val isArrayLiteral = v.isAnnotatedWith(classOf[LiteralNodeAnnotation])
    if (isArrayLiteral) {
      new LiteralArrayNode(transformSeq(v.elements).toArray)
    } else if (AstNodeHelper.containsConditionalElements(v)) {
      new DynamicArrayNode(transformSeq(v.elements))
    } else {
      new ArrayNode(transformSeq(v.elements).toArray)
    }
  }

  def transformHeadTailArrayNode(head: AstNode, tail: AstNode): HeadTailArrayNode = new HeadTailArrayNode(transform(head), transform(tail))

  def transformHeadTailObjectNode(headKey: AstNode, headValue: AstNode, tail: AstNode): HeadTailObjectNode = new HeadTailObjectNode(transform(headKey), transform(headValue), transform(tail))

  def transformAttributesNode(attrs: Seq[AstNode]): AttributesNode = new AttributesNode(transformSeq(attrs))

  def transformObjectNode(objectNode: ast.structure.ObjectNode): ValueNode[ObjectSeq] = {
    val isObjectLiteral = objectNode.isAnnotatedWith(classOf[LiteralNodeAnnotation])
    if (isObjectLiteral) {
      val nodes: Array[KeyValuePairNode] = transformSeq(objectNode.elements).toArray
      new LiteralObjectNode(nodes)
    } else {
      ObjectNode(transformSeq(objectNode.elements))
    }
  }

  def transformFunctionNode(fn: functions.FunctionNode, functionName: Option[NameIdentifier] = None): ValueNode[_] = {
    val isNativeCall = fn.body.isAnnotatedWith(classOf[NativeFunctionAnnotation])
    if (isNativeCall) {
      val value = AstNodeHelper.getNativeIdentifierCall(fn.body).getOrElse(throw new InvalidNativeNameException("Native name must be a literal string.", fn.body.location()))
      val functionValue: FunctionValue = NativeValueManager.getFunctionValue(value)
      functionValue.defaultName = functionName.map(_.name)
      functionValue match {
        case ulc: UpdateLocationCapable => {
          ulc.updateLocation(fn.body.location())
        }
        case _ =>
      }
      new LiteralFunctionValueNode(functionValue)
    } else {
      val bodyValue: ValueNode[_] = transform(fn.body)
      createFunctionNode(fn, functionName, bodyValue)
    }
  }

  def createFunctionNode(fn: functions.FunctionNode, functionName: Option[NameIdentifier], bodyValue: ValueNode[_]): ValueNode[_] = {
    val functionParameters: Array[FunctionParameterNode] = transformSeq(fn.params.paramList).toArray

    // Search a `JavaBasedDataWeaveFunction` definition to replace the function body value
    val bodyFunctionValue: ValueNode[_] = if (functionName.isDefined) {
      val maybeJavaBasedDataWeaveFunctionWithModule = JavaBasedDataWeaveFunctionManager.getFunction(parsingContext().nameIdentifier, functionName.get.name)
      if (maybeJavaBasedDataWeaveFunctionWithModule.isDefined) {
        val javaBasedDataWeaveFunctionWithModule = maybeJavaBasedDataWeaveFunctionWithModule.get
        javaBasedDataWeaveFunctionWithModule.fn match {
          case npf: NullaryJavaBasedDataWeaveFunction =>
            if (functionParameters.nonEmpty || npf.argument.isPresent) {
              throw new InvalidNullaryJavaBasedDataWeaveFunctionArgumentException(functionName.get.location())
            }
          case _ =>
            if ((functionParameters.nonEmpty && javaBasedDataWeaveFunctionWithModule.fn.argument.isEmpty) || (functionParameters.isEmpty && javaBasedDataWeaveFunctionWithModule.fn.argument.isPresent)) {
              throw new InvalidJavaBasedDataWeaveFunctionArgumentException(functionName.get.location())
            }
        }
        new JavaBasedDataWeaveFunctionBodyValueNode(javaBasedDataWeaveFunctionWithModule.functionFQNIdentifier, javaBasedDataWeaveFunctionWithModule.fn, functionParameters, bodyValue.location())
      } else {
        bodyValue
      }
    } else {
      bodyValue
    }

    val requiresMaterialize = fn.params.paramList.exists(arg => {
      arg.isAnnotatedWith(classOf[RequiresMaterializationAnnotation])
    })
    val returnTypeNode: Option[ValueNode[Type]] = if (fn.returnType.isDefined && !fn.returnType.get.isInstanceOf[DynamicReturnTypeNode]) Some(transform(fn.returnType.get)) else None
    if (transformingModule && astNavigator().granGranParentOf(fn.body).exists(_.isInstanceOf[ModuleNode])) {
      //Module Functions are always static
      new StaticFunctionNode(functionParameters, bodyFunctionValue, returnTypeNode, requiresMaterializedArguments = requiresMaterialize, functionName.map(_.name))
    } else {
      //We should be able to determine if a function is static or dynamic if it is self contained is static otherwise is dynamic
      //To determine if it self contain we should check that the body and params only reference things from inside the scope of the function
      val dynamicParameter = fn.params.paramList.exists((fp) => {
        //Is Dynamic Parameter
        if (fp.defaultValue.isDefined) {
          !AstNodeHelper.isLiteral(fp.defaultValue.get)
        } else {
          false
        }
      })

      new DynamicFunctionNode(functionParameters, bodyFunctionValue, returnTypeNode, requiresMaterializedArguments = requiresMaterialize, name = functionName.map(_.name), literalParameters = !dynamicParameter)
    }
  }

  def applyInterceptor(functionValue: ValueNode[_], functionName: NameIdentifier, codeAnnotations: Seq[AnnotationNode]): ValueNode[_] = {
    val interceptors: Seq[(AnnotationNode, ValueNode[Any])] =
      codeAnnotations.flatMap(annotation => {
        toInterceptor(annotation)
      })
    if (interceptors.nonEmpty) {
      val interceptorFunction: ValueNode[_] = interceptors.foldLeft[ValueNode[_]](functionValue)((functionValue, annotation) => {
        val annotationInterceptorFunction: ValueNode[Any] = annotation._2
        val args: Seq[KeyValuePairNode] = annotation._1.args
          .map(args =>
            args.args.map(arg => {
              val argValue: ValueNode[_] = transform(arg.value)
              new KeyValuePairNode(new KeyNode(RStringNode(arg.name.name), None, None), argValue, None, false)
            }))
          .getOrElse(Seq.empty)
        val decoratorNode = new InterceptedCheckFunctionDecoratorNode(functionValue, annotationInterceptorFunction, new ObjectNode(args.toArray), StringValue(functionName.name))
        decoratorNode._location = annotation._1._location
        decoratorNode
      })
      interceptorFunction
    } else {
      functionValue
    }
  }

  private def toInterceptor(annotation: AnnotationNode): Option[(AnnotationNode, ValueNode[Any])] = {
    val maybeInterceptorInvalidReference = annotation.annotation(classOf[InterceptorInvalidReferenceAnnotation])
    if (maybeInterceptorInvalidReference.isDefined) {
      val interceptorUsingInvalidReference = maybeInterceptorInvalidReference.get
      throw new CompilationExecutionException(interceptorUsingInvalidReference.invalidReference.location, s"Unable to resolve reference to `${interceptorUsingInvalidReference.invalidReference.reference}`")
    }

    val maybeInterceptorOnNativeFunction = annotation.annotation(classOf[InterceptorOnNativeFunctionAnnotation])
    maybeInterceptorOnNativeFunction match {
      case Some(interceptorOnNativeFunction) =>
        val nativeFunctionInterceptor: FunctionValue = NativeValueManager.getFunctionValue(interceptorOnNativeFunction.nativeFunctionValue.functionName)
        nativeFunctionInterceptor.defaultName = Some(interceptorOnNativeFunction.nativeFunctionValue.functionName.name)
        nativeFunctionInterceptor match {
          case ulc: UpdateLocationCapable =>
            ulc.updateLocation(interceptorOnNativeFunction.nativeFunctionValue.location.getOrElse(UnknownLocation))
          case _ =>
        }
        val functionValueNode: LiteralFunctionValueNode = new LiteralFunctionValueNode(nativeFunctionInterceptor)
        functionValueNode._location = interceptorOnNativeFunction.nativeFunctionValue.location
        Some((annotation, functionValueNode))
      case _ =>
        val maybeInterceptorOnReference = annotation.annotation(classOf[InterceptorOnReferenceAnnotation])
        maybeInterceptorOnReference match {
          case Some(interceptorOnReference) =>
            val maybeNameSlot = transformReferenceValue(interceptorOnReference.referenceValue)
            maybeNameSlot match {
              case Some(nameSlot) =>
                val referenceNode = RVariableReferenceNode(nameSlot)
                referenceNode._location = interceptorOnReference.referenceValue.referenceLocation
                Some((annotation, referenceNode))
              case None =>
                None
            }
          case _ =>
            None
        }
    }
  }

  def transformOverloadedFunctionNode(ofn: functions.OverloadedFunctionNode, name: String): ValueNode[_] = {
    val functionNodes =
      ofn.functionDirectives.map(fdn => {
        applyInterceptor(transform[ValueNode[_]](fdn.literal), fdn.variable, fdn.codeAnnotations)
      })

    val notCacheable = ofn.isAnnotatedWith(classOf[OverloadedFunctionNotCacheableAnnotation])

    val hasASince = ofn.functionDirectives.exists(fd => {
      fd.isAnnotatedWith(classOf[SinceAstNodeAnnotation])
    })
    if (hasASince) {
      val functionFilters = ofn.functionDirectives.map(fd => {
        val maybeSinceAnnotation = fd.annotation(classOf[SinceAstNodeAnnotation])
        if (maybeSinceAnnotation.isDefined) {
          new VersionFilter(maybeSinceAnnotation.get.version)
        } else {
          NoFilter
        }
      })
      new FilteredOverloadedFunctionNode(functionNodes.toArray, functionFilters.toArray, Some(name), !notCacheable)
    } else {
      new OverloadedFunctionNode(functionNodes.toArray, Some(name), !notCacheable)
    }
  }

  def transformNameValuePairNode(key: AstNode, value: AstNode, cond: Option[AstNode]): NameValuePairNode =
    new NameValuePairNode(transform(key), transform(value), transformOption(cond))

  def transformNamespaceNode(nsn: parser.ast.structure.NamespaceNode): NamespaceNode = {
    val maybeSlot = transformReference(nsn.prefix)
    if (maybeSlot.isDefined) {
      NamespaceNode(maybeSlot.get)
    } else {
      throw new ParseException(s"Unable to resolve namespace ${nsn.prefix.name}", nsn.prefix.location())
    }
  }

  def transformFunctionParameter(fp: FunctionParameter): FunctionParameterNode = {
    val materialize: Boolean = needsMaterialization(fp.variable)
    val isDesignOnly = fp.codeAnnotations.exists((annotation) => annotation.name.name == "DesignOnlyType")
    val parameterType: ValueNode[Type] = if (isDesignOnly) {
      //As is design only we just put ANY
      transform(TypeReferenceNode(NameIdentifier(TypeLiteral.ANY_TYPE_NAME)))
    } else {
      val typeNode = fp.wtype.getOrElse(TypeReferenceNode(NameIdentifier(TypeLiteral.ANY_TYPE_NAME)))
      transform(typeNode)
    }
    val paramTypeRequiresMaterialize = if (isDesignOnly) {
      false
    } else {
      fp.isAnnotatedWith(classOf[RequiresMaterializationAnnotation])
    }
    val node = new RLiteralTypeNode(parameterType)
    new FunctionParameterNode(transform(fp.variable), node, transformOption(fp.defaultValue), materialize, paramTypeRequiresMaterialize)
  }

  def transformLocalTimeNode(time: String)(implicit location: Location): LocalTimeNode = {
    try {
      new LocalTimeNode(LocalTime.parse(time))
    } catch {
      case dtpe: DateTimeParseException => throw new InvalidDateLiteralException(location, time)
    }
  }

  def transformKeyValuePairNode(key: AstNode, value: AstNode, cond: Option[AstNode]): KeyValuePairNode = {
    val isLiteralKey = key.isAnnotatedWith(classOf[LiteralNodeAnnotation])
    val isLiteralKeyValueNode = AstNodeHelper.isLiteral(value) && isLiteralKey
    new KeyValuePairNode(transform(key), transform(value), transformOption(cond), isLiteralKeyValueNode)
  }

  def transformNameNode(keyName: AstNode, cond: Option[AstNode]): ValueNode[QualifiedName] = RNameNode(transform(keyName), transformOption(cond))

  def transformDynamicNameNode(keyName: AstNode): DynamicNameNode = new DynamicNameNode(transform(keyName))

  def transformNumberNode(location: Location, v: String): NumberNode = {
    new NumberNode(Number(location, v))
  }

  def transformBooleanNode(value: String): BooleanNode =
    new BooleanNode(value.toBoolean)

  def transformKeyNode(key: parser.ast.structure.KeyNode): ValueNode[QualifiedName] = {
    val literalKey = key.isAnnotatedWith(classOf[LiteralNodeAnnotation])
    if (literalKey) {
      KeyNode.literalKeyNode(transform(key.keyName), transformOption(key.ns), transformOption(key.attr))
    } else {
      val isLiteralNamespace = key.ns match {
        case Some(value) =>
          value.isAnnotatedWith(classOf[LiteralNodeAnnotation])
        case None =>
          true
      }
      if (isLiteralNamespace && AstNodeHelper.isLiteral(key.keyName)) {
        KeyNode.literalNameKeyNode(transform(key.keyName), transformOption(key.ns), transformOption(key.attr))
      } else {
        KeyNode(transform(key.keyName), transformOption(key.ns), transformOption(key.attr))
      }
    }
  }

  def transformDynamicKeyNode(keyName: AstNode, attr: Option[AstNode]): DynamicKeyNode = {
    KeyNode(transform(keyName), transformOption(attr))
  }

  def transformNullSafeNode(selector: AstNode): ValueNode[_] = {
    selector match {
      case variableReferenceNode: ast.variables.VariableReferenceNode => {
        val value: ValueNode[_] = transform(variableReferenceNode)
        value
      }
      case _ => RNullSafeNode(transform(selector))
    }
  }

  def transformNullUnSafeNode(selector: AstNode): NullUnSafeNode =
    NullUnSafeNode(transform(selector))

  def transformExistsSelectorNode(selectable: AstNode): ExistsSelectorNode =
    new ExistsSelectorNode(transform(selectable))
}
