package org.mule.weave.v2.runtime

import org.mule.weave.v2.annotations.WeaveApi
import org.mule.weave.v2.interpreted.CompositeRuntimeModuleNodeCompiler
import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.parser.phase.CompositeModuleParsingPhasesManager
import org.mule.weave.v2.parser.phase.ModuleLoader
import org.mule.weave.v2.parser.phase.ModuleLoaderManager
import org.mule.weave.v2.parser.phase.ModuleParsingPhasesManager
import org.mule.weave.v2.sdk.ClassLoaderWeaveResourceResolver
import org.mule.weave.v2.sdk.CompositeWeaveResourceResolver
import org.mule.weave.v2.sdk.SPIBasedModuleLoaderProvider
import org.mule.weave.v2.sdk.TwoLevelWeaveResourceResolver
import org.mule.weave.v2.sdk.WLangOnlyWeaveResourceResolver
import org.mule.weave.v2.sdk.WeaveResourceResolver

/**
  * Allows to configure how the engine loads the scripts. This is a low level tool so use it with care.
  */
@WeaveApi()
trait ModuleComponentsFactory {
  def createComponents(): ModuleComponents
}

case class ModuleComponents(resourceResolver: WeaveResourceResolver, parser: ModuleParsingPhasesManager, compiler: RuntimeModuleNodeCompiler)

class DynamicModuleComponentFactory(systemLevel: WeaveResourceResolver, dynamicLevel: () => WeaveResourceResolver, systemFirst: Boolean) extends ModuleComponentsFactory {

  /**
    * Handles the parsing of the modules that are on the SystemClassLoader
    */
  private val systemModuleParser: ModuleParsingPhasesManager = {
    val moduleLoaderManager = ModuleLoaderManager(Seq(ModuleLoader(systemLevel)), SPIBasedModuleLoaderProvider)
    ModuleParsingPhasesManager(moduleLoaderManager)
  }

  /**
    * Handles the compilation of modules that are on the SystemClassLoader
    */
  private val systemModuleCompiler: RuntimeModuleNodeCompiler = RuntimeModuleNodeCompiler(systemModuleParser, systemLevel)

  override def createComponents(): ModuleComponents = {
    val localLevel: WeaveResourceResolver = dynamicLevel()
    val currentClassloader: ModuleParsingPhasesManager = ModuleParsingPhasesManager(ModuleLoaderManager(Seq(ModuleLoader(localLevel)), new SPIBasedModuleLoaderProvider(localLevel)))
    val parser: CompositeModuleParsingPhasesManager = if (systemFirst) CompositeModuleParsingPhasesManager(systemModuleParser, currentClassloader) else CompositeModuleParsingPhasesManager(currentClassloader, systemModuleParser)
    val compiler: RuntimeModuleNodeCompiler = RuntimeModuleNodeCompiler.chain(currentClassloader, localLevel, systemModuleCompiler, parentLast = !systemFirst)
    ModuleComponents(new TwoLevelWeaveResourceResolver(systemLevel, dynamicLevel), parser, compiler)
  }
}

object DynamicModuleComponentFactory {
  def apply(): DynamicModuleComponentFactory = {
    new DynamicModuleComponentFactory(ClassLoaderWeaveResourceResolver.noContextClassloader(), ClassLoaderWeaveResourceResolver.contextClassloaderOnly, true)
  }

  def apply(systemLevel: WeaveResourceResolver, dynamicLevel: () => WeaveResourceResolver, systemFirst: Boolean): DynamicModuleComponentFactory = {
    new DynamicModuleComponentFactory(systemLevel, dynamicLevel, systemFirst)
  }
}

class SimpleModuleComponentFactory(weaveResourceResolver: WeaveResourceResolver) extends ModuleComponentsFactory {

  /**
    * Handles the parsing of the modules that are on the SystemClassLoader
    */
  private lazy val moduleParser: ModuleParsingPhasesManager = {
    val moduleLoaderManager = ModuleLoaderManager(Seq(ModuleLoader(weaveResourceResolver)), SPIBasedModuleLoaderProvider)
    ModuleParsingPhasesManager(moduleLoaderManager)
  }

  /**
    * Handles the compilation of modules that are on the SystemClassLoader
    */
  private lazy val moduleCompiler: RuntimeModuleNodeCompiler = RuntimeModuleNodeCompiler(moduleParser, weaveResourceResolver)

  override def createComponents(): ModuleComponents = {
    ModuleComponents(weaveResourceResolver, moduleParser, moduleCompiler)
  }
}

class CompositeModuleComponentFactory(parent: ModuleComponentsFactory, child: ModuleComponentsFactory) extends ModuleComponentsFactory {
  override def createComponents(): ModuleComponents = {
    val parentComponent = parent.createComponents()
    val childComponent = child.createComponents()
    val weaveResourceResolver = CompositeWeaveResourceResolver(parentComponent.resourceResolver, childComponent.resourceResolver)
    val manager = CompositeModuleParsingPhasesManager(parentComponent.parser, childComponent.parser)
    val compiler = CompositeRuntimeModuleNodeCompiler(parentComponent.compiler, childComponent.compiler)
    ModuleComponents(weaveResourceResolver, manager, compiler)
  }
}

object SimpleModuleComponentFactory {

  def apply(weaveResourceResolver: WeaveResourceResolver): SimpleModuleComponentFactory = new SimpleModuleComponentFactory(weaveResourceResolver)

  def apply(): SimpleModuleComponentFactory = new SimpleModuleComponentFactory(ClassLoaderWeaveResourceResolver.apply())
}

object ModuleComponentsFactory {

  def wlangOnly(resolver: WeaveResourceResolver): ModuleComponentsFactory = {
    new SimpleModuleComponentFactory(new WLangOnlyWeaveResourceResolver(resolver))
  }

  def wlangOnly(): ModuleComponentsFactory = {
    new SimpleModuleComponentFactory(new WLangOnlyWeaveResourceResolver(ClassLoaderWeaveResourceResolver.apply()))
  }

  def apply(weaveResourceResolver: WeaveResourceResolver): ModuleComponentsFactory = {
    SimpleModuleComponentFactory(weaveResourceResolver)
  }

  def apply(): ModuleComponentsFactory = {
    SimpleModuleComponentFactory()
  }

  def twoLevel(parent: WeaveResourceResolver, child: WeaveResourceResolver, parentFirst: Boolean): ModuleComponentsFactory = {
    DynamicModuleComponentFactory(parent, () => child, parentFirst)
  }

  def withParent(parent: ModuleComponentsFactory): ModuleComponentsFactory = new CompositeModuleComponentFactory(parent, DynamicModuleComponentFactory())

  def composite(parent: ModuleComponentsFactory, child: ModuleComponentsFactory): ModuleComponentsFactory = new CompositeModuleComponentFactory(parent, child)

}
