package org.mule.weave.v2.interpreted.node.pattern

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.model.values.coercion.ArrayCoercer
import org.mule.weave.v2.runtime.core.functions.stringops.StringRegexMatchFunctionValue

import scala.util.matching.Regex

trait PatternNode extends ValueNode[(Array[Value[_]]) => Value[_]] {
  def matches(value: Value[_])(implicit ctx: ExecutionContext): Boolean

  def function: ValueNode[(Array[Value[_]]) => Value[_]]

  def call(value: Value[_])(implicit ctx: ExecutionContext): Value[_] = {
    function.execute.asInstanceOf[FunctionValue].call(value)
  }

  def requiresMaterialize: Boolean

  override def doExecute(implicit ctx: ExecutionContext): Value[(Array[Value[_]]) => Value[_]] = {
    function.execute
  }
}

class RegexPatternNode(pattern: ValueNode[Regex], val function: ValueNode[(Array[Value[_]]) => Value[_]]) extends PatternNode with Product2[ValueNode[_], ValueNode[_]] {

  override def _2: ValueNode[_] = function

  override def _1: ValueNode[_] = pattern

  override def call(value: Value[_])(implicit ctx: ExecutionContext): Value[_] = {
    val matchResult = StringRegexMatchFunctionValue.doExecute(StringType.coerce(value, pattern), pattern.execute)
    function.execute.asInstanceOf[FunctionValue].call(matchResult)
  }

  override def matches(value: Value[_])(implicit ctx: ExecutionContext): Boolean = new RegexPattern(pattern.execute).matches(value)

  override def requiresMaterialize: Boolean = false
}

class TypePatternNode(pattern: ValueNode[Type], val function: ValueNode[(Array[Value[_]]) => Value[_]], val typeRequiresMaterialize: Boolean) extends PatternNode with Product3[ValueNode[_], ValueNode[_], Boolean] {
  override def matches(value: Value[_])(implicit ctx: ExecutionContext): Boolean = new TypePattern(pattern.execute).matches(value)

  override def _2: ValueNode[_] = function

  override def _1: ValueNode[_] = pattern

  override def _3: Boolean = typeRequiresMaterialize

  override def requiresMaterialize: Boolean = typeRequiresMaterialize

}

class LiteralPatternNode(pattern: ValueNode[_], val function: ValueNode[(Array[Value[_]]) => Value[_]]) extends PatternNode with Product2[ValueNode[_], ValueNode[_]] {
  override def matches(value: Value[_])(implicit ctx: ExecutionContext): Boolean = new LiteralPattern(pattern.execute).matches(value)

  override def _2: ValueNode[_] = function

  override def _1: ValueNode[_] = pattern

  override def requiresMaterialize: Boolean = true
}

class ExpressionPatternNode(pattern: ValueNode[(Array[Value[_]]) => Value[_]], val function: ValueNode[(Array[Value[_]]) => Value[_]]) extends PatternNode with Product2[ValueNode[_], ValueNode[_]] {
  override def matches(value: Value[_])(implicit ctx: ExecutionContext): Boolean = new ExpressionPattern(pattern.execute).matches(value)

  override def _2: ValueNode[_] = function

  override def _1: ValueNode[_] = pattern

  override def requiresMaterialize: Boolean = true
}

class EmptyArrayPatternNode(val function: ValueNode[(Array[Value[_]]) => Value[_]]) extends PatternNode with Product1[ValueNode[(Array[Value[_]]) => Value[_]]] {
  override def matches(value: Value[_])(implicit ctx: ExecutionContext): Boolean = {
    EmptyArrayPattern.matches(value)
  }

  override def call(value: Value[_])(implicit ctx: ExecutionContext): Value[_] = {
    function.execute.asInstanceOf[FunctionValue].callInline()
  }

  override def requiresMaterialize: Boolean = false

  override def _1: ValueNode[(Array[Value[_]]) => Value[_]] = function
}

class DeconstructArrayPatternNode(val function: ValueNode[(Array[Value[_]]) => Value[_]]) extends PatternNode with Product1[ValueNode[(Array[Value[_]]) => Value[_]]] {
  override def matches(value: Value[_])(implicit ctx: ExecutionContext): Boolean = {
    ArrayPattern.matches(value)
  }

  override def call(value: Value[_])(implicit ctx: ExecutionContext): Value[_] = {
    val (head, tailSeq) = ArrayCoercer.coerceToArraySeq(value, this).headTail()
    val tail = ArrayValue(tailSeq, value)
    function.execute.asInstanceOf[FunctionValue].call(head, tail)
  }

  override def requiresMaterialize: Boolean = false

  override def _1: ValueNode[(Array[Value[_]]) => Value[_]] = function
}

class EmptyObjectPatternNode(val function: ValueNode[(Array[Value[_]]) => Value[_]]) extends PatternNode with Product1[ValueNode[(Array[Value[_]]) => Value[_]]] {
  override def matches(value: Value[_])(implicit ctx: ExecutionContext): Boolean = {
    EmptyObjectPattern.matches(value)
  }

  override def call(value: Value[_])(implicit ctx: ExecutionContext): Value[_] = {
    function.execute.asInstanceOf[FunctionValue].callInline()
  }

  override def requiresMaterialize: Boolean = false

  override def _1: ValueNode[(Array[Value[_]]) => Value[_]] = function
}

class DeconstructObjectPatternNode(val function: ValueNode[(Array[Value[_]]) => Value[_]]) extends PatternNode with Product1[ValueNode[(Array[Value[_]]) => Value[_]]] {
  override def matches(value: Value[_])(implicit ctx: ExecutionContext): Boolean = {
    ObjectPattern.matches(value)
  }

  override def call(value: Value[_])(implicit ctx: ExecutionContext): Value[_] = {
    val (head, tail) = ObjectType.coerce(value, this).evaluate.headTail()
    function.execute.asInstanceOf[FunctionValue].call(head._1, head._2, ObjectValue(tail))
  }

  override def requiresMaterialize: Boolean = false

  override def _1: ValueNode[(Array[Value[_]]) => Value[_]] = function
}

class DefaultPatternNode(val function: ValueNode[(Array[Value[_]]) => Value[_]]) extends PatternNode with Product1[ValueNode[_]] {

  override def _1: ValueNode[_] = function

  override def matches(value: Value[_])(implicit ctx: ExecutionContext): Boolean = true

  override def requiresMaterialize: Boolean = false
}
