package org.mule.weave.v2.compilation.loader

import org.mule.weave.v2.compilation.mapper.ParserInputBasedSerializerContext
import org.mule.weave.v2.compilation.mapper.ast.ModuleNodeSerializer
import org.mule.weave.v2.compilation.message.WeaveBinaryDeserializationError
import org.mule.weave.v2.compilation.serializer.SerializableAstNodeSerializer
import org.mule.weave.v2.parser.{ MessageCollector, SafeStringBasedParserInput }
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.phase._
import org.mule.weave.v2.sdk.{ BinaryWeaveResource, WeaveResourceResolver, WeaveResourceResolverAware }

import java.io.{ ByteArrayInputStream, DataInputStream }

class WeaveBinaryResourceModuleLoader extends ModuleLoader with WeaveResourceResolverAware {
  private var resolver: WeaveResourceResolver = _

  override def name(): Option[String] = Some("bdwl")

  /**
    * Builds if possible the module ast for the given NameIdentifier
    *
    * @param nameIdentifier The name of the module to be build
    * @param parentContext  The parsing context for this module
    * @return If was able to build the ParsingResult
    */
  override def loadModule(nameIdentifier: NameIdentifier, parentContext: ParsingContext): Option[PhaseResult[ParsingResult[ModuleNode]]] = {
    parentContext.notificationManager.startResolvingResource(nameIdentifier, resolver)
    val resolve = resolver.resolveBinary(nameIdentifier)
    parentContext.notificationManager.endResolvingResource(nameIdentifier, resolver, resolve)
    resolve.map(resource => parseBinaryModule(nameIdentifier, resource))
  }

  override def resolver(resolver: WeaveResourceResolver): Unit = {
    this.resolver = resolver
  }

  private def parseBinaryModule(identifier: NameIdentifier, resource: BinaryWeaveResource): PhaseResult[ParsingResult[ModuleNode]] = {
    val inputStream = new ByteArrayInputStream(resource.binaryContent)
    val dataInputStream = new DataInputStream(inputStream)
    val parserInput = SafeStringBasedParserInput(resource.content())
    val input: ParsingContentInput = ParsingContentInput(resource, identifier, parserInput)
    try {
      val deserializedNode = SerializableAstNodeSerializer.deserialize(dataInputStream)
      val moduleNode = ModuleNodeSerializer.deserialize(deserializedNode, ParserInputBasedSerializerContext(parserInput, identifier))
      new PhaseResult(Some(ParsingResult(input, moduleNode)), MessageCollector())
    } catch {
      case e: Throwable => new PhaseResult(None, MessageCollector().error(WeaveBinaryDeserializationError(identifier, e.getMessage), UnknownLocation))
    }
  }

  override def canResolveModule(nameIdentifier: NameIdentifier): Boolean = resolver.resolveBinary(nameIdentifier).isDefined
}
