package org.mule.weave.v2.runtime

import org.mule.weave.v2.interpreted.transform.phase.ConstantFoldingPhase
import org.mule.weave.v2.interpreted.BinaryCompilationMappingLoaderPhase
import org.mule.weave.v2.interpreted.BinaryCompilationModuleLoaderPhase
import org.mule.weave.v2.interpreted.InterpreterMappingCompilerPhase
import org.mule.weave.v2.interpreted.InterpreterModuleCompilerPhase
import org.mule.weave.v2.interpreted.InterpreterPreCompilerPhase
import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ModuleAstCloningPhase
import org.mule.weave.v2.parser.phase.CommonSubexpressionReductionPhase
import org.mule.weave.v2.parser.phase.CompilationPhase
import org.mule.weave.v2.parser.phase.LoggingContextInjectionPhase
import org.mule.weave.v2.parser.phase.ParsingContentInput
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.ScopeGraphResult
import org.mule.weave.v2.parser.MappingParser
import org.mule.weave.v2.parser.ModuleParser
import org.mule.weave.v2.sdk.BinaryWeaveResource
import org.mule.weave.v2.sdk.WeaveResource

object WeaveCompiler {

  private def fullPreCompilerPhase(): CompilationPhase[ParsingContentInput, ScopeGraphResult[DocumentNode]] = {
    MappingParser.typeCheckPhase()
      .chainWith(RuntimeDocumentNodeCompiler.preCompilerPhase())
  }

  private def fullCompilerPhase(moduleNodeLoader: RuntimeModuleNodeCompiler): CompilationPhase[ParsingContentInput, CompilationResult[DocumentNode]] = {
    MappingParser.typeCheckPhase()
      .chainWith(RuntimeDocumentNodeCompiler.compilerPhase(moduleNodeLoader))
  }

  private def fullModulePreCompilerPhase(): CompilationPhase[ParsingContentInput, ScopeGraphResult[ModuleNode]] = {
    ModuleParser.typeCheckPhase()
      .chainWith(RuntimeModuleCompiler.preCompilerPhase())
  }

  private def fullModuleCompilerPhase(moduleNodeLoader: RuntimeModuleNodeCompiler): CompilationPhase[ParsingContentInput, CompilationResult[ModuleNode]] = {
    ModuleParser.typeCheckPhase()
      .chainWith(RuntimeModuleCompiler.compilerPhase(moduleNodeLoader))
  }

  private def noCheckPreCompilerPhase(): CompilationPhase[ParsingContentInput, ScopeGraphResult[DocumentNode]] = {
    MappingParser.scopePhase()
      .chainWith(new CommonSubexpressionReductionPhase[DocumentNode]())
      .chainWith(new ConstantFoldingPhase[DocumentNode]())
      .chainWith(new LoggingContextInjectionPhase[DocumentNode]())
      .chainWith(RuntimeDocumentNodeCompiler.preCompilerPhase())
  }

  private def noCheckCompilerPhase(moduleNodeLoader: RuntimeModuleNodeCompiler): CompilationPhase[ParsingContentInput, CompilationResult[DocumentNode]] = {
    MappingParser.scopePhase()
      .chainWith(RuntimeDocumentNodeCompiler.compilerPhase(moduleNodeLoader))
  }

  private def noCheckModulePreCompilerPhase(): CompilationPhase[ParsingContentInput, ScopeGraphResult[ModuleNode]] = {
    ModuleParser.scopePhase()
      .chainWith(new CommonSubexpressionReductionPhase[ModuleNode]())
      .chainWith(new ConstantFoldingPhase[ModuleNode]())
      .chainWith(new LoggingContextInjectionPhase[ModuleNode]())
      .chainWith(RuntimeModuleCompiler.preCompilerPhase())
  }

  private def noCheckModuleCompilerPhase(moduleNodeLoader: RuntimeModuleNodeCompiler): CompilationPhase[ParsingContentInput, CompilationResult[ModuleNode]] = {
    ModuleParser.scopePhase().chainWith(RuntimeModuleCompiler.compilerPhase(moduleNodeLoader))
  }

  def compileWithNoCheck(input: WeaveResource, parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): PhaseResult[CompilationResult[DocumentNode]] = {
    MappingParser
      .parse(noCheckCompilerPhase(moduleNodeLoader), input, parsingContext)
  }

  def compileWithNoCheck(input: WeaveResource, parsingContext: ParsingContext): PhaseResult[CompilationResult[DocumentNode]] = {
    MappingParser.parse(noCheckCompilerPhase(RuntimeModuleNodeCompiler()), input, parsingContext)
  }

  def preCompile(input: WeaveResource, parsingContext: ParsingContext): PhaseResult[ScopeGraphResult[DocumentNode]] = {
    MappingParser.parse(fullPreCompilerPhase(), input, parsingContext)
  }

  def preCompileWithNoCheck(input: WeaveResource, parsingContext: ParsingContext): PhaseResult[ScopeGraphResult[DocumentNode]] = {
    MappingParser.parse(noCheckPreCompilerPhase(), input, parsingContext)
  }

  def compile(input: WeaveResource, parsingContext: ParsingContext): PhaseResult[CompilationResult[DocumentNode]] = {
    MappingParser.parse(fullCompilerPhase(RuntimeModuleNodeCompiler()), input, parsingContext)
  }

  def compile(input: WeaveResource, parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): PhaseResult[CompilationResult[DocumentNode]] = {
    MappingParser.parse(fullCompilerPhase(moduleNodeLoader), input, parsingContext)
  }

  def preCompileModule(input: WeaveResource, parsingContext: ParsingContext): PhaseResult[ScopeGraphResult[ModuleNode]] = {
    ModuleParser.parse(fullModulePreCompilerPhase(), input, parsingContext)
  }

  def preCompileModuleWithNoCheck(input: WeaveResource, parsingContext: ParsingContext): PhaseResult[ScopeGraphResult[ModuleNode]] = {
    ModuleParser.parse(noCheckModulePreCompilerPhase(), input, parsingContext)
  }

  def compileModule(input: WeaveResource, parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): PhaseResult[CompilationResult[ModuleNode]] = {
    ModuleParser.parse(fullModuleCompilerPhase(moduleNodeLoader), input, parsingContext)
  }

  def compileModuleWithNoCheck(input: WeaveResource, parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): PhaseResult[CompilationResult[ModuleNode]] = {
    ModuleParser.parse(noCheckModuleCompilerPhase(moduleNodeLoader), input, parsingContext)
  }

  def runtimeModuleCompilation(moduleNode: ScopeGraphResult[ModuleNode], parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): PhaseResult[CompilationResult[ModuleNode]] = {
    RuntimeModuleCompiler
      .compilerPhase(moduleNodeLoader)
      .call(moduleNode, parsingContext)
  }

  def runtimeCompilation(documentNode: ScopeGraphResult[DocumentNode], parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): PhaseResult[CompilationResult[DocumentNode]] = {
    RuntimeDocumentNodeCompiler
      .compilerPhase(moduleNodeLoader)
      .call(documentNode, parsingContext)
  }

  def compileBinary(weaveResource: BinaryWeaveResource, parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): PhaseResult[CompilationResult[DocumentNode]] = {
    new BinaryCompilationMappingLoaderPhase()
      .chainWith(new InterpreterMappingCompilerPhase(moduleNodeLoader))
      .call(weaveResource, parsingContext)
  }

  def compileModuleBinary(nameIdentifier: NameIdentifier, weaveResource: BinaryWeaveResource, parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): PhaseResult[CompilationResult[ModuleNode]] = {
    new BinaryCompilationModuleLoaderPhase()
      .chainWith(new InterpreterModuleCompilerPhase(moduleNodeLoader))
      .call((nameIdentifier, weaveResource), parsingContext)
  }
}

object RuntimeModuleCompiler {

  def preCompilerPhase(): CompilationPhase[ScopeGraphResult[ModuleNode], ScopeGraphResult[ModuleNode]] = {
    new ModuleAstCloningPhase()
      .chainWith(new CommonSubexpressionReductionPhase[ModuleNode]())
      .chainWith(new ConstantFoldingPhase[ModuleNode]())
      .chainWith(new LoggingContextInjectionPhase[ModuleNode]())
      .chainWith(new InterpreterPreCompilerPhase[ModuleNode, ScopeGraphResult[ModuleNode]]())
  }

  def compilerPhase(moduleNodeLoader: RuntimeModuleNodeCompiler): CompilationPhase[ScopeGraphResult[ModuleNode], CompilationResult[ModuleNode]] = {
    new ModuleAstCloningPhase()
      .chainWith(new CommonSubexpressionReductionPhase[ModuleNode]())
      .chainWith(new ConstantFoldingPhase[ModuleNode]())
      .chainWith(new LoggingContextInjectionPhase[ModuleNode]())
      .chainWith(new InterpreterPreCompilerPhase[ModuleNode, ScopeGraphResult[ModuleNode]]())
      .chainWith(new InterpreterModuleCompilerPhase(moduleNodeLoader))
  }
}

object RuntimeDocumentNodeCompiler {

  def preCompilerPhase(): CompilationPhase[ScopeGraphResult[DocumentNode], ScopeGraphResult[DocumentNode]] = {
    new CommonSubexpressionReductionPhase[DocumentNode]()
      .chainWith(new ConstantFoldingPhase[DocumentNode]())
      .chainWith(new LoggingContextInjectionPhase[DocumentNode]())
      .chainWith(new InterpreterPreCompilerPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
  }

  def compilerPhase(moduleNodeLoader: RuntimeModuleNodeCompiler): CompilationPhase[ScopeGraphResult[DocumentNode], CompilationResult[DocumentNode]] = {
    new CommonSubexpressionReductionPhase[DocumentNode]()
      .chainWith(new ConstantFoldingPhase[DocumentNode]())
      .chainWith(new LoggingContextInjectionPhase[DocumentNode]())
      .chainWith(new InterpreterPreCompilerPhase[DocumentNode, ScopeGraphResult[DocumentNode]]())
      .chainWith(new InterpreterMappingCompilerPhase(moduleNodeLoader))
  }
}
