package org.mule.weave.v2.compilation.mapper.nonAst

import org.mule.weave.v2.compilation.exception.{ AstNodeAnnotationDeSerializationException, AstNodeAnnotationSerializationException }
import org.mule.weave.v2.compilation.mapper.{ AstToSerializableMapper, DWSerializer, SerializerContext }
import org.mule.weave.v2.compilation.ArraySerializableAstNode
import org.mule.weave.v2.compilation.NodeSerializableAstNode
import org.mule.weave.v2.compilation.SerializableAstNode
import org.mule.weave.v2.compilation.SerializableAstNodeKind
import org.mule.weave.v2.compilation.SerializableAstNodeLocation
import org.mule.weave.v2.compilation.UnknownLocation
import org.mule.weave.v2.interpreted.marker.ConstantArgumentAnnotation
import org.mule.weave.v2.interpreted.marker.DoBlockNodeNonShadowedVariablesAnnotation
import org.mule.weave.v2.interpreted.marker.ExternalBindingAnnotation
import org.mule.weave.v2.interpreted.marker.FunctionCallNameAnnotation
import org.mule.weave.v2.interpreted.marker.InterceptorInvalidReferenceAnnotation
import org.mule.weave.v2.interpreted.marker.InterceptorOnNativeFunctionAnnotation
import org.mule.weave.v2.interpreted.marker.InterceptorOnReferenceAnnotation
import org.mule.weave.v2.interpreted.marker.LazyVarDirectiveAnnotation
import org.mule.weave.v2.interpreted.marker.LiteralNodeAnnotation
import org.mule.weave.v2.interpreted.marker.NativeFunctionAnnotation
import org.mule.weave.v2.interpreted.marker.OverloadedFunctionNotCacheableAnnotation
import org.mule.weave.v2.interpreted.marker.ParameterReferenceFunctionCallArgumentAnnotation
import org.mule.weave.v2.interpreted.marker.ReferenceAnnotation
import org.mule.weave.v2.interpreted.marker.RequiresMaterializationAnnotation
import org.mule.weave.v2.interpreted.marker.StaticFunctionCallAnnotation
import org.mule.weave.v2.interpreted.marker.StaticRecFunctionCallAnnotation
import org.mule.weave.v2.interpreted.marker.TypeParameterNodeAnnotation
import org.mule.weave.v2.parser.annotation.MetadataKeyAstNodeAnnotation
import org.mule.weave.v2.parser.annotation.{ AstNodeAnnotation, DataFormatExtensionAnnotation, DataFormatSettings, DescendantsIncludeThisAnnotation, InjectedNodeAnnotation, MaterializeTypeAnnotation, MaterializeVariableAnnotation, QuotedStringAnnotation, SinceAstNodeAnnotation, StreamingCapableVariableAnnotation, TailRecFunctionAnnotation, TailRecFunctionCallAnnotation, UntrustedCodeAstNodeAnnotation }

object AstNodeAnnotationSerializer extends DWSerializer[AstNodeAnnotation] {
  override def serialize(node: AstNodeAnnotation): NodeSerializableAstNode = {
    node match {
      case MaterializeVariableAnnotation(needMaterialize, _) =>
        val children = Seq(serializeBoolean(needMaterialize))
        toNodeSerializableAstNode(SerializableAstNodeKind.MATERIALIZE_VARIABLE_ANNOTATION_NODE, children = children)
      case MaterializeTypeAnnotation(needMaterialize) =>
        val children = Seq(serializeBoolean(needMaterialize))
        toNodeSerializableAstNode(SerializableAstNodeKind.MATERIALIZE_TYPE_ANNOTATION_NODE, children = children)
      case QuotedStringAnnotation(quoteChar) =>
        val children = Seq(serializeString(quoteChar.toString))
        toNodeSerializableAstNode(SerializableAstNodeKind.QUOTED_STRING_ANNOTATION_NODE, children = children)
      case SinceAstNodeAnnotation(version) =>
        val sversionNode = SVersionSerializer.serialize(version)
        toNodeSerializableAstNode(SerializableAstNodeKind.SINCE_AST_NODE_ANNOTATION_NODE, children = Seq(sversionNode))
      case StreamingCapableVariableAnnotation(canStream, _) =>
        val children = Seq(serializeBoolean(canStream))
        toNodeSerializableAstNode(SerializableAstNodeKind.STREAMING_CAPABLE_VARIABLE_ANNOTATION_NODE, children = children)
      case DescendantsIncludeThisAnnotation()                 => toNodeSerializableAstNode(SerializableAstNodeKind.DESCENDANTS_INCLUDE_THIS_ANNOTATION_NODE)
      case InjectedNodeAnnotation()                           => toNodeSerializableAstNode(SerializableAstNodeKind.INJECTED_NODE_ANNOTATION_NODE)
      case TailRecFunctionAnnotation()                        => toNodeSerializableAstNode(SerializableAstNodeKind.TAIL_REC_FUNCTION_ANNOTATION_NODE)
      case ConstantArgumentAnnotation()                       => toNodeSerializableAstNode(SerializableAstNodeKind.CONSTANT_ARGUMENT_ANNOTATION_NODE)
      case DoBlockNodeNonShadowedVariablesAnnotation()        => toNodeSerializableAstNode(SerializableAstNodeKind.DO_BLOCK_NODE_NODE_NON_SHADOWED_VARIABLES_ANNOTATION_NODE)
      case LazyVarDirectiveAnnotation()                       => toNodeSerializableAstNode(SerializableAstNodeKind.LAZY_VAR_DIRECTIVE_ANNOTATION_NODE)
      case LiteralNodeAnnotation()                            => toNodeSerializableAstNode(SerializableAstNodeKind.LITERAL_NODE_ANNOTATION_NODE)
      case NativeFunctionAnnotation()                         => toNodeSerializableAstNode(SerializableAstNodeKind.NATIVE_FUNCTION_ANNOTATION_NODE)
      case OverloadedFunctionNotCacheableAnnotation()         => toNodeSerializableAstNode(SerializableAstNodeKind.OVERLOADED_FUNCTION_NOT_CACHEABLE_ANNOTATION_NODE)
      case ParameterReferenceFunctionCallArgumentAnnotation() => toNodeSerializableAstNode(SerializableAstNodeKind.PARAMETER_REFERENCE_FUNCTION_CALL_ARGUMENT_ANNOTATION_NODE)
      case RequiresMaterializationAnnotation()                => toNodeSerializableAstNode(SerializableAstNodeKind.REQUIRES_MATERIALIZATION_ANNOTATION_NODE)
      case StaticRecFunctionCallAnnotation()                  => toNodeSerializableAstNode(SerializableAstNodeKind.STATIC_REC_FUNCTION_CALL_ANNOTATION_NODE)
      case TypeParameterNodeAnnotation()                      => toNodeSerializableAstNode(SerializableAstNodeKind.TYPE_PARAMETER_NODE_ANNOTATION_NODE)
      case ExternalBindingAnnotation(localVariableName) =>
        val children = Seq(serializeString(localVariableName))
        toNodeSerializableAstNode(SerializableAstNodeKind.EXTERNAL_BINDING_ANNOTATION_NODE, children = children)
      case FunctionCallNameAnnotation(functionName) =>
        val children = Seq(serializeString(functionName))
        toNodeSerializableAstNode(SerializableAstNodeKind.FUNCTION_CALL_NAME_ANNOTATION_NODE, children = children)
      case InterceptorInvalidReferenceAnnotation(invalidReference) =>
        val children = Seq(InvalidReferenceValueSerializer.serialize(invalidReference))
        toNodeSerializableAstNode(SerializableAstNodeKind.INTERCEPTOR_INVALID_REFERENCE_ANNOTATION_NODE, children = children)
      case InterceptorOnNativeFunctionAnnotation(nativeFunctionValue) =>
        val children = Seq(InterceptorNativeFunctionValueSerializer.serialize(nativeFunctionValue))
        toNodeSerializableAstNode(SerializableAstNodeKind.INTERCEPTOR_ON_NATIVE_FUNCTION_ANNOTATION, children = children)
      case InterceptorOnReferenceAnnotation(referenceVal) =>
        val children = Seq(ReferenceValueSerializer.serialize(referenceVal))
        toNodeSerializableAstNode(SerializableAstNodeKind.INTERCEPTOR_ON_REFERENCE_ANNOTATION_NODE, children = children)
      case ReferenceAnnotation(referenceVal) =>
        val children = Seq(ReferenceValueSerializer.serialize(referenceVal))
        toNodeSerializableAstNode(SerializableAstNodeKind.REFERENCE_ANNOTATION_NODE, children = children)
      case StaticFunctionCallAnnotation(arities) =>
        val children = Seq(ArraySerializableAstNode(arities.map(serializeInt)))
        toNodeSerializableAstNode(SerializableAstNodeKind.STATIC_FUNCTION_CALL_ANNOTATION_NODE, children = children)
      case UntrustedCodeAstNodeAnnotation(privileges) =>
        val children = Seq(ArraySerializableAstNode(privileges.map(serializeString)))
        toNodeSerializableAstNode(SerializableAstNodeKind.UNTRUSTED_CODE_AST_NODE_ANNOTATION, children = children)
      case DataFormatExtensionAnnotation(settings) =>
        val children = Seq(DataFormatSettingsValueSerializer.serialize(settings))
        toNodeSerializableAstNode(SerializableAstNodeKind.DATA_FORMAT_EXTENSION_ANNOTATION_NODE, children = children)
      case TailRecFunctionCallAnnotation() => toNodeSerializableAstNode(SerializableAstNodeKind.TAIL_REC_FUNCTION_CALL_ANNOTATION_NODE)
      case MetadataKeyAstNodeAnnotation(metadataKeyName) =>
        val children = Seq(serializeString(metadataKeyName))
        toNodeSerializableAstNode(SerializableAstNodeKind.METADATA_KEY_NODE_ANNOTATION, children = children)
      case _ => throw AstNodeAnnotationSerializationException(s"Can't serialize annotation of class: ${node.getClass.getName}")
    }
  }

  override def deserialize(node: SerializableAstNode, context: SerializerContext): AstNodeAnnotation = {
    node.kind() match {
      case SerializableAstNodeKind.DESCENDANTS_INCLUDE_THIS_ANNOTATION_NODE => DescendantsIncludeThisAnnotation()
      case SerializableAstNodeKind.INJECTED_NODE_ANNOTATION_NODE            => InjectedNodeAnnotation()
      case SerializableAstNodeKind.TAIL_REC_FUNCTION_ANNOTATION_NODE        => TailRecFunctionAnnotation()
      case SerializableAstNodeKind.MATERIALIZE_VARIABLE_ANNOTATION_NODE =>
        val shouldMaterialize = deserializeBoolean(node.children().head)
        MaterializeVariableAnnotation(shouldMaterialize)
      case SerializableAstNodeKind.MATERIALIZE_TYPE_ANNOTATION_NODE =>
        val shouldMaterialize = deserializeBoolean(node.children().head)
        MaterializeTypeAnnotation(shouldMaterialize)
      case SerializableAstNodeKind.QUOTED_STRING_ANNOTATION_NODE =>
        val quoteChar = deserializeString(node.children().head).toCharArray.head
        QuotedStringAnnotation(quoteChar)
      case SerializableAstNodeKind.SINCE_AST_NODE_ANNOTATION_NODE =>
        val sversionNode = node.children().head
        val sversion = SVersionSerializer.deserialize(sversionNode, context)
        SinceAstNodeAnnotation(sversion)
      case SerializableAstNodeKind.STREAMING_CAPABLE_VARIABLE_ANNOTATION_NODE =>
        val canStream = deserializeBoolean(node.children().head)
        StreamingCapableVariableAnnotation(canStream)
      case SerializableAstNodeKind.CONSTANT_ARGUMENT_ANNOTATION_NODE                          => ConstantArgumentAnnotation()
      case SerializableAstNodeKind.DO_BLOCK_NODE_NODE_NON_SHADOWED_VARIABLES_ANNOTATION_NODE  => DoBlockNodeNonShadowedVariablesAnnotation()
      case SerializableAstNodeKind.LAZY_VAR_DIRECTIVE_ANNOTATION_NODE                         => LazyVarDirectiveAnnotation()
      case SerializableAstNodeKind.LITERAL_NODE_ANNOTATION_NODE                               => LiteralNodeAnnotation()
      case SerializableAstNodeKind.NATIVE_FUNCTION_ANNOTATION_NODE                            => NativeFunctionAnnotation()
      case SerializableAstNodeKind.OVERLOADED_FUNCTION_NOT_CACHEABLE_ANNOTATION_NODE          => OverloadedFunctionNotCacheableAnnotation()
      case SerializableAstNodeKind.PARAMETER_REFERENCE_FUNCTION_CALL_ARGUMENT_ANNOTATION_NODE => ParameterReferenceFunctionCallArgumentAnnotation()
      case SerializableAstNodeKind.REQUIRES_MATERIALIZATION_ANNOTATION_NODE                   => RequiresMaterializationAnnotation()
      case SerializableAstNodeKind.STATIC_REC_FUNCTION_CALL_ANNOTATION_NODE                   => StaticRecFunctionCallAnnotation()
      case SerializableAstNodeKind.TYPE_PARAMETER_NODE_ANNOTATION_NODE                        => TypeParameterNodeAnnotation()
      case SerializableAstNodeKind.EXTERNAL_BINDING_ANNOTATION_NODE =>
        val name = AstToSerializableMapper.deserializeString(node.children().head)
        ExternalBindingAnnotation(name)
      case SerializableAstNodeKind.FUNCTION_CALL_NAME_ANNOTATION_NODE =>
        val name = AstToSerializableMapper.deserializeString(node.children().head)
        FunctionCallNameAnnotation(name)
      case SerializableAstNodeKind.INTERCEPTOR_INVALID_REFERENCE_ANNOTATION_NODE =>
        val invalidRef = InvalidReferenceValueSerializer.deserialize(node.children().head, context)
        InterceptorInvalidReferenceAnnotation(invalidRef)
      case SerializableAstNodeKind.INTERCEPTOR_ON_NATIVE_FUNCTION_ANNOTATION =>
        val invalidRef = InterceptorNativeFunctionValueSerializer.deserialize(node.children().head, context)
        InterceptorOnNativeFunctionAnnotation(invalidRef)
      case SerializableAstNodeKind.INTERCEPTOR_ON_REFERENCE_ANNOTATION_NODE =>
        val refValue = ReferenceValueSerializer.deserialize(node.children().head, context)
        InterceptorOnReferenceAnnotation(refValue)
      case SerializableAstNodeKind.REFERENCE_ANNOTATION_NODE =>
        val refValue = ReferenceValueSerializer.deserialize(node.children().head, context)
        ReferenceAnnotation(refValue)
      case SerializableAstNodeKind.STATIC_FUNCTION_CALL_ANNOTATION_NODE =>
        val aritiesNode = node.children().head.asInstanceOf[ArraySerializableAstNode]
        val arities = aritiesNode.children.map(deserializeInt).toList
        StaticFunctionCallAnnotation(arities)
      case SerializableAstNodeKind.UNTRUSTED_CODE_AST_NODE_ANNOTATION =>
        val privilegesNode = node.children().head.asInstanceOf[ArraySerializableAstNode]
        val privileges = privilegesNode.children.map(deserializeString).toList
        UntrustedCodeAstNodeAnnotation(privileges)
      case SerializableAstNodeKind.TAIL_REC_FUNCTION_CALL_ANNOTATION_NODE => TailRecFunctionCallAnnotation()
      case SerializableAstNodeKind.DATA_FORMAT_EXTENSION_ANNOTATION_NODE =>
        val dataFormatSettingsValue = DataFormatSettingsValueSerializer.deserialize(node.children().head, context)
        DataFormatExtensionAnnotation(dataFormatSettingsValue)
      case SerializableAstNodeKind.METADATA_KEY_NODE_ANNOTATION =>
        val metadataKeyValue = deserializeString(node.children().head)
        MetadataKeyAstNodeAnnotation(metadataKeyValue)
      case _ => throw AstNodeAnnotationDeSerializationException(s"Can't deserialize annotation of kind: ${node.kind()}")
    }
  }

  private def toNodeSerializableAstNode(kind: Short, location: SerializableAstNodeLocation = UnknownLocation, children: Seq[SerializableAstNode] = Seq()): NodeSerializableAstNode = {
    NodeSerializableAstNode(kind, location, children)
  }
}
