package org.mule.weave.v2.interpreted.debugger.server

import javax.xml.namespace.QName

import org.mule.weave.v2.core.functions.WriteFunctionValue
import org.mule.weave.v2.debugger._
import org.mule.weave.v2.interpreted.debugger.server.DebuggerConverters.toDebuggerLocation
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.AttributesCapable
import org.mule.weave.v2.model.structure._
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.TypeValue
import org.mule.weave.v2.model.values.Value

object DebuggerValueFactory {

  def toString(qualifiedName: QualifiedName)(implicit ctx: EvaluationContext): String = {
    val namespace: Option[Namespace] = qualifiedName.namespace
    new QName(namespace.map(_.uri).orNull, qualifiedName.name, namespace.map(_.prefix).getOrElse("")).toString
  }

  def toAttributeValue(nameValuePair: NameValuePair, maxElements: Int, maxDepth: Int, currentDepth: Int)(implicit ctx: EvaluationContext): AttributeDebuggerValue = {
    val qualifiedName: QualifiedName = nameValuePair._1.evaluate
    AttributeDebuggerValue(toString(qualifiedName), DebuggerValueFactory.create(nameValuePair._2, maxElements, maxDepth, currentDepth), toDebuggerLocation(nameValuePair._1.location()))
  }

  def toKeyDebuggerValue(keyValue: Value[QualifiedName], maxElements: Int, maxDepth: Int, currentDepth: Int)(implicit ctx: EvaluationContext): KeyDebuggerValue = {
    val attributes = keyValue match {
      case ac: AttributesCapable => {
        val attributes: Option[Value[NameSeq]] = ac.attributes
        attributes
          .map(_.evaluate.toIterator().map((nameValue) => toAttributeValue(nameValue, maxElements, maxDepth, currentDepth)).toArray)
          .getOrElse(Array[AttributeDebuggerValue]())
      }
      case _ => Array[AttributeDebuggerValue]()
    }

    KeyDebuggerValue(toString(keyValue.evaluate), attributes, toDebuggerLocation(keyValue.location()))
  }

  def toFieldDebuggerValue(kvp: KeyValuePair, maxElements: Int, maxDepth: Int, currentDepth: Int)(implicit ctx: EvaluationContext): FieldDebuggerValue = {
    FieldDebuggerValue(toKeyDebuggerValue(kvp._1, maxElements, maxDepth, currentDepth), DebuggerValueFactory.create(kvp._2, maxElements, maxDepth, currentDepth), toDebuggerLocation(kvp._1.location()))
  }

  def create(value: Value[_], maxElements: Int, maxDepth: Int, currentDepth: Int = 0)(implicit ctx: EvaluationContext): DebuggerValue = {
    value match {
      case ov: Value[_] if ObjectType.accepts(ov) => {
        val fields: Array[FieldDebuggerValue] = {
          if (currentDepth == maxDepth) {
            Array.empty
          } else {
            val pairs = ObjectType.coerce(ov).evaluate.toIterator().slice(0, maxElements)
            pairs
              .map((value: KeyValuePair) => {
                toFieldDebuggerValue(value, maxElements, maxDepth, currentDepth + 1)
              })
              .toArray
          }
        }
        ObjectDebuggerValue(fields, StringType.coerce(TypeValue(ov.valueType, ov)).evaluate.toString, toDebuggerLocation(ov.location()))
      }
      case av: Value[_] if ArrayType.accepts(av) => {
        val items: Array[DebuggerValue] = {
          if (currentDepth == maxDepth) {
            Array.empty
          } else {
            val items = ArrayType.coerce(av).evaluate.toSeq().slice(0, maxElements)
            items
              .map((item) => {
                DebuggerValueFactory.create(item, maxElements, maxDepth, currentDepth + 1)
              })
              .toArray
          }
        }
        ArrayDebuggerValue(items, StringType.coerce(TypeValue(av.valueType, av)).evaluate.toString, toDebuggerLocation(av.location()))
      }
      case range: Value[_] if RangeType.accepts(range) => {
        val evaluate: RangeType.T = RangeType.coerce(range).evaluate
        SimpleDebuggerValue(s"[${evaluate.start}..${evaluate.end}]", StringType.coerce(TypeValue(range.valueType, range)).evaluate.toString, toDebuggerLocation(range.location()))
      }
      case func: Value[_] if FunctionType.accepts(func) => {
        val fv: FunctionValue = FunctionType.coerce(func)
        DebuggerFunction(fv.parameters.map((param) => param.name + ":" + param.wtype.name), StringType.coerce(TypeValue(func.valueType, func)).evaluate.toString, toDebuggerLocation(func.location()))
      }
      case nullVal if NullType.accepts(nullVal) => {
        SimpleDebuggerValue("null", StringType.coerce(TypeValue(nullVal.valueType, nullVal)).evaluate.toString, toDebuggerLocation(nullVal.location()))
      }
      case v => {
        SimpleDebuggerValue(WriteFunctionValue.toDwString(v, ignoreSchema = true), StringType.coerce(TypeValue(v.valueType, v)).evaluate.toString, toDebuggerLocation(v.location()))
      }
    }
  }
}
