package org.mule.weave.v2.interpreted.node.structure

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.types.BooleanType
import org.mule.weave.v2.model.types.KeyType
import org.mule.weave.v2.model.values.KeyValuePairValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.wrappers.LazyValue

class KeyValuePairNode(val key: ValueNode[_], val value: ValueNode[_], val cond: Option[ValueNode[_]], val literal: Boolean) extends ValueNode[KeyValuePair] with ConditionalCapableNode {

  var keyValuePairNode: KeyValuePair = _

  override def doExecute(implicit executionContext: ExecutionContext): Value[KeyValuePair] = {
    val result = toKeyValuePair(executionContext)
    KeyValuePairValue(result, this)
  }

  //Helper method to avoid generating Value object when is not needed
  def toKeyValuePair(implicit executionContext: ExecutionContext): KeyValuePair = {
    if (literal) {
      if (keyValuePairNode == null) {
        keyValuePairNode = createKeyValuePair
      }
      keyValuePairNode
    } else {
      createKeyValuePair
    }
  }

  private def createKeyValuePair(implicit executionContext: ExecutionContext) = {
    KeyValuePair(KeyType.coerce(key.execute(executionContext), this), value.execute(executionContext))
  }

  //Helper method to avoid generating Value object when is not needed
  def toLazyKeyValuePair(implicit executionContext: ExecutionContext): KeyValuePair = {
    if (literal) {
      if (keyValuePairNode == null) {
        keyValuePairNode = createKeyValuePair
      }
      keyValuePairNode
    } else {
      val frame = executionContext.executionStack().activeFrame()
      KeyValuePair(KeyType.coerce(key.execute(executionContext), this), LazyValue(executionContext.runInFrame(frame, value.execute(executionContext)), value))
    }
  }

  override def productElement(n: Int): Any = {
    n match {
      case 0 => key
      case 1 => value
      case 2 => cond.get
    }
  }

  override def productArity: Int = {
    if (cond.isDefined) 3 else 2
  }

  override def condition(implicit ctx: ExecutionContext): Boolean = {
    if (cond.isDefined) {
      val conditionalExpression = cond.get
      BooleanType.coerce(conditionalExpression.execute(ctx), this).evaluate(ctx)
    } else {
      true
    }
  }
}

object KeyValuePairNode {
  def apply(key: ValueNode[_], value: ValueNode[_], cond: Option[ValueNode[_]] = None, literal: Boolean = false) = new KeyValuePairNode(key, value, cond, literal)
}
