package org.mule.weave.v2.interpreted.node

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.structure.FrameBasedSchemaResolver
import org.mule.weave.v2.interpreted.node.structure.LiteralNameNode
import org.mule.weave.v2.interpreted.node.structure.NameNode
import org.mule.weave.v2.interpreted.node.structure.NamespaceNode
import org.mule.weave.v2.interpreted.node.structure.StringNode
import org.mule.weave.v2.interpreted.node.structure.schema.SchemaNode
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.types.ExtendedType
import org.mule.weave.v2.model.types.IntersectionType
import org.mule.weave.v2.model.types.KeyType
import org.mule.weave.v2.model.types.KeyValuePairType
import org.mule.weave.v2.model.types.NameType
import org.mule.weave.v2.model.types.NamespaceType
import org.mule.weave.v2.model.types.NothingType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.ReferenceType
import org.mule.weave.v2.model.types.SelectorReferenceType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.types.Types
import org.mule.weave.v2.model.types.UnionType
import org.mule.weave.v2.model.values.SchemaProvider
import org.mule.weave.v2.model.values.TypeValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.exception.WeaveRuntimeException
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.utils.AnnotationSchemaNode

import scala.annotation.tailrec

class TypeSelectorReferenceNode(val typeReferenceNode: ValueNode[Type], val fieldName: ValueNode[QualifiedName], val schemaConstraint: Option[SchemaNode], val metadata: Option[SchemaNode]) extends ValueNode[Type] with Product4[ValueNode[Type], ValueNode[QualifiedName], Option[SchemaNode], Option[SchemaNode]] {

  /**
    * Execute methods that does the real magic
    *
    * @param ctx The execution context
    * @return The new value
    */

  override protected def doExecute(implicit ctx: ExecutionContext): Value[Type] = {
    val activeFrame = ctx.executionStack().activeFrame()
    val selectorN = selectorName(fieldName)
    val referenceN = referenceName(typeReferenceNode)
    val typeResolver = referenceResolver(activeFrame, fieldName, typeReferenceNode)
    val selectorReferenceType = SelectorReferenceType(referenceN, selectorN, typeResolver)
    val schemaProvider = metadata.map(m => {
      val typeSchemaResolver = new TypeSelectorReferenceNodeSchemaResolver(activeFrame, selectorReferenceType, m)
      SchemaProvider(typeSchemaResolver)
    })
    TypeValue(SelectorReferenceType(referenceN, selectorN, typeResolver).withSchema(schemaConstraint.map(_.execute.evaluate)), this, schemaProvider)
  }

  private def selectorName(fieldName: ValueNode[_]): String = {
    fieldName match {
      case strn: StringNode     => strn.str
      case lsn: LiteralNameNode => lsn.keyName.str
      case ns: NamespaceNode    => ns.prefix.name
      case fqn: NameNode        => fqn.ns.map(node => selectorName(node) + "#").getOrElse("") + selectorName(fqn.keyName)
    }
  }

  @tailrec
  private def referenceName(typeReferenceNode: ValueNode[Type]): String = {
    typeReferenceNode match {
      case typeSelectorRefNode: TypeSelectorReferenceNode => referenceName(typeSelectorRefNode.typeReferenceNode)
      case typeSelectorRefNode: TypeReferenceWithTypeParamNode => typeSelectorRefNode.name
      case referenceTypeNode: TypeReferenceNode => referenceTypeNode.variable.name
      case _ => throw new WeaveRuntimeException(s"Cannot do selection on Type: ${typeReferenceNode.toString}", typeReferenceNode.location())
    }
  }

  private def referenceResolver(frame: Frame, qnNode: ValueNode[QualifiedName], typeReferenceNode: ValueNode[Type]): EvaluationContext => Type = (ectx: EvaluationContext) => {
    implicit val context: ExecutionContext = ectx match {
      case executionContext: ExecutionContext =>
        executionContext
      case _ =>
        ExecutionContext(frame, ectx)
    }
    context.runInFrame(
      frame, {
      val keyName: Value[QualifiedName] = qnNode.execute
      val fqName: QualifiedName = keyName.evaluate
      val v = typeReferenceNode.execute.evaluate(context)
      selectType(fqName, fieldName.location(), v)
    })
  }

  private def resolveIntersectionOfObjects(types: Type)(implicit ctx: ExecutionContext): Option[Type] = {
    val allOfs = Types.normalizeType(types)
    val allOfsResolved = allOfs.map(intersection => {
      lazy val allObjs = intersection.forall(_.isInstanceOf[ObjectType])
      if (intersection.nonEmpty && allObjs) {
        intersection.asInstanceOf[Seq[ObjectType]].reduce((l, r) =>
          ObjectType(Types.intersectProperties(l.keyValuePairs, r.keyValuePairs), l.open || r.open))
      } else {
        //Selection is available only if all the types are Objects inside the intersection.
        NothingType
      }
    })
    if (allOfsResolved.nonEmpty && allOfsResolved.forall(_.isInstanceOf[ObjectType])) {
      if (allOfsResolved.length > 1) Some(UnionType(allOfsResolved)) else Some(allOfsResolved.head)
    } else {
      None
    }
  }

  private def selectType(fqn: QualifiedName, location: Location, targetType: Type)(implicit ctx: ExecutionContext): Type = {
    targetType match {
      case ObjectType(keyValuePairs, _) => findTypeWithQName(keyValuePairs, fqn).getOrElse(
        throw new WeaveRuntimeException("Key not found", location)).value
      case refT @ ReferenceType(_, _) => selectType(fqn, location, refT.referencedType)
      case extendedType: ExtendedType => selectType(fqn, location, extendedType.baseType)
      case UnionType(of) =>
        UnionType(of.map(typeFromUnion => {
          selectType(fqn, location, typeFromUnion)
        }))
      case IntersectionType(_) =>
        val obj = resolveIntersectionOfObjects(targetType)
        selectType(fqn, location, obj.getOrElse(throw new WeaveRuntimeException(s"Cannot do selection on Type: ${targetType.toString}", location)))
      case _ => throw new WeaveRuntimeException(s"Cannot do selection on Type: ${targetType.toString}", location)
    }
  }

  private def findTypeWithQName(keyValuePairs: Seq[KeyValuePairType], qn: QualifiedName): Option[KeyValuePairType] = {
    var result = false
    keyValuePairs.find(kvp => kvp.key match {
      case KeyType(Some(NameType(Some(name), maybeNs)), _) => {
        if (name == qn.name) {
          maybeNs match {
            case Some(NamespaceType(Some(uri))) =>
              result = qn.hasNamespace() && uri == qn.namespace.get.uri
            case _ => result = !qn.hasNamespace()
          }
        } else {
          result = false
        }
        result
      }
      case _ => false
    })
  }

  override def _1: ValueNode[Type] = typeReferenceNode

  override def _2: ValueNode[QualifiedName] = fieldName

  override def _3: Option[SchemaNode] = schemaConstraint

  override def _4: Option[SchemaNode] = metadata

  override def shouldNotify: Boolean = false

}

class TypeSelectorReferenceNodeSchemaResolver(val frame: Frame, referenceType: ReferenceType, metadata: SchemaNode) extends FrameBasedSchemaResolver {

  override def resolveSchema(executionContext: ExecutionContext): Option[Schema] = {
    val schema = metadata match {
      case value @ SchemaNode(_, AnnotationSchemaNode) =>
        mergeSchemas(value.execute(executionContext).evaluate(executionContext), referenceType.schema(executionContext))
      case _ =>
        metadata.execute(executionContext).evaluate(executionContext)
    }
    Some(schema)
  }

  private def mergeSchemas(annotationSchemas: Schema, mayBeSpecifiedSchemas: Option[Schema]): Schema = {
    if (mayBeSpecifiedSchemas.isDefined) {
      val specifiedSchemas = mayBeSpecifiedSchemas.get
      Schema(annotationSchemas, specifiedSchemas)
    } else {
      annotationSchemas
    }
  }
}