package org.mule.weave.v2.compilation.loader

import org.mule.weave.v2.interpreted.BinaryCompilationModuleLoaderPhase
import org.mule.weave.v2.parser.{ MessageCollector, SafeStringBasedParserInput }
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.phase.ModuleLoader.BINARY_LOADER_NAME
import org.mule.weave.v2.parser.phase._
import org.mule.weave.v2.sdk.{ BinaryWeaveResource, WeaveResourceResolver, WeaveResourceResolverAware }

class WeaveBinaryResourceModuleLoader extends ModuleLoader with WeaveResourceResolverAware {
  private var resolver: WeaveResourceResolver = _

  override def name(): Option[String] = Some(BINARY_LOADER_NAME)

  /**
    * Builds if possible the module ast for the given NameIdentifier
    *
    * @param nameIdentifier The name of the module to be build
    * @param parsingContext  The parsing context for this module
    * @return If was able to build the ParsingResult
    */
  override def loadModule(nameIdentifier: NameIdentifier, parsingContext: ParsingContext): Option[PhaseResult[ParsingResult[ModuleNode]]] = {
    parsingContext.notificationManager.startResolvingResource(nameIdentifier, resolver)
    val resolve = resolver.resolveBinary(nameIdentifier)
    parsingContext.notificationManager.endResolvingResource(nameIdentifier, resolver, resolve)
    resolve.map(resource => parseBinaryModule(nameIdentifier, resource, parsingContext))
  }

  override def resolver(resolver: WeaveResourceResolver): Unit = {
    this.resolver = resolver
  }

  private def parseBinaryModule(identifier: NameIdentifier, resource: BinaryWeaveResource, parsingContext: ParsingContext): PhaseResult[ParsingResult[ModuleNode]] = {
    val result = new BinaryCompilationModuleLoaderPhase().call((identifier, resource), parsingContext)
    if (result.noResult()) {
      FailureResult(result.messages())
    } else {
      val parserInput = SafeStringBasedParserInput(resource.content())
      val input: ParsingContentInput = ParsingContentInput(resource, identifier, parserInput)
      new PhaseResult(Some(ParsingResult(input, result.getResult().astNode)), MessageCollector())
    }
  }

  override def canResolveModule(nameIdentifier: NameIdentifier): Boolean = resolver.canResolveResource(nameIdentifier)
}
