package org.mule.weave.v2.interpreted

import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.END_READING
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.KIND_PROPERTY
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.START_READING
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.VARIABLE_PROPERTY
import org.mule.weave.v2.core.telemetry.service.api.TelemetryEvent.data
import org.mule.weave.v2.core.telemetry.service.api.TelemetryService
import org.mule.weave.v2.interpreted.node.structure.header.directives.DirectiveOption
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.structure.schema.SchemaProperty
import org.mule.weave.v2.model.values.BinaryValue
import org.mule.weave.v2.model.values.NullValue
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.wrappers.LazyValue
import org.mule.weave.v2.model.values.wrappers.WrapperValue
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.option.ConfigurableSchemaSetting
import org.mule.weave.v2.module.option.SchemaStreamAwareSetting
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.module.reader.SourceProviderAwareReader
import org.mule.weave.v2.module.writer.WriterHelper
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.module.MimeType

import java.io.File
import scala.util.Failure
import scala.util.Success
import scala.util.Try

class ReaderValue(reader: Reader, name: String, options: Seq[DirectiveOption]) extends WrapperValue with DumpCapable {

  private var readerValue: Value[_] = _
  private var shouldMaterialize = false
  private var valueSchema: Option[Schema] = _

  override def value(implicit ctx: EvaluationContext): Value[Any] = {
    if (readerValue == null) {

      val enabled: Boolean = ctx.serviceManager.settingsService.telemetry().enabled
      if (enabled) {
        val telemetryService: Option[TelemetryService] = ctx.serviceManager.telemetryService
        if (telemetryService.isDefined) {
          telemetryService.get
            .publishEvent(
              START_READING,
              location(),
              name,
              data(VARIABLE_PROPERTY, name, KIND_PROPERTY, reader.getName()))
        }
      }
      options.foreach((option) => {
        ConfigurationHelper.configure(reader, option)
      })
      if (shouldMaterialize) {
        readerValue = reader.read(name).materialize
      } else {
        readerValue = reader.read(name)
      }
      if (enabled) {
        val telemetryService: Option[TelemetryService] = ctx.serviceManager.telemetryService
        if (telemetryService.isDefined) {
          telemetryService.get
            .publishEvent(
              END_READING,
              location(),
              name,
              data(VARIABLE_PROPERTY, name, KIND_PROPERTY, reader.getName()))
        }
      }
    }
    readerValue
  }

  override def schema(implicit ctx: EvaluationContext): Option[Schema] = {
    if (valueSchema == null) {
      valueSchema = Some(Schema(buildSchema, super.schema.getOrElse(Schema.empty)))
    }
    valueSchema
  }

  private def buildSchema(implicit ctx: EvaluationContext) = {
    new ReaderValueSchema(reader, name)
  }

  override def location(): Location = Location(name)

  override def materialize(implicit ctx: EvaluationContext): Value[_] = {
    if (readerValue != null) {
      readerValue = readerValue.materialize
    } else {
      shouldMaterialize = true
    }
    this
  }

  override def dumpValue(intoFolder: File)(implicit ctx: EvaluationContext): Unit = {
    val dumpFile = new File(intoFolder, name + reader.dataFormat.get.fileExtensions.headOption.getOrElse(".dwl"))
    if (readerValue == null) {
      reader match {
        case spar: SourceProviderAwareReader => {
          spar.copyContent(dumpFile)
        }
        case _ =>
          Dumper.dumpUsingWeave(dumpFile, value)
      }
    } else {
      reader match {
        case spar: SourceProviderAwareReader => {
          if (spar.sourceProvider.consumedMultipleTimes) {
            spar.copyContent(dumpFile)
          } else {
            reader.dataFormat.foreach((df) => {
              val writer = df.writer(Some(dumpFile))
              val map = reader.settings.settingsValues()
              //Configure the reader based on the mimeType properties
              val options = df.writerOptions().iterator
              while (options.hasNext) {
                val keyValuePair = options.next()
                val param = map.get(keyValuePair._1.toLowerCase)
                if (param.isDefined) {
                  writer.setOption(UnknownLocation, keyValuePair._1, param.get)
                }
              }
              Try({
                WriterHelper.writeValue(writer, readerValue, UnknownLocationCapable)
              })
            })
          }
        }
        case _ => {
          Dumper.dumpUsingWeave(dumpFile, readerValue)
        }
      }
    }
    reader.settings match {
      case configurableSchema: ConfigurableSchemaSetting => configurableSchema.dumpSchema(intoFolder, "readerSchema")
      case flatFileSchema: SchemaStreamAwareSetting => flatFileSchema.dumpFlatFileSchema(intoFolder, ctx, "schemaFlatFileReader.ffd")
      case _ =>
    }
  }
}

class ReaderValueSchema(val typedValue: Reader, name: String)(implicit ctx: EvaluationContext) extends Schema {

  private val sourceProvider = typedValue match {
    case reader: SourceProviderAwareReader => {
      Some(reader.sourceProvider)
    }
    case _ => None
  }

  private lazy val rawValue = sourceProvider match {
    //If it can not be consumed multiple times then we shouldn't return it
    case Some(sourceProvider) if (sourceProvider.consumedMultipleTimes) => {
      DataFormatManager.byContentType(MimeType.JAVA_MIME_TYPE) match {
        case Some(dataFormat) => {
          dataFormat.reader(SourceProvider(sourceProvider.underling)).read(name + "." + Schema.RAW_PROPERTY_NAME).materialize
        }
        case None => {
          val asInputStream = Try(sourceProvider.asInputStream)
          asInputStream match {
            case Failure(_)           => NullValue
            case Success(inputStream) => BinaryValue(inputStream).materialize
          }
        }
      }
    }
    case _ => NullValue
  }

  private lazy val encoding = sourceProvider match {
    case Some(sourceProvider) => StringValue(sourceProvider.charset.name())
    case None                 => NullValue
  }

  private lazy val mediaType = sourceProvider match {
    case Some(sourceProvider) => {
      sourceProvider.mimeType match {
        case Some(mimeType) => {
          StringValue(mimeType.toString)
        }
        case None => NullValue
      }
    }
    case None => NullValue
  }

  override def valueOf(propertyName: String)(implicit ctx: EvaluationContext): Option[Value[Any]] = {
    propertyName match {
      case Schema.RAW_PROPERTY_NAME        => Some(LazyValue(rawValue))
      case Schema.ENCODING_PROPERTY_NAME   => Some(encoding)
      case Schema.MEDIA_TYPE_PROPERTY_NAME => Some(mediaType)
      case _                               => None
    }
  }

  override def properties()(implicit ctx: EvaluationContext): Seq[SchemaProperty] = {
    val properties = Seq(
      SchemaProperty(StringValue(Schema.RAW_PROPERTY_NAME), LazyValue(rawValue), internal = true),
      SchemaProperty(StringValue(Schema.ENCODING_PROPERTY_NAME), encoding),
      SchemaProperty(StringValue(Schema.MEDIA_TYPE_PROPERTY_NAME), mediaType))
    properties
  }
}
