package org.mule.weave.v2.runtime.core.operator.math

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.runtime.core.exception.AdditionExecutionException
import org.mule.weave.v2.runtime.core.operator.math.exception.InvalidTemporalAddition

import java.time.DateTimeException

class NumberAdditionOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = NumberType

  override val R = NumberType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val leftNumber = leftValue.evaluate
    val rightNumber = rightValue.evaluate
    NumberValue(doAddition(leftNumber, rightNumber), this)
  }

  def doAddition(leftNumber: math.Number, rightNumber: math.Number): math.Number = {
    try {
      leftNumber + rightNumber
    } catch {
      case ae: ArithmeticException => {
        throw new AdditionExecutionException(leftNumber.toString, rightNumber.toString, ae.toString, location)
      }
    }
  }
}

class LocalDateAdditionPeriodOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = LocalDateType

  override val R = PeriodType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      LocalDateValue(leftValue.evaluate.plus(rightValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalAddition(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = leftArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }
}

class PeriodAdditionLocalDateOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = PeriodType

  override val R = LocalDateType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      LocalDateValue(rightValue.evaluate.plus(leftValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalAddition(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = rightArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }

}

class LocalDateTimeAdditionPeriodOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = LocalDateTimeType

  override val R = PeriodType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      LocalDateTimeValue(leftValue.evaluate.plus(rightValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalAddition(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = leftArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }

}

class PeriodAdditionLocalDateTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = PeriodType

  override val R = LocalDateTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      LocalDateTimeValue(rightValue.evaluate.plus(leftValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalAddition(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = rightArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }
}

class DateTimeAdditionPeriodOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = DateTimeType

  override val R = PeriodType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      DateTimeValue(leftValue.evaluate.plus(rightValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalAddition(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = leftArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }
}

class PeriodAdditionDateTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = PeriodType

  override val R = DateTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      DateTimeValue(rightValue.evaluate.plus(leftValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalAddition(e.getMessage, location)
      }
    }
  }

  override def allowUseCachedOnCoerce(leftArgumentType: Type, rightArgumentType: Type): Boolean = {
    val allow = rightArgumentType match {
      case _: StringType => false
      case _             => true
    }
    allow
  }
}

class TimeAdditionPeriodOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = TimeType

  override val R = PeriodType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      TimeValue(leftValue.evaluate.plus(rightValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalAddition(e.getMessage, location)
      }
    }
  }
}

class PeriodAdditionTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = PeriodType

  override val R = TimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      TimeValue(rightValue.evaluate.plus(leftValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalAddition(e.getMessage, location)
      }
    }
  }
}

class LocalTimeAdditionPeriodOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = LocalTimeType

  override val R = PeriodType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      LocalTimeValue(leftValue.evaluate.plus(rightValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalAddition(e.getMessage, location)
      }
    }
  }
}

class PeriodAdditionLocalTimeOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = PeriodType

  override val R = LocalTimeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    try {
      LocalTimeValue(rightValue.evaluate.plus(leftValue.evaluate), this)
    } catch {
      case e @ (_: DateTimeException | _: ArithmeticException) => {
        throw new InvalidTemporalAddition(e.getMessage, location)
      }
    }
  }
}

class ArrayAdditionAnyOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = ArrayType

  override val R = AnyType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    ArrayValue(leftValue.evaluate.append(rightValue), this)
  }
}
