package org.mule.weave.v2.runtime.tools

import org.mule.weave.v2.core.RuntimeConfigProperties
import org.mule.weave.v2.core.versioning.CompatibilityFlag
import org.mule.weave.v2.core.versioning.Setting
import org.mule.weave.v2.core.versioning.SystemSetting
import org.mule.weave.v2.model.service.DefaultSettingsService
import org.mule.weave.v2.utils.DataWeaveVersion

import java.io.BufferedWriter
import java.io.File
import java.io.FileWriter

object SystemPropertiesDocGenerator {
  def main(args: Array[String]): Unit = {
    if (args.length == 0) {
      println("Missing target directory param or DW version")
      return
    }
    val folder = args(0)
    val targetFolder = new File(folder)
    if (targetFolder.isFile) {
      println("Target should be a directory")
      return
    }

    targetFolder.mkdirs()

    val settings = DefaultSettingsService.settings()

    val systemSettings = settings.flatMap(_.settings()) ++ RuntimeConfigProperties.runtimeConfigProperties

    val targetFile = new File(targetFolder, "dataweave-system-properties.adoc")

    println(s"Creating documentation for system properties at ${targetFile.getAbsolutePath}")

    val docHeader = s"""|= DataWeave System Properties
                       |
                       |DataWeave supports several system properties. To use these properties in application development and deployments, see xref:mule-runtime:mule-app-properties-system.adoc[System Properties].
                       |
                       |[%header%autowidth.spread,cols=",a"]
                       ||===
                       ||Property |Description\n""".stripMargin

    val docEnding = "|==="

    val entries: Seq[String] = systemSettings.filter(!_.internal).sortBy(_.fullName).map(generateEntry)

    val content = Seq(docHeader, entries.mkString("\n"), docEnding).mkString("\n")

    val writer = new BufferedWriter(new FileWriter(targetFile))
    try {
      writer.write(content)
    } finally {
      writer.close()
    }
  }

  private def generateEntry(setting: Setting[_]): String = {
    setting match {
      case flag: CompatibilityFlag   => generateCompatibilityFlagEntry(flag)
      case setting: SystemSetting[_] => generateSystemSettingEntry(setting)
      case _                         => ""
    }
  }

  private def generateSystemSettingEntry(systemSetting: SystemSetting[_]): String = {
    val description = if (systemSetting.experimental) {
      s"_Experimental:_ ${systemSetting.description} \n"
    } else {
      systemSetting.description
    }

    val experimentalDisclaimer = if (systemSetting.experimental) {
      s"`\n${systemSetting.fullName}` is an _experimental feature_ that is subject to change or removal from future versions of DataWeave. See xref:dataweave-troubleshoot.adoc[]."
    } else {
      ""
    }

    s"""|//${systemSetting.fullName}//
       ||`${systemSetting.fullName}`|
       |$description\n
       |* Type: `${systemSetting.defaultValue.getClass.getSimpleName.capitalize}`
       |* Default: `${systemSetting.defaultValueDisplay.getOrElse(systemSetting.defaultValue)}`
       |$experimentalDisclaimer\n""".stripMargin
  }

  private def generateCompatibilityFlagEntry(cf: CompatibilityFlag): String = {
    val llRange = generateLanguageLevelRange(cf)

    s"""|//${cf.fullName}//
        ||`${cf.fullName}`|\n
        |${cf.description}\n
        |* Type: `${cf.defaultValue.getClass.getSimpleName.capitalize}`\n
        |* Default:\n
        |${generateDefaultValues(cf)}
        |Available for language levels: $llRange\n\n""".stripMargin
  }

  private def generateLanguageLevelRange(cf: CompatibilityFlag): String = {
    val runtimeVersion = DataWeaveVersion()
    var currentVersion: DataWeaveVersion = cf.creationVersion
    var range: Seq[DataWeaveVersion] = Seq()

    while (currentVersion < cf.deletionVersion && currentVersion <= runtimeVersion) {
      range = range :+ currentVersion
      currentVersion = DataWeaveVersion(currentVersion.major, currentVersion.minor + 1)
    }

    range.map(_.toString).mkString(", ")
  }

  private def generateDefaultValues(cf: CompatibilityFlag): String = {
    val runtimeVersion = DataWeaveVersion()
    if (cf.behaviorChangeVersion == cf.deletionVersion) {
      return cf.initialValue.toString
    }

    var initialValueRange: Seq[DataWeaveVersion] = Seq()
    var newBehaviorValueRange: Seq[DataWeaveVersion] = Seq()

    var currentVersion: DataWeaveVersion = cf.creationVersion

    while (currentVersion < cf.deletionVersion && currentVersion <= runtimeVersion) {
      if (currentVersion < cf.behaviorChangeVersion || cf.initialValue == cf.newBehaviorValue) {
        initialValueRange = initialValueRange :+ currentVersion
      } else {
        newBehaviorValueRange = newBehaviorValueRange :+ currentVersion
      }

      currentVersion = DataWeaveVersion(currentVersion.major, currentVersion.minor + 1)
    }

    val initialValRangeStr = initialValueRange.map(_.toString).mkString(", ")
    val newBehaviorValRangeStr = newBehaviorValueRange.map(_.toString).mkString(", ")

    var res = ""

    if (initialValRangeStr.nonEmpty) {
      res += s"   - `${cf.defaultValueDisplay.getOrElse(cf.initialValue)}` for language levels: $initialValRangeStr\n\n"
    }

    if (newBehaviorValRangeStr.nonEmpty) {
      res += s"   - `${cf.defaultValueDisplay.getOrElse(cf.newBehaviorValue)}` for language levels: $newBehaviorValRangeStr\n"
    }

    res
  }
}
