package org.mule.weave.v2.interpreted.node.structure.function

import org.mule.weave.v2.core.functions.BaseBinaryFunctionValue
import org.mule.weave.v2.core.functions.BaseTernaryFunctionValue
import org.mule.weave.v2.core.functions.BaseUnaryFunctionValue
import org.mule.weave.v2.core.functions.BinaryInterceptedFunctionValue
import org.mule.weave.v2.core.functions.EmptyFunctionValue
import org.mule.weave.v2.core.functions.EmptyInterceptedFunctionValue
import org.mule.weave.v2.core.functions.InterceptedFunctionValue
import org.mule.weave.v2.core.functions.TernaryInterceptedFunctionValue
import org.mule.weave.v2.core.functions.UnaryInterceptedFunctionValue
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value

/**
  * This nodes wraps the function to do function interception. This is useful for things like
  *  - PermissionValidation
  *  - Caching
  *  - Tracing
  *
  * @param interceptedFunctionNode The function node
  * @param annotationArgs          The annotation arguments
  */
class InterceptedCheckFunctionDecoratorNode(interceptedFunctionNode: ValueNode[_], interceptorFunction: ValueNode[_], annotationValueArgs: ValueNode[_], functionName: StringValue) extends ValueNode[(Array[Value[_]]) => Value[_]] {

  override protected def doExecute(implicit ctx: ExecutionContext): Value[(Array[Value[_]]) => Value[_]] = {
    val value: Value[_] = interceptedFunctionNode.execute(ctx)
    val interceptor: FunctionValue = FunctionType.coerce(interceptorFunction.execute)

    val annotationArgs: Value[_] = annotationValueArgs.execute

    value match {
      case empty: EmptyFunctionValue => {
        new EmptyInterceptedFunctionValue(empty, annotationArgs, functionName, interceptor)
      }
      case unary: BaseUnaryFunctionValue => {
        new UnaryInterceptedFunctionValue(unary, annotationArgs, functionName, interceptor)
      }
      case binary: BaseBinaryFunctionValue => {
        new BinaryInterceptedFunctionValue(binary, annotationArgs, functionName, interceptor)
      }
      case ternary: BaseTernaryFunctionValue => {
        new TernaryInterceptedFunctionValue(ternary, annotationArgs, functionName, interceptor)
      }
      case fv: FunctionValue => {
        new InterceptedFunctionValue(fv, annotationArgs, functionName, interceptor)
      }
    }

  }
}
