package org.mule.weave.v2.interpreted.node.structure.function

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.node.ExecutionNode
import org.mule.weave.v2.interpreted.node.NameSlot
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.structure.TypeNode
import org.mule.weave.v2.model.types.AnyType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values.FunctionParameter
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.location.WeaveLocation

import scala.annotation.switch

class DynamicFunctionNode(params: Array[FunctionParameterNode], body: ValueNode[_], returnTypeNode: Option[ValueNode[Type]], requiresMaterializedArguments: Boolean, val name: Option[String] = None, literalParameters: Boolean = false)

    extends ValueNode[(Array[Value[_]]) => Value[_]]
    with Product5[Seq[FunctionParameterNode], ValueNode[_], Boolean, Option[String], Boolean] {

  private var functionParams: Array[FunctionParameter] = _

  val minParams: Int = params.count((param) => param.defaultValue.isEmpty)

  val maxParams: Int = params.length

  //This is for injected functions
  override def location(): WeaveLocation = _location.getOrElse(body.location())

  override def _1: Seq[FunctionParameterNode] = params

  override def _2: ValueNode[_] = body

  override def _3: Boolean = requiresMaterializedArguments

  override def _4: Option[String] = name

  override def _5: Boolean = literalParameters

  override protected def doExecute(implicit ctx: ExecutionContext): FunctionValue = {
    val frame = ctx.executionStack().activeFrame()

    (params.length: @switch) match {
      case 0 => new EmptyFunctionExecutionContextAwareFunction(frame, body, returnTypeNode, this, name)
      case 1 => {
        val parameters = calculateFunctionParams(ctx)
        new UnaryFunctionExecutionContextAwareFunction(parameters.head, params.head, frame, body, returnTypeNode, this, name, minParams, requiresMaterializedArguments)
      }
      case 2 => {
        val parameters = calculateFunctionParams(ctx)
        new BinaryFunctionExecutionContextAwareFunction(parameters(0), parameters(1), params(0), params(1), frame, body, returnTypeNode, this, name, minParams, requiresMaterializedArguments)
      }
      case 3 => {
        val parameters = calculateFunctionParams(ctx)
        new TernaryFunctionExecutionContextAwareFunction(parameters(0), parameters(1), parameters(2), params(0), params(1), params(2), frame, body, returnTypeNode, this, name, minParams, requiresMaterializedArguments)
      }
      case 4 => {
        val parameters = calculateFunctionParams(ctx)
        new QuaternaryFunctionExecutionContextAwareFunction(
          parameters(0),
          parameters(1),
          parameters(2),
          parameters(3),
          params(0),
          params(1),
          params(2),
          params(3),
          frame,
          body,
          returnTypeNode,
          this,
          name,
          minParams,
          requiresMaterializedArguments)
      }
      case _ => DefaultExecutionContextAwareFunction(calculateFunctionParams(ctx), params, body, returnTypeNode, name, frame, this, minParams, maxParams, requiresMaterializedArguments)
    }

  }

  private def calculateFunctionParams(ctx: ExecutionContext): Array[FunctionParameter] = {
    if (literalParameters) {
      if (functionParams == null) {
        functionParams = ExecutionContextAwareFunctionValue.calculateParams(params, ctx)
      }
      functionParams
    } else {
      ExecutionContextAwareFunctionValue.calculateParams(params, ctx)
    }
  }

  override def shouldNotify: Boolean = false
}

object DynamicFunctionNode {
  def apply(params: Array[FunctionParameterNode], body: ValueNode[_], returnTypeNode: Option[ValueNode[Type]], requiresMaterializedArguments: Boolean, name: Option[String] = None, literalParameters: Boolean = false): DynamicFunctionNode = new DynamicFunctionNode(params, body, returnTypeNode, requiresMaterializedArguments, name, literalParameters)
}

case class FunctionParameterNode(variable: NameSlot, wtype: ValueNode[Type], defaultValue: Option[ValueNode[_]] = None, materialize: Boolean, typeRequiresMaterialize: Boolean) extends ExecutionNode with Product4[NameSlot, ValueNode[Type], Option[ValueNode[_]], Boolean] {
  override def _1: NameSlot = variable

  override def _2: ValueNode[Type] = wtype

  override def _3: Option[ValueNode[_]] = defaultValue

  override def _4: Boolean = materialize
}

object FunctionParameterNode {
  def apply(variable: NameSlot, materialize: Boolean): FunctionParameterNode = {
    new FunctionParameterNode(variable, new TypeNode(AnyType, None), materialize = materialize, typeRequiresMaterialize = false)
  }

  def apply(variable: NameSlot, wtype: TypeNode, materialize: Boolean) = new FunctionParameterNode(variable, wtype, materialize = materialize, typeRequiresMaterialize = false)
}
