package org.mule.weave.v2.runtime.core.operator.selectors

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.core.exception.InvalidSelectionException
import org.mule.weave.v2.core.exception.KeyNotFoundException
import org.mule.weave.v2.core.exception.StrictSelectionOverNullSelection
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.NameType
import org.mule.weave.v2.model.types.NullType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.parser.location.WeaveLocation

class KeyMultiValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = ObjectType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    leftValue.evaluate.allKeyValuesOf(rightValue) match {
      case Some(values) =>
        val selectedValues = values.toIterator().map((kvp) => {
          AttributeDelegateValue(kvp)
        })
        ArrayValue(selectedValues, this)
      case _ => throw InvalidSelectionException(new KeyNotFoundException(this, rightValue.evaluate))
    }
  }

}

object KeyMultiValueSelectorOperator {
  def apply(location: WeaveLocation): KeyMultiValueSelectorOperator = new KeyMultiValueSelectorOperator(location)
}

class ArrayMultiValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = ArrayType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val objects = leftValue.evaluate
      .toIterator()
      .filter(ObjectType.accepts(_))
    //We double flatten so that the descendant story with .. * works
    val result: Iterator[AttributeDelegateValue] = objects.flatMap((objectValue: Value[_]) => {
      objectValue.evaluate.asInstanceOf[ObjectSeq].allKeyValuesOf(rightValue)
        .map((values) => {
          values.toSeq().map((kvp) => {
            AttributeDelegateValue(kvp)
          })
        }).iterator
    }).flatten
    if (result.hasNext) {
      ArrayValue(result, this)
    } else {
      throw InvalidSelectionException(new KeyNotFoundException(this, rightValue.evaluate))
    }
  }

}

class NullMultiValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = NullType

  override val R = NameType

  override def doExecute(leftValue: Value[Null], rightValue: Value[QualifiedName])(implicit ctx: EvaluationContext) = {
    throw InvalidSelectionException(new StrictSelectionOverNullSelection(this))
  }
}

object ArrayMultiValueSelectorOperator {
  def apply(location: WeaveLocation): ArrayMultiValueSelectorOperator = new ArrayMultiValueSelectorOperator(location)
}

