package org.mule.weave.v2.interpreted

import org.mule.weave.v2.compilation.mapper.{ LazyPositionFactory, LazyPositionResolverSerializerContext }
import org.mule.weave.v2.compilation.mapper.ast.DocumentNodeSerializer
import org.mule.weave.v2.compilation.message.WeaveBinaryDeserializationError
import org.mule.weave.v2.compilation.serializer.SerializableAstNodeSerializer
import org.mule.weave.v2.parser.MessageCollector
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.phase._
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.sdk.BinaryWeaveResource

import java.io.{ ByteArrayInputStream, DataInputStream }

class BinaryCompilationMappingLoaderPhase extends CompilationPhase[BinaryWeaveResource, AstNodeResultAware[DocumentNode] with AstNavigatorResultAware] {

  override def doCall(resource: BinaryWeaveResource, context: ParsingContext): PhaseResult[_ <: AstNodeResultAware[DocumentNode] with AstNavigatorResultAware] = {
    val dataInputStream = new DataInputStream(new ByteArrayInputStream(resource.binaryContent))
    val serializerContext = LazyPositionResolverSerializerContext(new LazyPositionFactory(resource.sourceWeaveResource), context.nameIdentifier)
    try {
      val serializableAstNode = SerializableAstNodeSerializer.deserialize(dataInputStream)
      val documentNode = DocumentNodeSerializer.deserialize(serializableAstNode, serializerContext)
      new PhaseResult(Option.apply(new BinaryCompilationMappingResult(documentNode)), context.messageCollector)
    } catch {
      case e: Throwable => FailureResult(MessageCollector().error(WeaveBinaryDeserializationError(context.nameIdentifier, e.getMessage), UnknownLocation))
    } finally {
      try {
        if (serializerContext != null) {
          serializerContext.close()
        }
      } catch {
        case _: Exception =>
      }
      try {
        if (dataInputStream != null) {
          dataInputStream.close()
        }
      } catch {
        case _: Exception =>
      }
    }
  }
}

class BinaryCompilationMappingResult(private val documentNode: DocumentNode) extends AstNodeResultAware[DocumentNode] with AstNavigatorResultAware {
  private lazy val _astNavigator: AstNavigator = AstNavigator.apply(documentNode)

  override def astNode: DocumentNode = documentNode

  override def astNavigator: AstNavigator = _astNavigator
}

