package org.mule.weave.v2.interpreted.transform

import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.interpreted.node.{ ExecutionNode, ModuleNode }
import org.mule.weave.v2.interpreted.node.structure.DocumentNode
import org.mule.weave.v2.interpreted.node.structure.header.ExternalBindings
import org.mule.weave.v2.interpreted.node.structure.header.VariableTable
import org.mule.weave.v2.parser
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.exception.WeaveRuntimeException
import org.mule.weave.v2.parser.location.Location
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.runtime.exception.InternalErrorException
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.utils.SchemaNodeOrigin

import scala.collection.mutable.ArrayBuffer

class EngineGrammarTransformation(val parsingContext: ParsingContext, val moduleLoader: RuntimeModuleNodeCompiler)
    extends EngineDirectiveTransformations
    with EngineDocumentTransformations
    with EngineFunctionTransformations
    with EngineHeaderTransformations
    with EnginePatternTransformations
    with EngineSchemaTransformations
    with EngineStructureTransformations
    with EngineVariableTransformations
    with EngineModuleTransformations
    with EngineUndefinedTransformations
    with EngineUpdaterTransformations {

  var rootElement: AstNode = _

  override val _variablesTable: VariableTable = new VariableTable()

  override val _modulesNameTable: VariableTable = new VariableTable()

  override val _externalBindings: ExternalBindings = new ExternalBindings()

  def transformDocument(document: parser.ast.structure.DocumentNode): DocumentNode = {
    val transformationStack = new TransformationStack()
    rootElement = document
    try {
      transform(document, transformationStack)
    } finally {
      rootElement = null
    }
  }

  def transformModule(module: parser.ast.module.ModuleNode): ModuleNode = {
    val transformationStack = new TransformationStack()
    rootElement = module
    try {
      transform(module, transformationStack)
    } finally {
      rootElement = null
    }
  }

  def transform[T <: ExecutionNode](astNode: parser.ast.AstNode, transformationStack: TransformationStack): T = {
    transformationStack.pushNode(astNode)
    val executionNode = astNode match {
      case baseAstNode: AstNode => {
        implicit val location: Location = astNode.location()

        val result = baseAstNode match {
          //
          case dn: parser.ast.structure.DocumentNode => transformDocumentNode(dn, transformationStack)
          case mn: parser.ast.module.ModuleNode => transformModuleNode(mn, transformationStack)
          //
          case parser.ast.header.HeaderNode(directives) => transformHeaderNode(directives, transformationStack)
          case parser.ast.UndefinedExpressionNode() => transformUndefinedNode()
          //
          case parser.ast.header.directives.NamespaceDirective(prefix, uri, _) => transformNamespaceDirective(prefix, uri, transformationStack)
          case parser.ast.header.directives.VersionMajor(v) => transformVersionMajor(v)
          case parser.ast.header.directives.DirectiveOption(name, value) => transformDirectiveOption(name, value, transformationStack)
          case parser.ast.header.directives.TypeDirective(variable, params, t, codeAnnotations) => transformTypeDirective(variable, params, t, codeAnnotations, transformationStack)
          case parser.ast.header.directives.VersionDirective(major, minor, _) => transformVersionDirective(major, minor, transformationStack)
          case parser.ast.header.directives.OutputDirective(id, mime, options, _, _) => transformOutputDirective(id, mime, options, transformationStack)
          case parser.ast.header.directives.VersionMinor(v) => transformVersionMinor(v)
          case parser.ast.header.directives.VarDirective(variable, value, _, annotations) => transformVarDirective(variable, value, annotations, transformationStack)
          case fn @ parser.ast.header.directives.FunctionDirectiveNode(variable, literal, codeAnnotations) => transformFunctionDirective(variable, literal, codeAnnotations, fn, transformationStack)
          //
          case parser.ast.patterns.PatternMatcherNode(lhs, patterns, _) => transformPatternMatcherNode(lhs, patterns, transformationStack)

          case parser.ast.patterns.ExpressionPatternNode(pattern, name, function) => transformExpressionPatternNode(pattern, name, function, transformationStack)
          case parser.ast.patterns.LiteralPatternNode(pattern, name, function) => transformLiteralPatternNode(pattern, name, function, transformationStack)
          case parser.ast.patterns.RegexPatternNode(pattern, name, function) => transformRegexPatternNode(pattern, name, function, transformationStack)
          case parser.ast.patterns.DefaultPatternNode(value, name) => transformDefaultPatternNode(name, value, transformationStack)
          case tpn: parser.ast.patterns.TypePatternNode => transformTypePatternNode(tpn, transformationStack)
          case parser.ast.patterns.EmptyArrayPatternNode(function) => transformEmptyArrayNode(function, transformationStack)
          case parser.ast.patterns.DeconstructArrayPatternNode(head, tail, function) => transformDeconstructArrayNode(head, tail, function, transformationStack)

          case parser.ast.patterns.EmptyObjectPatternNode(function) => transformEmptyObjectNode(function, transformationStack)
          case parser.ast.patterns.DeconstructObjectPatternNode(headKey, headValue, tail, function) => transformDeconstructObjectNode(headKey, headValue, tail, function, transformationStack)
          //
          case parser.ast.structure.DateTimeNode(v, _) => transformDateTimeNode(v)
          case sn: parser.ast.structure.StringNode => transformStringNode(sn)

          case parser.ast.structure.TimeNode(v, _) => transformTimeNode(v)
          case parser.ast.structure.StringInterpolationNode(v) => transformStringInterpolationNode(v, transformationStack)
          case parser.ast.structure.LocalDateTimeNode(v, _) => transformLocalDateTimeNode(v)
          case un @ parser.ast.structure.UriNode(v, _) => transformUriNode(v, un)
          case parser.ast.structure.ConditionalNode(value, cond) => transformConditionalNode(value, cond, transformationStack)
          case parser.ast.structure.PeriodNode(v, _) => transformPeriodNode(v)
          case parser.ast.structure.RegexNode(regex, _) => transformRegexNode(regex)
          case trn: parser.ast.types.TypeReferenceNode => transformTypeReferenceNode(trn, transformationStack = transformationStack)
          case wType: parser.ast.types.WeaveTypeNode => transformTypeNode(wType, transformationStack = transformationStack)
          case parser.ast.structure.TimeZoneNode(v, _) => transformTimeZoneNode(v)
          case parser.ast.structure.LocalDateNode(v, _) => transformLocalDateNode(v)
          case parser.ast.structure.NullNode(_) => transformNullNode()
          case an: parser.ast.structure.ArrayNode => transformArrayNode(an, transformationStack)
          case parser.ast.structure.HeadTailArrayNode(head, tail, _) => transformHeadTailArrayNode(head, tail, transformationStack)
          case parser.ast.structure.HeadTailObjectNode(headKey, headValue, tail, _) => transformHeadTailObjectNode(headKey, headValue, tail, transformationStack)

          case parser.ast.structure.AttributesNode(attrs) => transformAttributesNode(attrs, transformationStack)
          case on: parser.ast.structure.ObjectNode => transformObjectNode(on, transformationStack)
          case fn: parser.ast.functions.FunctionNode => transformFunctionNode(fn, transformationStack = transformationStack)
          case parser.ast.structure.NameValuePairNode(key, value, cond) => transformNameValuePairNode(key, value, cond, transformationStack)
          case nn: parser.ast.structure.NamespaceNode => transformNamespaceNode(nn)
          case fp: parser.ast.functions.FunctionParameter => transformFunctionParameter(fp, transformationStack)
          case parser.ast.structure.LocalTimeNode(v, _) => transformLocalTimeNode(v)
          case parser.ast.structure.KeyValuePairNode(key, value, cond) => transformKeyValuePairNode(key, value, cond, transformationStack)
          case parser.ast.structure.NameNode(keyName, cond, _) => transformNameNode(keyName, cond, transformationStack)
          case parser.ast.structure.DynamicNameNode(keyName) => transformDynamicNameNode(keyName, transformationStack)
          case parser.ast.structure.NumberNode(v, _) => transformNumberNode(baseAstNode.location(), v)
          case parser.ast.structure.BooleanNode(v, _) => transformBooleanNode(v)
          case kn: parser.ast.structure.KeyNode => transformKeyNode(kn, transformationStack)
          case parser.ast.structure.DynamicKeyNode(keyName, attr) => transformDynamicKeyNode(keyName, attr, transformationStack)
          case parser.ast.functions.UsingNode(vars, expr, _) => transformUsing(vars.assignmentSeq, expr, transformationStack)
          case doBlock: parser.ast.functions.DoBlockNode => transformDoBlock(doBlock, transformationStack)
          case parser.ast.functions.UsingVariableAssignment(name, expr) => transformVarDirective(name, expr, Seq(), transformationStack)
          //
          case parser.ast.selectors.NullSafeNode(selector, _) => transformNullSafeNode(selector, transformationStack)
          case parser.ast.selectors.NullUnSafeNode(selector, _) => transformNullUnSafeNode(selector, transformationStack)
          case parser.ast.selectors.ExistsSelectorNode(selectable) => transformExistsSelectorNode(selectable, transformationStack)
          //
          case vrn: parser.ast.variables.VariableReferenceNode => transformVariableReferenceNode(vrn)
          case ni: parser.ast.variables.NameIdentifier => transformNameSlot(ni)
          //
          case parser.ast.conditional.UnlessNode(ifExpr, condition, elseExpr, _) => transformUnlessNode(ifExpr, condition, elseExpr, transformationStack)
          case parser.ast.conditional.IfNode(expression, condition, elseExpr, _) => transformIfNode(expression, condition, elseExpr, transformationStack)
          case parser.ast.conditional.DefaultNode(lhs, rhs, _) => transformDefaultNode(lhs, rhs, transformationStack)
          //
          case fcn: parser.ast.functions.FunctionCallNode => transformFunctionCallNode(fcn, transformationStack)
          case bon: parser.ast.operators.BinaryOpNode => transformBinaryOpNode(bon, transformationStack)
          case uop: parser.ast.operators.UnaryOpNode => transformUnaryOpNode(uop, transformationStack)
          //
          case parser.ast.structure.schema.SchemaPropertyNode(name, value, condition) => transformSchemaPropertyNode(name, value, condition, transformationStack)
          case node @ parser.ast.structure.schema.SchemaNode(properties) => transformSchemaNode(properties, SchemaNodeOrigin(node), transformationStack)
          //
          //
          case parser.ast.logical.AndNode(lhs, rhs, _) => transformAndNode(lhs, rhs, transformationStack)
          case parser.ast.logical.OrNode(lhs, rhs, _) => transformOrNode(lhs, rhs, transformationStack)
          //
          // UPDATES!!!!
          case un: parser.ast.updates.UpdateNode => transformUpdate(un, transformationStack)
          case un: parser.ast.updates.UpdateExpressionsNode => transformUpdateExpressionsNode(un, transformationStack)
          case un: parser.ast.updates.UpdateExpressionNode => transformUpdateExpressionNode(un, transformationStack)
          case un: parser.ast.updates.UpdateSelectorNode => transformUpdateSelectorNode(un, transformationStack)
          case _ => throw new InternalErrorException(astNode.location(), s"No conversion available for ${astNode.getClass}")
        }

        val resultAstNode: ExecutionNode = result
        resultAstNode._location = Some(astNode.location())
        resultAstNode.asInstanceOf[T]
      }
      case _ => throw new InternalErrorException(astNode.location(), s"No conversion available for ${astNode.getClass}")
    }
    transformationStack.dropNode()
    executionNode
  }

  override val _modules: ArrayBuffer[VariableTable] = ArrayBuffer()

}

object EngineGrammarTransformation {

  def apply(parsingContext: ParsingContext, moduleNodeLoader: RuntimeModuleNodeCompiler): EngineGrammarTransformation = {
    new EngineGrammarTransformation(parsingContext, moduleNodeLoader)
  }
}
