package org.mule.weave.v2.interpreted.transform

import org.mule.weave.v2.interpreted.marker.RequiresMaterializationAnnotation
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.pattern._
import org.mule.weave.v2.interpreted.node.structure.RLiteralTypeNode
import org.mule.weave.v2.interpreted.node.structure.function.DynamicFunctionNode
import org.mule.weave.v2.interpreted.node.structure.function.FunctionParameterNode
import org.mule.weave.v2.parser
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

trait EnginePatternTransformations extends AstTransformation {

  def transformExpressionPatternNode(pattern: AstNode, name: NameIdentifier, function: AstNode, transformationStack: TransformationStack): ExpressionPatternNode = {
    transformExpressionPatternNode(pattern, name, function, transform(function, transformationStack), transformationStack)
  }

  def transformExpressionPatternNode(pattern: AstNode, name: NameIdentifier, function: AstNode, body: ValueNode[_], transformationStack: TransformationStack): ExpressionPatternNode = {
    val functionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(name, transformationStack), needsMaterialization(name))), body, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location

    val patternExpression: ValueNode[_] = transform(pattern, transformationStack)

    val patternFunctionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(name, transformationStack), needsMaterialization(name))), patternExpression, None, requiresMaterializedArguments = false, literalParameters = true)
    patternFunctionNode._location = pattern._location
    new ExpressionPatternNode(patternFunctionNode, functionNode)
  }

  def transformLiteralPatternNode(pattern: AstNode, name: NameIdentifier, function: AstNode, transformationStack: TransformationStack): LiteralPatternNode = {
    transformLiteralPatternNode(pattern, name, function, transform(function, transformationStack), transformationStack)
  }

  def transformLiteralPatternNode(pattern: AstNode, name: NameIdentifier, onMatch: AstNode, body: ValueNode[_], transformationStack: TransformationStack): LiteralPatternNode = {
    val onMatchFunctionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(name, transformationStack), needsMaterialization(name))), body, None, requiresMaterializedArguments = false, literalParameters = true)
    onMatchFunctionNode._location = onMatch._location
    new LiteralPatternNode(transform(pattern, transformationStack), onMatchFunctionNode)
  }

  def transformRegexPatternNode(pattern: AstNode, name: NameIdentifier, function: AstNode, transformationStack: TransformationStack): RegexPatternNode = {
    transformRegexPatternNode(pattern, name, function, transform(function, transformationStack), transformationStack)
  }

  def transformRegexPatternNode(pattern: AstNode, name: NameIdentifier, function: AstNode, body: ValueNode[_], transformationStack: TransformationStack): RegexPatternNode = {
    val functionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(name, transformationStack), needsMaterialization(name))), body, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location
    new RegexPatternNode(transform(pattern, transformationStack), functionNode)
  }

  def transformDefaultPatternNode(nameIdentifier: NameIdentifier, value: AstNode, transformationStack: TransformationStack): DefaultPatternNode = {
    transformDefaultPatternNode(nameIdentifier, value, transform(value, transformationStack), transformationStack)
  }

  def transformDefaultPatternNode(nameIdentifier: NameIdentifier, value: AstNode, body: ValueNode[_], transformationStack: TransformationStack): DefaultPatternNode = {
    val onMatchFunctionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(nameIdentifier, transformationStack), needsMaterialization(nameIdentifier))), body, None, requiresMaterializedArguments = false, literalParameters = true)
    onMatchFunctionNode._location = value._location
    new DefaultPatternNode(onMatchFunctionNode)
  }

  def transformTypePatternNode(tpn: parser.ast.patterns.TypePatternNode, transformationStack: TransformationStack): TypePatternNode = {
    transformTypePatternNode(tpn, transform(tpn.onMatch, transformationStack), transformationStack)
  }

  def transformTypePatternNode(tpn: parser.ast.patterns.TypePatternNode, body: ValueNode[_], transformationStack: TransformationStack): TypePatternNode = {
    val onMatchFunctionNode: DynamicFunctionNode = DynamicFunctionNode(Array(FunctionParameterNode(transform(tpn.name, transformationStack), needsMaterialization(tpn.name))), body, None, requiresMaterializedArguments = false, literalParameters = true)
    onMatchFunctionNode._location = tpn.onMatch._location
    val theType = new RLiteralTypeNode(transform(tpn.pattern, transformationStack))
    val typeRequiresMaterialize = tpn.pattern.isAnnotatedWith(classOf[RequiresMaterializationAnnotation])
    new TypePatternNode(theType, onMatchFunctionNode, typeRequiresMaterialize)
  }

  def transformPatternMatcherNode(lhs: AstNode, patterns: parser.ast.patterns.PatternExpressionsNode, transformationStack: TransformationStack): PatternMatcherNode = {
    new PatternMatcherNode(transform(lhs, transformationStack), transformSeq(patterns.patterns, transformationStack).toArray)
  }

  def transformEmptyArrayNode(function: AstNode, transformationStack: TransformationStack): EmptyArrayPatternNode = {
    transformEmptyArrayNode(function, transform(function, transformationStack), transformationStack)
  }

  def transformEmptyArrayNode(function: AstNode, body: ValueNode[_], transformationStack: TransformationStack): EmptyArrayPatternNode = {
    val functionNode = DynamicFunctionNode(Array(), body, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location
    new EmptyArrayPatternNode(functionNode)
  }

  def transformDeconstructArrayNode(head: NameIdentifier, tail: NameIdentifier, function: AstNode, transformationStack: TransformationStack): DeconstructArrayPatternNode = {
    transformDeconstructArrayNode(head, tail, function, transform(function, transformationStack), transformationStack)
  }

  def transformDeconstructArrayNode(head: NameIdentifier, tail: NameIdentifier, function: AstNode, body: ValueNode[_], transformationStack: TransformationStack): DeconstructArrayPatternNode = {
    val parameterNodes = Array(FunctionParameterNode(transform(head, transformationStack), needsMaterialization(head)), FunctionParameterNode(transform(tail, transformationStack), needsMaterialization(tail)))
    val functionNode: DynamicFunctionNode = DynamicFunctionNode(parameterNodes, body, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location
    new DeconstructArrayPatternNode(functionNode)
  }

  def transformEmptyObjectNode(function: AstNode, transformationStack: TransformationStack): EmptyObjectPatternNode = {
    transformEmptyObjectNode(function, transform(function, transformationStack), transformationStack)
  }

  def transformEmptyObjectNode(function: AstNode, value: ValueNode[_], transformationStack: TransformationStack): EmptyObjectPatternNode = {
    val functionNode = DynamicFunctionNode(Array(), value, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location
    new EmptyObjectPatternNode(functionNode)
  }

  def transformDeconstructObjectNode(headKey: NameIdentifier, headValue: NameIdentifier, tail: NameIdentifier, function: AstNode, transformationStack: TransformationStack): DeconstructObjectPatternNode = {
    transformDeconstructObjectNode(headKey, headValue, tail, function, transform(function, transformationStack), transformationStack)
  }

  def transformDeconstructObjectNode(headKey: NameIdentifier, headValue: NameIdentifier, tail: NameIdentifier, function: AstNode, body: ValueNode[_], transformationStack: TransformationStack): DeconstructObjectPatternNode = {
    val parameterNodes = Array(
      FunctionParameterNode(transform(headKey, transformationStack), needsMaterialization(headKey)),
      FunctionParameterNode(transform(headValue, transformationStack), needsMaterialization(headValue)),
      FunctionParameterNode(transform(tail, transformationStack), needsMaterialization(tail)))
    val functionNode = DynamicFunctionNode(parameterNodes, body, None, requiresMaterializedArguments = false, literalParameters = true)
    functionNode._location = function._location
    new DeconstructObjectPatternNode(functionNode)
  }
}
