%dw 2.8
import fail from dw::Runtime

type Executor<InputType <: Object, ResultType, ResultErrorType <: ResultFailure> = (InputType) -> Result<ResultType, ResultErrorType>

type ResultSuccess<ResultType> = {|
   success: true,
   value: ResultType
|}

type ResultFailure<ErrorValueType> = {|
   success: false,
   error: {
      description: String | Null,
      value: ErrorValueType
   }
|}

type Result<ResultSuccessType, ResultFailureType <: ResultFailure> = ResultSuccess<ResultSuccessType> | ResultFailureType

fun success<ResultType>(value: ResultType): ResultSuccess<ResultType> = {
    success: true,
    value: value
}

fun failure<ResultErrorValueType>(value: ResultErrorValueType, errorDescription: String | Null = null): ResultFailure<ResultErrorValueType> = {
    success: false,
    error: {
        description: errorDescription,
        value: value
    }
}
fun onCompletionRun<InputType, ResultType, ResultErrorType <: ResultFailure, NewResultType, NewResultErrorType <: ResultFailure>(
    executor: Executor<InputType, ResultType, ResultErrorType>,
    postProcessing: {
        onSuccess: (ResultType) -> Result<NewResultType, NewResultErrorType>,
        onFailure: (ResultErrorType) -> Result<NewResultType, NewResultErrorType>
    }
): Executor<InputType, NewResultType, NewResultErrorType> =
    (inputType: InputType): Result<NewResultType, NewResultErrorType> ->
        executor(inputType) match {
            case result is ResultSuccess<ResultType> ->
                postProcessing.onSuccess(result.value)
            case result is ResultErrorType ->
                postProcessing.onFailure(result)
   }

var sum: Executor<{ left: Number, right: Number }, Number, ResultFailure<-1>> = (parameter) ->
    if (0 <= parameter.left and 0 <= parameter.right)
      success(parameter.left + parameter.right)
    else
      failure(-1)

var failsOnNegativesAndMediumResults = sum
  onCompletionRun {
    onSuccess: (v) ->
      if (v < 30)
        success("Small result")
      else
        failure("Medium-sized result"),
    onFailure: (v) ->
      if (v.error.description == "Number too big")
        success("Large result")
      else
        failure("Negative argument")
  }
---
failsOnNegativesAndMediumResults({ left: 1, right: 1 })