package org.mule.weave.v2.interpreted.extension

import org.mule.weave.v2.core.util.ObjectValueUtils
import org.mule.weave.v2.interpreted.{ InterpretedModuleExecutableWeave, RuntimeModuleNodeCompiler }
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types.{ ArrayType, ObjectType, StringType }
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.option.{ BooleanModuleOption, IntModuleOption, ModuleOption, StringModuleOption }
import org.mule.weave.v2.module.{ DataFormat, DataFormatExtensionsLoaderService }
import org.mule.weave.v2.parser.ModuleParser
import org.mule.weave.v2.parser.annotation._
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.header.directives.VarDirective
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.{ AstNode, AstNodeHelper }
import org.mule.weave.v2.parser.exception.WeaveRuntimeException
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.module.MimeType
import org.mule.weave.v2.parser.phase.{ AbstractScopeAnnotationProcessor, DataFormatSettingsHelper, ParsingContext, ScopePhaseAnnotationContext }
import org.mule.weave.v2.runtime.WeaveCompiler
import org.mule.weave.v2.sdk.WeaveResourceResolver
import org.mule.weave.v2.ts
import org.mule.weave.v2.ts._

import java.nio.charset.Charset
import scala.annotation.tailrec
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

class WeaveBasedDataFormatExtensionLoaderService(parsingContextFactory: ParsingContextCreator, resourceResolver: WeaveResourceResolver, moduleLoader: RuntimeModuleNodeCompiler) extends DataFormatExtensionsLoaderService {

  private def toModuleOption(dataFormatSetting: DataFormatSetting[_]): ModuleOption = {
    dataFormatSetting match {
      case stringSetting: StringSetting   => StringModuleOption(stringSetting.name, defaultValue = Option(stringSetting.defaultValue).orNull, possibleValues = stringSetting.possibleValues.toSet)
      case booleanSetting: BooleanSetting => BooleanModuleOption(booleanSetting.name, defaultValue = booleanSetting.defaultValue)
      case intSettingOption: IntSetting   => IntModuleOption(intSettingOption.name, defaultValue = Option(intSettingOption.defaultValue).getOrElse(0))
      case _                              => StringModuleOption(dataFormatSetting.name)
    }
  }

  @tailrec
  private def toModuleOption(name: String, value: WeaveType, kvpDocumentation: Option[String]): ModuleOption = {
    val simpleDescription = value.getDocumentation().orElse(kvpDocumentation).getOrElse("")
    val defaultValue = value.getMetadataConstraint("defaultValue").map(_.value)
    val options = value.getMetadataConstraint("possibleValues").map(_.value.toString.split(",").map(_.trim).toSet)
    value match {
      case _: ts.StringType  => StringModuleOption(name, simpleDescription = simpleDescription, defaultValue = defaultValue.map(_.toString).orNull, possibleValues = options.getOrElse(Set()))
      case _: ts.BooleanType => BooleanModuleOption(name, simpleDescription = simpleDescription, defaultValue = defaultValue.forall(_.toString.toBoolean))
      case _: ts.NumberType  => IntModuleOption(name, simpleDescription = simpleDescription, defaultValue = defaultValue.map(_.toString.toInt).getOrElse(0))
      case rt: ReferenceType => toModuleOption(name, rt.resolveType(), kvpDocumentation)
      case _                 => StringModuleOption(name, simpleDescription = simpleDescription)
    }
  }

  private def collectReaderOptions(value: TypeNode): Map[String, ModuleOption] = {
    DataFormatSettingsHelper.collectReaderSettings(value, toModuleOption).map(opt => (opt.name, opt)).toMap
  }

  private def collectWriterOptions(value: TypeNode): Map[String, ModuleOption] = {
    DataFormatSettingsHelper.collectWriterSettings(value, toModuleOption).map(opt => (opt.name, opt)).toMap
  }

  override protected def loadModules()(implicit ctx: EvaluationContext): Seq[DataFormat[_, _]] = {
    def createWeaveBasedDataFormat(moduleDescriptorCtx: EvaluationContext, entry: (String, Value[_]), name: String, readerOptions: Map[String, ModuleOption], writerOptions: Map[String, ModuleOption]) = {
      val value = ObjectType.coerce(entry._2)(moduleDescriptorCtx)
      val mimeTypesValue: Seq[MimeType] = ObjectValueUtils
        .selectStringSeq(value.evaluate(moduleDescriptorCtx), "acceptedMimeTypes")(moduleDescriptorCtx)
        .map(_.map(MimeType.fromSimpleString))
        .getOrElse(Seq())

      val labelValue: Option[String] = ObjectValueUtils
        .selectString(value.evaluate(moduleDescriptorCtx), "label")(moduleDescriptorCtx)

      val fileExtensionsValue: Seq[String] = ObjectValueUtils
        .selectStringSeq(value.evaluate(moduleDescriptorCtx), "fileExtensions")(moduleDescriptorCtx)
        .getOrElse(Seq())

      val binaryDataFormatValue: Boolean = ObjectValueUtils
        .selectBoolean(value.evaluate(moduleDescriptorCtx), "binaryFormat")(moduleDescriptorCtx)
        .getOrElse(false)

      val defaultCharsetValue: Option[Charset] = ObjectValueUtils
        .selectString(value.evaluate(moduleDescriptorCtx), "defaultCharset")(moduleDescriptorCtx)
        .map(Charset.forName)
      new WeaveBasedDataFormatProxy(
        value,
        () => new WeaveBasedDataFormatSettings(readerOptions),
        () => new WeaveBasedDataFormatWriterSettings(writerOptions),
        name,
        mimeTypesValue,
        fileExtensionsValue,
        labelValue,
        binaryDataFormatValue,
        defaultCharsetValue)
    }

    def loadModuleFromDwl(moduleDescriptorCtx: EvaluationContext, descriptor: Value[_], moduleName: NameIdentifier) = {
      val moduleParsingContext = parsingContextFactory.createParsingContext(moduleName, true)
      val dfProcessor = new DataFormatExtensionAnnotationProcessor()
      moduleParsingContext.registerAnnotationProcessor(Extensions.dfExtension, dfProcessor)
      val moduleResource = resourceResolver
        .resolveAll(moduleName)
        .headOption
        .getOrElse(throw new WeaveRuntimeException(s"Unable to found resource ${moduleName}", descriptor.location().asInstanceOf[WeaveLocation]))

      val moduleTypeCheck = ModuleParser.parse(ModuleParser.typeCheckPhase(), moduleResource, moduleParsingContext)
      if (moduleTypeCheck.noErrors()) {
        val moduleCompilationResult = WeaveCompiler.runtimeModuleCompilation(moduleTypeCheck.getResult(), moduleParsingContext, moduleLoader)
        val executableModule = moduleCompilationResult.getResult().executable.asInstanceOf[InterpretedModuleExecutableWeave]
        val dataFormats = executableModule.collectVariables(dfProcessor.declaredDataFormats.map(_.name))(moduleDescriptorCtx)
        dataFormats.map((entry) => {
          val name = entry._1

          val varIdentifier = dfProcessor.declaredDataFormats.find(_.name == name).get
          val maybeNode: Option[TypeNode] = moduleTypeCheck.getResult().typeGraph.findNode(varIdentifier)

          val readerOptions: Map[String, ModuleOption] = maybeNode.map(collectReaderOptions).getOrElse(Map.empty)

          val writerOptions: Map[String, ModuleOption] = maybeNode.map(collectWriterOptions).getOrElse(Map.empty)

          createWeaveBasedDataFormat(moduleDescriptorCtx, entry, name, readerOptions, writerOptions)
        })
      } else {
        moduleDescriptorCtx.serviceManager.loggingService.logError(s"Unable to load module `${moduleName}` reasons: ")
        moduleTypeCheck
          .messages()
          .errorMessages
          .foreach((message) => {
            moduleDescriptorCtx.serviceManager.loggingService.logError(message._2.message + "\n" + message._1.locationString)
          })
        Seq()
      }
    }

    val extensions = resourceResolver.resolveAll(Extensions.nameIdentifier)
    extensions.flatMap((extension) => {
      val extensionDescriptorParsingContext: ParsingContext = parsingContextFactory.createParsingContext(Extensions.nameIdentifier, false)
      extensionDescriptorParsingContext.skipVerification = true
      val compilationResult = WeaveCompiler.compileWithNoCheck(extension, extensionDescriptorParsingContext)
      if (compilationResult.noErrors()) {
        val moduleDescriptorCtx = EvaluationContext()
        try {
          val declaredModulesValue = compilationResult.getResult().executable.execute()(moduleDescriptorCtx)
          val declaredModules = ArrayType.coerce(declaredModulesValue)(moduleDescriptorCtx).evaluate(moduleDescriptorCtx).toIterator()
          declaredModules
            .flatMap((descriptor) => {
              val moduleName = NameIdentifier(StringType.coerce(descriptor)(moduleDescriptorCtx).evaluate(moduleDescriptorCtx).toString)
              val maybeResourceBinary = resourceResolver.resolveBinary(moduleName)
              if (maybeResourceBinary.isDefined) {
                val moduleParsingContext = parsingContextFactory.createParsingContext(moduleName, false)
                moduleParsingContext.skipVerification = true
                val compilationResult = WeaveCompiler.compileModuleBinary(moduleName, maybeResourceBinary.get, moduleParsingContext, moduleLoader)
                if (compilationResult.noErrors()) {
                  try {
                    val astNode = compilationResult.getResult().astNode
                    val executableModule = compilationResult.getResult().executable.asInstanceOf[InterpretedModuleExecutableWeave]
                    val dataFormatVariableNodes = AstNodeHelper
                      .collectChildren(astNode, {
                        case vd: VarDirective =>
                          vd.isAnnotatedWith(classOf[DataFormatExtensionAnnotation])
                        case _ => false
                      })

                    val dataFormats = executableModule
                      .collectVariables(dataFormatVariableNodes.flatMap(varDirectiveNode => AstNodeHelper.collectChildrenWith(varDirectiveNode, classOf[NameIdentifier]).headOption).map(_.name))(moduleDescriptorCtx)
                    if (dataFormats.isEmpty) {
                      // Case of a binary file without the annotation
                      moduleDescriptorCtx.serviceManager.loggingService.logWarn(s"Unable to load module `${moduleName}` from binary model as ast is not annotated with the expected information. Start loading it from source model.")
                      loadModuleFromDwl(moduleDescriptorCtx, descriptor, moduleName)
                    } else {
                      dataFormats.map((entry) => {
                        val name = entry._1

                        val varAstNode = dataFormatVariableNodes
                          .find(varDirectiveNode => AstNodeHelper.collectChildrenWith(varDirectiveNode, classOf[NameIdentifier]).headOption.map(_.name).get == name).get
                        val dataFormatExtensionAnnotation = varAstNode.annotation(classOf[DataFormatExtensionAnnotation]).get

                        createWeaveBasedDataFormat(
                          moduleDescriptorCtx,
                          entry,
                          name,
                          dataFormatExtensionAnnotation.dataFormatSetting.readerSettings.map(setting => (setting.name, toModuleOption(setting))).toMap,
                          dataFormatExtensionAnnotation.dataFormatSetting.writerSettings.map(setting => (setting.name, toModuleOption(setting))).toMap)
                      })
                    }
                  } catch {
                    case e: Exception => {
                      moduleDescriptorCtx.serviceManager.loggingService.logWarn(s"Unable to load module `${moduleName}` from binary model, reasons: `${e.getMessage}. Start loading it from source model.")
                      loadModuleFromDwl(moduleDescriptorCtx, descriptor, moduleName)
                    }
                  }
                } else {
                  moduleDescriptorCtx.serviceManager.loggingService.logError(s"Unable to load module `${moduleName}` reasons: ")
                  compilationResult
                    .messages()
                    .errorMessages
                    .foreach((message) => {
                      moduleDescriptorCtx.serviceManager.loggingService.logError(message._2.message + "\n" + message._1.locationString)
                    })
                  Seq()
                }
              } else {
                loadModuleFromDwl(moduleDescriptorCtx, descriptor, moduleName)
              }
            })
        } finally {
          moduleDescriptorCtx.close()
        }

      } else {
        ctx.serviceManager.loggingService.logError(s"Unable to load descriptor `${extension.url()}` reasons: ")
        compilationResult
          .messages()
          .errorMessages
          .foreach((message) => {
            ctx.serviceManager.loggingService.logError(message._2.message + "\n" + message._1.locationString)
          })
        Seq()
      }
    })
  }

}
object WeaveBasedDataFormatExtensionLoaderService {
  def apply(parsingContextFactory: ParsingContextCreator, resourceResolver: WeaveResourceResolver, moduleLoader: RuntimeModuleNodeCompiler): WeaveBasedDataFormatExtensionLoaderService = {
    new WeaveBasedDataFormatExtensionLoaderService(parsingContextFactory, resourceResolver, moduleLoader)
  }
}

class DataFormatExtensionAnnotationProcessor extends AbstractScopeAnnotationProcessor {

  val declaredDataFormats: ArrayBuffer[NameIdentifier] = mutable.ArrayBuffer[NameIdentifier]()

  override def run(annotatedNode: AstNode, annotation: AnnotationNode, context: ScopePhaseAnnotationContext): Unit = {
    annotatedNode match {
      case vd: VarDirective => {
        declaredDataFormats.+=(vd.variable)
      }
      case _ => //this should never happen
    }
  }
}

object Extensions {
  val nameIdentifier = NameIdentifier("META-INF::dw-extensions")

  val dfExtension = NameIdentifier("dw::extension::DataFormat::DataFormatExtension")
}