package org.mule.weave.v2.interpreted.marker

import org.mule.weave.v2.interpreted.InterpreterPreCompilerMarker
import org.mule.weave.v2.parser.annotation.WithValueAstNodeAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNodeHelper
import org.mule.weave.v2.parser.ast.header.directives.AnnotationDirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.InputDirective
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.NoSourceLocation
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.scope.Reference
import org.mule.weave.v2.scope.ScopesNavigator

class NameSlotMarker extends InterpreterPreCompilerMarker {

  override def mark(node: AstNode, scope: ScopesNavigator, context: ParsingContext): Unit = {
    node match {
      case vr: VariableReferenceNode =>
        val maybeResolvedVariable = scope.resolveVariable(vr.variable)
        if (maybeResolvedVariable.isDefined) {
          val reference = maybeResolvedVariable.get
          annotateReferenceNode(vr.variable, reference)
        }

      case trn: TypeReferenceNode =>
        val maybeResolvedVariable = scope.resolveVariable(trn.variable)
        if (maybeResolvedVariable.isDefined) {
          val reference = maybeResolvedVariable.get
          annotateReferenceNode(trn.variable, reference)
        }

      case ns: NamespaceNode =>
        val maybeResolvedVariable = scope.resolveVariable(ns.prefix)
        if (maybeResolvedVariable.isDefined) {
          val reference = maybeResolvedVariable.get
          annotateReferenceNode(ns.prefix, reference)
        }

      case an: AnnotationNode =>
        val maybeResolvedVariable = scope.resolveVariable(an.name)
        if (maybeResolvedVariable.isDefined) {
          val reference = maybeResolvedVariable.get
          annotateReferenceNode(an.name, reference)

          // @Interceptor
          annotateInterceptor(an, scope, context)
        }

      case _ =>
      // Nothing to do
    }
  }

  private def annotateReferenceNode(referenceNode: NameIdentifier, reference: Reference): Unit = {
    if (reference.isLocalReference) {
      val navigator = reference.scope.astNavigator()
      val maybeInputDirective = navigator.parentWithType(reference.referencedNode, classOf[InputDirective])
      maybeInputDirective match {
        case Some(id) =>
          if (!id.isAnnotatedWith(classOf[ExternalBindingAnnotation])) {
            val externalBindingAnnotation = ExternalBindingAnnotation(reference.referencedNode.name)
            id.annotate(externalBindingAnnotation)
          }
        case _ =>
        // Nothing to do
      }
    }

    if (!referenceNode.isAnnotatedWith(classOf[ReferenceAnnotation])) {
      val referenceValue = toReferenceValue(reference, Some(referenceNode.location()))
      referenceNode.annotate(ReferenceAnnotation(referenceValue))
    }
  }

  private def toReferenceValue(reference: Reference, location: Option[WeaveLocation]): ReferenceValue = {
    if (reference.isLocalReference) {
      ReferenceValue(None, reference.referencedNode.name, location.map(NoSourceLocation.fromLocation))
    } else {
      val moduleSource = reference.moduleSource.get
      ReferenceValue(Some(moduleSource), reference.referencedNode.name, location.map(NoSourceLocation.fromLocation))
    }
  }

  private def annotateInterceptor(annotation: AnnotationNode, scopeNavigator: ScopesNavigator, parsingContext: ParsingContext): Unit = {
    val isAnnotated = annotation.isAnnotatedWith(classOf[InterceptorOnNativeFunctionAnnotation]) ||
      annotation.isAnnotatedWith(classOf[InterceptorOnReferenceAnnotation]) ||
      annotation.isAnnotatedWith(classOf[InterceptorInvalidReferenceAnnotation])

    if (!isAnnotated) {
      val maybeAnnotationReference = scopeNavigator.resolveVariable(annotation.name)
      if (maybeAnnotationReference.isDefined) {
        val ref = maybeAnnotationReference.get
        val annotationRefScope = ref.scope
        val annotationAstNavigator = annotationRefScope.astNavigator()
        val maybeAnnotationDirectiveNode = annotationAstNavigator.parentWithType(ref.referencedNode, classOf[AnnotationDirectiveNode])
        if (maybeAnnotationDirectiveNode.isDefined) {
          val annotationDirective = maybeAnnotationDirectiveNode.get
          // We detected that the annotation is an InterceptorAnnotation
          val maybeInterceptorAnnotation = annotationDirective.codeAnnotations.find(ann => {
            val resolvedAnnotationRef = annotationRefScope.resolveVariable(ann.name)
            resolvedAnnotationRef.exists(ref => ref.fqnReferenceName == NameIdentifier.INTERCEPTOR_ANNOTATION)
          })
          if (maybeInterceptorAnnotation.isDefined) {
            val interceptorAnnotation = maybeInterceptorAnnotation.get
            val maybeInterceptorFunctionRef = AnnotationNodeHelper.arg("interceptorFunction", interceptorAnnotation)
            if (maybeInterceptorFunctionRef.isDefined) {
              val interceptorFunctionRef = maybeInterceptorFunctionRef.get
              interceptorFunctionRef match {
                case functionName: LiteralValueAstNode =>
                  // Annotate native function
                  val nativeInterceptorName = NameIdentifier(functionName.literalValue.substring("@native".length).trim)
                  val interceptorNativeFunctionValue = InterceptorNativeFunctionValue(nativeInterceptorName, functionName._location.map(NoSourceLocation.fromLocation))
                  annotation.annotate(InterceptorOnNativeFunctionAnnotation(interceptorNativeFunctionValue))
                case vrn: VariableReferenceNode =>
                  val maybeReference = annotationRefScope.resolveVariable(vrn.variable)
                  maybeReference match {
                    case Some(functionReference) =>
                      // Annotate reference
                      val isLocalReference = functionReference.fqnReferenceName.parent().contains(parsingContext.nameIdentifier)
                      val reference = if (isLocalReference) {
                        functionReference
                      } else {
                        Reference(functionReference.referencedNode, functionReference.scope, functionReference.fqnReferenceName.parent())
                      }

                      val referenceValue = toReferenceValue(reference, vrn._location)
                      annotation.annotate(InterceptorOnReferenceAnnotation(referenceValue))
                    case None =>
                      annotation.annotate(InterceptorInvalidReferenceAnnotation(InvalidReferenceValue(vrn.variable, NoSourceLocation.fromLocation(interceptorAnnotation.location()))))
                  }
                case _ =>
                // Nothing to do!
              }
            }
          }
        }
      }
    }
  }
}

case class ReferenceAnnotation(referenceValue: ReferenceValue) extends WithValueAstNodeAnnotation[ReferenceValue] {

  override def name(): String = {
    "Reference"
  }

  override def value(): ReferenceValue = {
    referenceValue
  }
}

case class ReferenceValue(moduleFQN: Option[NameIdentifier], localVariableName: String, referenceLocation: Option[NoSourceLocation])

case class ExternalBindingAnnotation(localVariableName: String) extends WithValueAstNodeAnnotation[String] {

  override def name(): String = {
    "ExternalBinding"
  }

  override def value(): String = {
    localVariableName
  }
}

case class InterceptorOnNativeFunctionAnnotation(nativeFunctionValue: InterceptorNativeFunctionValue) extends WithValueAstNodeAnnotation[InterceptorNativeFunctionValue] {

  override def name(): String = {
    "InterceptorOnNativeFunction"
  }

  override def value(): InterceptorNativeFunctionValue = {
    nativeFunctionValue
  }
}

case class InterceptorNativeFunctionValue(functionName: NameIdentifier, location: Option[NoSourceLocation])

case class InterceptorOnReferenceAnnotation(referenceValue: ReferenceValue) extends WithValueAstNodeAnnotation[ReferenceValue] {

  override def name(): String = {
    "InterceptorOnReference"
  }

  override def value(): ReferenceValue = {
    referenceValue
  }
}

case class InterceptorInvalidReferenceAnnotation(invalidReference: InvalidReferenceValue) extends WithValueAstNodeAnnotation[InvalidReferenceValue] {

  override def name(): String = {
    "InterceptorInvalidReference"
  }

  override def value(): InvalidReferenceValue = {
    invalidReference
  }
}

case class InvalidReferenceValue(reference: NameIdentifier, location: NoSourceLocation)

