package org.mule.weave.v2.compilation.mapper

import org.mule.weave.v2.compilation.ArraySerializableAstNode
import org.mule.weave.v2.compilation.BooleanSerializableValueAstNode
import org.mule.weave.v2.compilation.IntSerializableValueAstNode
import org.mule.weave.v2.compilation.NoneSerializableValueAstNode
import org.mule.weave.v2.compilation.SerializableAstNode
import org.mule.weave.v2.compilation.StringSerializableValueAstNode
import org.mule.weave.v2.parser.SafeStringBasedParserInput
import org.mule.weave.v2.parser.annotation.AstNodeAnnotation
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.{ ParserPosition, Position }
import org.mule.weave.v2.sdk.WeaveResource

import java.io.Closeable
import java.util

trait DWSerializer[T] {
  def serialize(value: T): SerializableAstNode
  def deserialize(node: SerializableAstNode, context: SerializerContext): T

  def deserializeMaybe(node: SerializableAstNode, context: SerializerContext): Option[T] = {
    node match {
      case NoneSerializableValueAstNode() => None
      case _                              => Some(deserialize(node, context))
    }
  }

  def deserializeSeq(node: SerializableAstNode, context: SerializerContext): Seq[T] = {
    node.asInstanceOf[ArraySerializableAstNode].children.map(deserialize(_, context)).toList
  }

  def deserializeMaybeSeq(node: SerializableAstNode, context: SerializerContext): Option[Seq[T]] = {
    node match {
      case NoneSerializableValueAstNode() => None
      case _                              => Some(deserializeSeq(node, context))
    }
  }

  def deserializeString(node: SerializableAstNode): String = {
    node.asInstanceOf[StringSerializableValueAstNode].value
  }

  def deserializeBoolean(node: SerializableAstNode): Boolean = {
    node.asInstanceOf[BooleanSerializableValueAstNode].value
  }

  def deserializeInt(node: SerializableAstNode): Int = {
    node.asInstanceOf[IntSerializableValueAstNode].value
  }

  def deserializeMaybeString(node: SerializableAstNode): Option[String] = {
    node match {
      case NoneSerializableValueAstNode()        => None
      case StringSerializableValueAstNode(value) => Some(value)
      case _                                     => None //TODO: throw?
    }
  }

  def serializeMaybe(maybeValue: Option[T]): SerializableAstNode = {
    maybeValue match {
      case Some(value) => serialize(value)
      case None        => NoneSerializableValueAstNode()
    }
  }

  def serializeMaybeSeq(maybeValue: Option[Seq[T]]): SerializableAstNode = {
    maybeValue match {
      case Some(value) => serializeSeq(value)
      case None        => NoneSerializableValueAstNode()
    }
  }

  def serializeSeq(seq: Seq[T]): ArraySerializableAstNode = {
    ArraySerializableAstNode(seq.map(serialize))
  }

  def serializeString(str: String): StringSerializableValueAstNode = StringSerializableValueAstNode(str)

  def serializeBoolean(b: Boolean): BooleanSerializableValueAstNode = BooleanSerializableValueAstNode(b)

  def serializeInt(n: Int): IntSerializableValueAstNode = IntSerializableValueAstNode(n)

  def serializeMaybeString(maybeStr: Option[String]): SerializableAstNode = {
    maybeStr match {
      case Some(str) => serializeString(str)
      case None      => NoneSerializableValueAstNode()
    }
  }
}

trait SerializerContext extends Closeable {
  def identifier(): NameIdentifier
  def position(index: Int): Position

  private var names = new util.HashMap[String, String]()
  private var nameIdentifiers = new util.HashMap[(Option[String], String), NameIdentifier]()
  private var astNodeAnnotations = new util.HashMap[String, AstNodeAnnotation]()

  def canonicalName(name: String): String = {
    names.computeIfAbsent(name, _ => name)
  }

  def canonicalNameIdentifier(nameIdentifier: NameIdentifier): NameIdentifier = {
    nameIdentifiers.computeIfAbsent((nameIdentifier.loader, nameIdentifier.name), _ => nameIdentifier)
  }

  def canonicalAstAnnotation(identifier: String, astNodeAnnotationProvider: String => AstNodeAnnotation): AstNodeAnnotation = {
    astNodeAnnotations.computeIfAbsent(identifier, (t: String) => astNodeAnnotationProvider.apply(t))
  }

  override def close(): Unit = {
    names = null
    nameIdentifiers = null
    astNodeAnnotations = null
  }
}

case class ResourceBasedSerializerContext(resource: WeaveResource, identifier: NameIdentifier) extends SerializerContext {
  private val parserInput = SafeStringBasedParserInput(resource.content())
  private var positions = new util.HashMap[Int, Position]()

  override def position(idx: Int): Position = {
    positions.computeIfAbsent(idx, _ => ParserPosition(idx, parserInput))
  }

  override def close(): Unit = {
    super.close()
    positions = null
  }
}

case class LazyPositionResolverSerializerContext(positionFactory: LazyPositionFactory, identifier: NameIdentifier) extends SerializerContext {
  private var positions = new util.HashMap[Int, Position]()

  override def position(idx: Int): Position = {
    positions.computeIfAbsent(idx, _ => positionFactory.createPosition(idx))
  }

  override def close(): Unit = {
    super.close()
    positions = null
  }
}

class LazyPositionFactory(resource: WeaveResource) {
  private lazy val parserInput = SafeStringBasedParserInput(resource.content())

  def createPosition(idx: Int): Position = new Position {

    private lazy val parserPosition = ParserPosition(index, parserInput)

    override def index: Int = idx

    override def line: Int = parserPosition.line

    override def column: Int = parserPosition.column

    override def source: () => String = parserPosition.source
  }

}

