package org.mule.weave.v2.runtime.core.operator.selectors

import java.time.Duration
import java.time.temporal._

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.core.exception.InvalidSelectionException
import org.mule.weave.v2.core.exception.KeyNotFoundException
import org.mule.weave.v2.core.exception.StrictSelectionOverNullSelection
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure._
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values.AttributeDelegateValue
import org.mule.weave.v2.model.values.math.Number
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.parser.ast.structure.TimeZoneNode
import org.mule.weave.v2.parser.location.WeaveLocation

import scala.util.Try

class ObjectNameValueSelectorOperator(override val location: WeaveLocation, staticSelection: Boolean = false) extends BinaryFunctionValue {
  override val L = ObjectType

  override val R = NameType

  @volatile
  var index: Int = -1

  override def doExecute(leftValue: Value[ObjectSeq], rightValue: Value[QualifiedName])(implicit ctx: EvaluationContext): Value[_] = {
    val leftObject: ObjectSeq = leftValue.evaluate
    if (leftObject.onlySimpleKeys()) {
      // With simple keys (no attrs or ns) we can avoid wrapping with an attribute delegate
      if (staticSelection) {
        selectStaticValue(leftObject, rightValue)
      } else {
        selectValue(leftObject, rightValue)
      }
    } else {
      val selectedValue: KeyValuePair =
        if (staticSelection) {
          selectStaticKeyValuePair(leftObject, rightValue)
        } else {
          selectKeyValuePair(leftObject, rightValue)
        }

      if (selectedValue.isSimpleKey()) {
        selectedValue._2
      } else {
        AttributeDelegateValue(selectedValue)
      }
    }
  }

  private def selectValue(leftObject: ObjectSeq, rightValue: Value[QualifiedName])(implicit ctx: EvaluationContext): Value[_] = {
    val value = leftObject.selectValue(rightValue)
    if (value == null) {
      throw InvalidSelectionException(new KeyNotFoundException(this, rightValue.evaluate))
    } else {
      value
    }
  }

  private def selectKeyValuePair(leftObject: ObjectSeq, rightValue: Value[QualifiedName])(implicit ctx: EvaluationContext): KeyValuePair = {
    val maybePair = leftObject.selectKeyValue(rightValue)
    if (maybePair != null) {
      maybePair
    } else {
      throw InvalidSelectionException(new KeyNotFoundException(this, rightValue.evaluate))
    }
  }

  def selectIndexedKeyValuePair(indexedObjectSeq: IndexedObjectSeq, rightValue: Value[QualifiedName])(implicit ctx: EvaluationContext): KeyValuePair = {
    var result: KeyValuePair = null

    if (index != -1) {
      val maybePair: KeyValuePair = indexedObjectSeq.apply(index)
      if (maybePair != null && rightValue.evaluate.matches(maybePair._1.evaluate))
        result = maybePair
    }

    if (result == null) {
      val tuple = indexedObjectSeq.keyValueOfWithIndex(rightValue)
      if (tuple == null) {
        throw InvalidSelectionException(new KeyNotFoundException(this, rightValue.evaluate))
      } else {
        index = tuple.index
        result = tuple.selection
      }
    }
    result
  }

  def selectIndexedValue(indexedObjectSeq: IndexedObjectSeq, rightValue: Value[QualifiedName])(implicit ctx: EvaluationContext): Value[_] = {
    var result: Value[_] = null

    if (index != -1) {
      val maybePair: KeyValuePair = indexedObjectSeq.apply(index)
      if (maybePair != null && rightValue.evaluate.matches(maybePair._1.evaluate))
        result = maybePair._2
    }

    if (result == null) {
      val tuple = indexedObjectSeq.selectValueWithIndex(rightValue)
      if (tuple == null) {
        throw InvalidSelectionException(new KeyNotFoundException(this, rightValue.evaluate))
      } else {
        index = tuple.index
        result = tuple.selection
      }
    }
    result
  }

  private def selectStaticValue(leftObject: ObjectSeq, rightValue: Value[QualifiedName])(implicit ctx: EvaluationContext): Value[_] = {
    leftObject match {
      case indexedObjectSeq: IndexedObjectSeq if (!ctx.serviceManager.settingsService.execution().selectAlwaysFirst) => {
        selectIndexedValue(indexedObjectSeq, rightValue)
      }
      case _ => {
        selectValue(leftObject, rightValue)
      }
    }
  }

  private def selectStaticKeyValuePair(leftObject: ObjectSeq, rightValue: Value[QualifiedName])(implicit ctx: EvaluationContext): KeyValuePair = {
    leftObject match {
      case indexedObjectSeq: IndexedObjectSeq if (!ctx.serviceManager.settingsService.execution().selectAlwaysFirst) => {
        selectIndexedKeyValuePair(indexedObjectSeq, rightValue)
      }
      case _ => {
        selectKeyValuePair(leftObject, rightValue)
      }
    }
  }
}

class NullNameValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = NullType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: Value[QualifiedName])(implicit ctx: EvaluationContext) = {
    throw InvalidSelectionException(new StrictSelectionOverNullSelection(this))
  }
}

object ObjectNameValueSelectorOperator {
  def apply(location: WeaveLocation, staticSelection: Boolean = false): ObjectNameValueSelectorOperator =
    new ObjectNameValueSelectorOperator(location, staticSelection)
}

class ObjectStringValueSelectorOperator(override val location: WeaveLocation, staticSelection: Boolean) extends BinaryFunctionValue {
  val delegate = ObjectNameValueSelectorOperator(location, staticSelection)

  override val L = ObjectType

  override val R = StringType

  override def doExecute(leftValue: Value[ObjectSeq], rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    delegate.doExecute(leftValue, NameType.coerce(rightValue))
  }
}

class ArrayNameValueSelectorOperator(override val location: WeaveLocation, staticSelection: Boolean) extends BinaryFunctionValue {

  val objectSelector = ObjectNameValueSelectorOperator(location)

  override val L = ArrayType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val objects = leftValue.evaluate
      .toIterator()
      .filter((item) => ObjectType.accepts(item))
    val result = objects.flatMap((v: Value[_]) => {
      Try(objectSelector.doExecute(v.asInstanceOf[Value[ObjectSeq]], rightValue)).toOption
    })
    if (result.nonEmpty) {
      ArrayValue(result, this)
    } else {
      throw InvalidSelectionException(new KeyNotFoundException(this, rightValue.evaluate))
    }
  }
}

object ArrayNameValueSelectorOperator {
  def apply(location: WeaveLocation, staticSelection: Boolean): ArrayNameValueSelectorOperator = new ArrayNameValueSelectorOperator(location, staticSelection)
}

class ArrayStringValueSelectorOperator(override val location: WeaveLocation, staticSelection: Boolean) extends BinaryFunctionValue {

  val delegate = ArrayNameValueSelectorOperator(location, staticSelection)

  override val L = ArrayType

  override val R = StringType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    delegate.doExecute(leftValue, NameType.coerce(rightValue))
  }
}

class LocalDateValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = LocalDateType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val keyName = rightValue.evaluate
    try {
      NumberValue(Number(leftValue.evaluate.get(DateFieldHelper.temporalField(keyName))))
    } catch {
      case _: UnsupportedTemporalTypeException => throw InvalidSelectionException(new KeyNotFoundException(UnknownLocationCapable, keyName))
    }
  }
}

class DateTimeValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = DateTimeType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val qName: QualifiedName = rightValue.evaluate
    try {
      if (qName.name == TimeZoneNode.TIMEZONE) {
        TimeZoneValue(leftValue.evaluate.getOffset, UnknownLocationCapable)
      } else {
        NumberValue(Number(leftValue.evaluate.get(DateFieldHelper.temporalField(qName))))
      }
    } catch {
      case _: UnsupportedTemporalTypeException => throw InvalidSelectionException(new KeyNotFoundException(UnknownLocationCapable, qName))
    }
  }
}

class LocalDateTimeValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = LocalDateTimeType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val keyName = rightValue.evaluate
    try {
      NumberValue(Number(leftValue.evaluate.get(DateFieldHelper.temporalField(keyName))))
    } catch {
      case _: UnsupportedTemporalTypeException => throw InvalidSelectionException(new KeyNotFoundException(UnknownLocationCapable, keyName))
    }
  }
}

class TimeValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = TimeType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val qName: QualifiedName = rightValue.evaluate
    try {
      if (qName.name == TimeZoneNode.TIMEZONE) {
        TimeZoneValue(leftValue.evaluate.getOffset, UnknownLocationCapable)
      } else {
        NumberValue(Number(leftValue.evaluate.get(DateFieldHelper.temporalField(qName))))
      }
    } catch {
      case _: UnsupportedTemporalTypeException => throw InvalidSelectionException(new KeyNotFoundException(UnknownLocationCapable, qName))
    }
  }
}

class LocalTimeValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = LocalTimeType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val keyName = rightValue.evaluate
    try {
      NumberValue(Number(leftValue.evaluate.get(DateFieldHelper.temporalField(keyName))))
    } catch {
      case _: UnsupportedTemporalTypeException => throw InvalidSelectionException(new KeyNotFoundException(UnknownLocationCapable, keyName))
    }
  }
}

class PeriodValueSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {

  override val L = PeriodType

  override val R = NameType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val keyName = rightValue.evaluate
    val period: PeriodType.T = leftValue.evaluate
    try {
      val result = period match {
        case duration: Duration => {
          val seconds = duration.getSeconds
          keyName.name match {
            case "days"    => ((seconds / DateFieldHelper.SECONDS_PER_HOUR) / 24).toInt
            case "hours"   => ((seconds / DateFieldHelper.SECONDS_PER_HOUR) % 24).toInt
            case "minutes" => ((seconds % DateFieldHelper.SECONDS_PER_HOUR) / DateFieldHelper.SECONDS_PER_MINUTE).toInt
            case "secs"    => (seconds % DateFieldHelper.SECONDS_PER_MINUTE).toInt
            case _         => period.get(DateFieldHelper.temporalUnit(keyName))
          }
        }
        case _ => period.get(DateFieldHelper.temporalUnit(keyName))
      }
      NumberValue(Number(result))
    } catch {
      case _: UnsupportedTemporalTypeException => throw InvalidSelectionException(new KeyNotFoundException(UnknownLocationCapable, keyName))
    }
  }
}

object DateFieldHelper {

  /**
    * Hours per day.
    */
  val HOURS_PER_DAY = 24

  /**
    * Minutes per hour.
    */
  val MINUTES_PER_HOUR = 60

  /**
    * Minutes per day.
    */
  val MINUTES_PER_DAY = MINUTES_PER_HOUR * HOURS_PER_DAY

  /**
    * Seconds per minute.
    */
  val SECONDS_PER_MINUTE = 60

  /**
    * Seconds per hour.
    */
  val SECONDS_PER_HOUR = SECONDS_PER_MINUTE * MINUTES_PER_HOUR

  /**
    * Seconds per day.
    */
  val SECONDS_PER_DAY = SECONDS_PER_HOUR * HOURS_PER_DAY

  /**
    * Milliseconds per day.
    */
  val MILLIS_PER_DAY = SECONDS_PER_DAY * 1000L

  /**
    * Microseconds per day.
    */
  val MICROS_PER_DAY = SECONDS_PER_DAY * 1000000L

  /**
    * Nanos per second.
    */
  val NANOS_PER_SECOND = 1000000000L

  /**
    * Nanos per minute.
    */
  val NANOS_PER_MINUTE = NANOS_PER_SECOND * SECONDS_PER_MINUTE

  /**
    * Nanos per hour.
    */
  val NANOS_PER_HOUR = NANOS_PER_MINUTE * MINUTES_PER_HOUR

  /**
    * Nanos per day.
    */
  val NANOS_PER_DAY = NANOS_PER_HOUR * HOURS_PER_DAY

  def temporalField(name: QualifiedName): TemporalField = {
    name.name match {
      case "nanoseconds"   => ChronoField.NANO_OF_SECOND
      case "milliseconds"  => ChronoField.MILLI_OF_SECOND
      case "seconds"       => ChronoField.SECOND_OF_MINUTE
      case "minutes"       => ChronoField.MINUTE_OF_HOUR
      case "hour"          => ChronoField.HOUR_OF_DAY
      case "day"           => ChronoField.DAY_OF_MONTH
      case "month"         => ChronoField.MONTH_OF_YEAR
      case "year"          => ChronoField.YEAR
      case "quarter"       => IsoFields.QUARTER_OF_YEAR
      case "dayOfWeek"     => ChronoField.DAY_OF_WEEK
      case "dayOfYear"     => ChronoField.DAY_OF_YEAR
      case "offsetSeconds" => ChronoField.OFFSET_SECONDS
      case _               => throw InvalidSelectionException(new KeyNotFoundException(UnknownLocationCapable, name))
    }
  }

  def temporalUnit(name: QualifiedName): TemporalUnit = {
    name.name match {
      case "nanos"        => ChronoUnit.NANOS
      case "micros"       => ChronoUnit.MICROS
      case "milliseconds" => ChronoUnit.MILLIS
      case "seconds"      => ChronoUnit.SECONDS
      case "minutes"      => ChronoUnit.MINUTES
      case "hours"        => ChronoUnit.HOURS
      case "halfDays"     => ChronoUnit.HALF_DAYS
      case "months"       => ChronoUnit.MONTHS
      case "days"         => ChronoUnit.DAYS
      case "weeks"        => ChronoUnit.WEEKS
      case "years"        => ChronoUnit.YEARS
      case "decades"      => ChronoUnit.DECADES
      case "centuries"    => ChronoUnit.CENTURIES
      case "millennia"    => ChronoUnit.MILLENNIA
      case "eras"         => ChronoUnit.ERAS
      case "forever"      => ChronoUnit.FOREVER
      case _              => throw InvalidSelectionException(new KeyNotFoundException(UnknownLocationCapable, name))
    }
  }
}
