package org.mule.weave.v2.interpreted.transform

import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.interpreted.node.ExecutionNode
import org.mule.weave.v2.parser
import org.mule.weave.v2.parser.annotation.MaterializeVariableAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.exception.WeaveRuntimeException
import org.mule.weave.v2.parser.phase.ParsingContext

trait AstTransformation {
  def transformOption[T <: ExecutionNode](source: Option[parser.ast.AstNode], transformationStack: TransformationStack): Option[T] = {
    val result: Option[T] = source.map(node => transform[T](node, transformationStack))
    result
  }

  def transformSeq[T <: ExecutionNode](source: Seq[parser.ast.AstNode], transformationStack: TransformationStack): Seq[T] = {
    val result: Seq[T] = source.map(node => transform[T](node, transformationStack))
    result
  }

  def transformOptionSeq[T <: ExecutionNode](source: Option[Seq[parser.ast.AstNode]], transformationStack: TransformationStack): Option[Seq[T]] = {
    val result: Option[Seq[T]] = source.map(_.map(node => transform[T](node, transformationStack)))
    result
  }

  def needsMaterialization(variable: NameIdentifier): Boolean = {
    variable.annotation(classOf[MaterializeVariableAnnotation]).forall(_.needMaterialize)
  }

  def transform[T <: ExecutionNode](astNode: parser.ast.AstNode, transformationStack: TransformationStack): T

  def parsingContext(): ParsingContext

  def moduleLoader(): RuntimeModuleNodeCompiler

  var rootElement: AstNode

  def transformingModule: Boolean = rootElement.isInstanceOf[ModuleNode]

}

/**
  * Stack to keep track of the current node being transformed.Implemented uses an array to keep track of the current node being transformed.
  * @param maxDepth to limit the size of the stack to avoid memory overflow.
  * @since 2.11
  */
class TransformationStack(maxDepth: Int = 512) {
  private var stack: Array[AstNode] = new Array[AstNode](Math.min(maxDepth, 16))
  private var index: Int = -1

  /**
    * Returns the list of nodes in the stack from the root node to the current node.
    * @return list of nodes in the stack.
    */
  def nodes(): Array[AstNode] = {
    stack.slice(0, index + 1)
  }

  def pushNode(astNode: AstNode): Unit = {
    index = index + 1
    if (index >= stack.length) {
      // Grow array when needed
      val newSize = Math.min(stack.length * 2, maxDepth)
      if (newSize <= index) {
        throw new WeaveRuntimeException(s"Engine Grammar Transformation stack overflow at depth ${index}", astNode.location())
      }
      val newStack = new Array[AstNode](newSize)
      System.arraycopy(stack, 0, newStack, 0, stack.length)
      stack = newStack
    }
    stack(index) = astNode
  }

  def dropNode(): Unit = {
    if (index >= 0) {
      stack(index) = null // Help GC
      index = index - 1
    }
  }

}

