package org.mule.weave.v2.compilation.mapper.nonAst

import org.mule.weave.v2.compilation.SerializableAstNodeKind.DATA_FORMAT_SETTINGS_VALUE_NODE
import org.mule.weave.v2.compilation._
import org.mule.weave.v2.compilation.exception.{ AstNodeAnnotationDeSerializationException, AstNodeAnnotationSerializationException }
import org.mule.weave.v2.compilation.mapper.{ AstToSerializableMapper, DWSerializer, SerializerContext }
import org.mule.weave.v2.parser.annotation._

object DataFormatSettingsValueSerializer extends DWSerializer[DataFormatSettings] {

  override def serialize(dataFormatSettings: DataFormatSettings): SerializableAstNode = {
    val children = Seq(
      ArraySerializableAstNode(dataFormatSettings.readerSettings.map(setting => DataFormatSettingValueSerializer.serialize(setting))),
      ArraySerializableAstNode(dataFormatSettings.writerSettings.map(setting => DataFormatSettingValueSerializer.serialize(setting))))
    toNodeSerializableAstNode(DATA_FORMAT_SETTINGS_VALUE_NODE, children)
  }

  override def deserialize(node: SerializableAstNode, context: SerializerContext): DataFormatSettings = {
    val readerSettingsNode = node.children().head.asInstanceOf[ArraySerializableAstNode]
    val readerSettings = readerSettingsNode.children.map(ron => DataFormatSettingValueSerializer.deserialize(ron, context)).toList

    val writerSettingsNode = node.children()(1).asInstanceOf[ArraySerializableAstNode]
    val writerSettings = writerSettingsNode.children.map(won => DataFormatSettingValueSerializer.deserialize(won, context)).toList
    DataFormatSettings(readerSettings, writerSettings)
  }

  private def toNodeSerializableAstNode(kind: Short, children: Seq[SerializableAstNode] = Seq()): NodeSerializableAstNode = {
    NodeSerializableAstNode(kind, UnknownLocation, children)
  }

}

private object DataFormatSettingValueSerializer extends DWSerializer[DataFormatSetting[_]] {

  override def serialize(dataFormatSetting: DataFormatSetting[_]): SerializableAstNode = {
    dataFormatSetting match {
      case stringSetting: StringSetting =>
        val children = Seq(
          serializeString(stringSetting.name),
          serializeMaybeString(Option(stringSetting.defaultValue)),
          ArraySerializableAstNode(stringSetting.possibleValues.map(pv => serializeString(pv))))
        toNodeSerializableAstNode(kind = SerializableAstNodeKind.DATA_FORMAT_STRING_SETTING_VALUE_NODE, children = children)
      case booleanSetting: BooleanSetting =>
        val children = Seq(
          serializeString(booleanSetting.name),
          serializeBoolean(booleanSetting.defaultValue))
        toNodeSerializableAstNode(kind = SerializableAstNodeKind.DATA_FORMAT_BOOLEAN_SETTING_VALUE_NODE, children = children)
      case intSetting: IntSetting =>
        val children = Seq(
          serializeString(intSetting.name),
          serializeInt(intSetting.defaultValue))
        toNodeSerializableAstNode(kind = SerializableAstNodeKind.DATA_FORMAT_INT_SETTING_VALUE_NODE, children = children)
      case _ => throw AstNodeAnnotationSerializationException(s"Can't serialize values for annotation: ${DataFormatExtensionAnnotation.getClass.getName} as ${dataFormatSetting.getClass.getName} is not supported.")
    }
  }

  override def deserialize(node: SerializableAstNode, context: SerializerContext): DataFormatSetting[_] = {
    node.kind() match {
      case SerializableAstNodeKind.DATA_FORMAT_STRING_SETTING_VALUE_NODE =>
        val name = AstToSerializableMapper.deserializeString(node.children().head)
        val defaultValue = AstToSerializableMapper.deserializeMaybeString(node.children()(1)).orNull
        val possibleValuesNode = node.children()(2).asInstanceOf[ArraySerializableAstNode]
        val possibleValues = possibleValuesNode.children.map(deserializeString)
        StringSetting(name, defaultValue, possibleValues)
      case SerializableAstNodeKind.DATA_FORMAT_BOOLEAN_SETTING_VALUE_NODE =>
        val name = AstToSerializableMapper.deserializeString(node.children().head)
        val defaultValue = AstToSerializableMapper.deserializeBoolean(node.children()(1))
        BooleanSetting(name, defaultValue)
      case SerializableAstNodeKind.DATA_FORMAT_INT_SETTING_VALUE_NODE =>
        val name = AstToSerializableMapper.deserializeString(node.children().head)
        val defaultValue = AstToSerializableMapper.deserializeInt(node.children()(1))
        IntSetting(name, defaultValue)
      case _ =>
        // Case if a new Setting type has been added and with an old version of DW Runtime
        // we need to deserialize it at least we list the setting as String.
        val name = AstToSerializableMapper.deserializeString(node.children().head)
        StringSetting(name, null)
    }
  }

  private def toNodeSerializableAstNode(kind: Short, children: Seq[SerializableAstNode] = Seq()): NodeSerializableAstNode = {
    NodeSerializableAstNode(kind, UnknownLocation, children)
  }

}