package org.mule.weave.v2.interpreted.module.reader

import org.mule.weave.v2.interpreted.module.reader.exception.WeaveReaderException
import org.mule.weave.v2.module.reader.ReaderInput
import org.mule.weave.v2.module.reader.ReaderLocation
import org.mule.weave.v2.module.reader.SourceReader
import org.mule.weave.v2.module.reader.StreamingReaderLocation

import java.lang.{ StringBuilder => JStringBuilder }
import scala.annotation.switch
import scala.annotation.tailrec

trait WeaveLiteralTokenizer {

  def input: SourceReader

  val name: String

  protected var cursorCharLocation: Long = 0

  protected var cursorChar: Char = _

  protected val strBuffer: Array[Char] = new Array(WeaveTokenizer.charbuffersize)

  protected def location(): ReaderLocation = {
    if (input.supportsSeek) {
      ReaderLocation(cursorCharLocation, input)
    } else {
      new StreamingReaderLocation(cursorCharLocation, input)
    }
  }

  def parseString(): String = {
    var additional: JStringBuilder = null
    var i = 0
    var keepReading = true
    while (keepReading) {
      while (i < WeaveTokenizer.charbuffersize && keepReading) {
        if (((1L << cursorChar) & ((31 - cursorChar) >> 31) & 0x7ffffffbefffffffL) != 0L) {
          strBuffer.update(i, cursorChar)
        } else {
          cursorChar match {
            case '"' | ReaderInput.EOI => {
              keepReading = false
            }
            case '\\' => {
              advance()
              (cursorChar: @switch) match {
                case '"' | '/' | '\\' => strBuffer.update(i, cursorChar)
                case 'n'              => strBuffer.update(i, '\n')
                case 'b'              => strBuffer.update(i, '\b')
                case 'f'              => strBuffer.update(i, '\f')
                case 'r'              => strBuffer.update(i, '\r')
                case 't'              => strBuffer.update(i, '\t')
                case '$'              => strBuffer.update(i, '$');
                case 'u' => {
                  val hexChars = new Array[Char](4)
                  advance()
                  hexChars.update(0, cursorChar)
                  advance()
                  hexChars.update(1, cursorChar)
                  advance()
                  hexChars.update(2, cursorChar)
                  advance()
                  hexChars.update(3, cursorChar)
                  val codePoint = Integer.parseInt(new String(hexChars), 16)
                  if (Character.isBmpCodePoint(codePoint)) {
                    strBuffer.update(i, codePoint.toChar)
                  } else if (Character.isValidCodePoint(codePoint)) {
                    if (i + 1 >= WeaveTokenizer.charbuffersize) {
                      if (additional == null) {
                        additional = new JStringBuilder(WeaveTokenizer.charbuffersize * 2)
                      }
                      additional.append(new String(strBuffer))
                      i = 0
                    }
                    strBuffer.update(i + 1, Character.lowSurrogate(codePoint))
                    strBuffer.update(i, Character.highSurrogate(codePoint))
                    i = i + 1
                  } else {
                    throw new IllegalArgumentException
                  }
                }
              }
            }
            case ReaderInput.EOI => {
              fail("End of input reached but expected '\"'")
            }
            case c => {
              strBuffer.update(i, c)
              keepReading = c >= ' '
            }
          }

        }
        //Advance only if we need to keep reading
        if (keepReading) {
          advanceChar()
          i = i + 1
        }
      }

      if (keepReading) {
        //Is a long text use the string builder to store the pars
        if (additional == null) {
          additional = new JStringBuilder(WeaveTokenizer.charbuffersize * 2)
        }
        additional.append(new String(strBuffer))
        i = 0
      }
    }

    val result = if (additional != null) {
      additional.append(new String(strBuffer, 0, i))
      additional.toString
    } else {
      new String(strBuffer, 0, i)
    }
    result
  }

  protected def readChar(): Boolean = {
    if (((1L << cursorChar) & ((31 - cursorChar) >> 31) & 0x7ffffffbefffffffL) != 0L) {
      true
    } else {
      cursorChar match {
        case '"' | ReaderInput.EOI => false
        case '\\' => {
          advance()
          readEscapedChar()
          true
        }

        case c => {
          c >= ' '
        }
      }
    }
  }

  protected def readEscapedChar(target: Option[StringBuilder] = None): Unit = {
    (cursorChar: @switch) match {
      case '"' | '/' | '\\' | 'b' | 'f' | 'n' | 'r' | 't' | '$' =>
      case 'u' =>
        advance(); advance(); advance(); advance()
      case _ => fail("Invalid escape sequence")
    }
  }

  def eof(): Unit = {
    if (cursorChar != ReaderInput.EOI) {
      fail(s"Expecting end of input but got `${cursorChar}")
    }
  }

  protected def requireFalse(): Unit = {
    advance()
    require('a')
    require('l')
    require('s')
    require('e')
    ws()
  }

  protected def requireNull(): Unit = {
    advance()
    require('u')
    require('l')
    require('l')
    ws()
  }

  protected def requireTrue(): Unit = {
    advance()
    require('r')
    require('u')
    require('e')
    ws()
  }

  def advance(): Boolean = {
    cursorCharLocation = input.position()
    cursorChar = input.readAscii()
    true
  }

  protected def advanceChar(): Boolean = {
    cursorCharLocation = input.position()
    cursorChar = input.read()
    true
  }

  def ws(c: Char): Boolean = {
    if (advanceIf(c)) {
      ws()
      true
    } else false
  }

  @tailrec
  final def ws(): Unit = {
    if (((1L << cursorChar) & ((cursorChar - 64) >> 31) & 0x100002600L) != 0L) {
      advance()
      ws()
    }
  }

  protected def advanceIf(c: Char): Boolean = {
    if (cursorChar == c) {
      advance()
      true
    } else {
      false
    }
  }

  protected def advanceIfNot(c: Char): Boolean = {
    if (cursorChar != c) {
      advance()
      true
    } else {
      false
    }
  }

  protected def require(c: Char, possibleUnicode: Boolean = false): Unit = {
    val requiredChar = if (possibleUnicode) {
      advanceCharIf(c)
    } else {
      advanceIf(c)
    }
    if (!requiredChar) {
      fail(s"Expecting '$c' but was '${cursorChar}'")
    }
  }

  protected def advanceCharIf(c: Char): Boolean = {
    if (cursorChar == c) {
      advanceChar()
      true
    } else {
      false
    }
  }

  def fail(target: String, location: ReaderLocation = location, errorChar: Char = cursorChar) = {
    throw new WeaveReaderException(target, location)
  }

}

object WeaveTokenizer {
  var charbuffersize = 1024
}