package org.mule.weave.v2.interpreted.node

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.structure.FrameBasedSchemaResolver
import org.mule.weave.v2.interpreted.node.structure.SchemaNodeBasedSchemaResolver
import org.mule.weave.v2.interpreted.node.structure.schema.SchemaNode
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.types.ReferenceType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.types.TypeType
import org.mule.weave.v2.model.values.ReferenceTypeValue
import org.mule.weave.v2.model.values.SchemaProvider
import org.mule.weave.v2.model.values.TypeValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.runtime.exception.InvalidTypeNameException
import org.mule.weave.v2.utils.ExplicitSchemaNode

class TypeReferenceNode(val variable: NameSlot, val schemaConstraint: Option[SchemaNode], val metadata: Option[SchemaNode]) extends ValueNode[Type] with Product3[NameSlot, Option[SchemaNode], Option[SchemaNode]] {
  override def doExecute(implicit ctx: ExecutionContext): Value[Type] = {
    val activeFrame = ctx.executionStack().activeFrame()
    val typeResolver = referenceResolver(activeFrame)
    val typeSchemaResolver = new TypeReferenceNodeSchemaResolver(variable, activeFrame, metadata, location())
    val schemaProvider = SchemaProvider(typeSchemaResolver)
    new ReferenceTypeValue(ReferenceType(variable.name, typeResolver), this, schemaProvider)
  }

  override def _1: NameSlot = variable

  override def _2: Option[SchemaNode] = schemaConstraint

  override def _3: Option[SchemaNode] = metadata

  private def referenceResolver(frame: Frame): EvaluationContext => Type = (ectx: EvaluationContext) => {
    implicit val context: ExecutionContext = ectx match {
      case executionContext: ExecutionContext =>
        executionContext
      case _ =>
        ExecutionContext(frame, ectx)
    }
    context.runInFrame(
      frame, {
      val v = if (variable.module.isDefined) {
        val moduleName: NameSlot = variable.module.get
        context.executionStack().getVariable(moduleName.slot, variable.slot)
      } else {
        context.executionStack().getVariable(variable.slot)
      }
      v match {
        case typeValue: Value[Type] if TypeType.accepts(typeValue) => typeValue.evaluate.withSchema(schemaConstraint.map(_.doExecute.evaluate(ectx)))
        case _ => throw new InvalidTypeNameException(location(), variable.name)
      }
    })
  }

  override def shouldNotify: Boolean = false
}

class TypeReferenceNodeSchemaResolver(variable: NameSlot, val frame: Frame, metadata: Option[SchemaNode], location: WeaveLocation) extends FrameBasedSchemaResolver {

  override def resolveSchema(executionContext: ExecutionContext): Option[Schema] = {
    metadata match {
      case Some(SchemaNode(_, ExplicitSchemaNode)) =>
        metadata.map(_.execute(executionContext).evaluate(executionContext))
      case _ =>
        val v = if (variable.module.isDefined) {
          val moduleName: NameSlot = variable.module.get
          executionContext.executionStack().getVariable(moduleName.slot, variable.slot)
        } else {
          executionContext.executionStack().getVariable(variable.slot)
        }
        v match {
          case typeValue: Value[Type] if TypeType.accepts(typeValue)(executionContext) =>
            mergeSchemas(metadata.map(_.execute(executionContext).evaluate(executionContext)), typeValue.schema(executionContext))
          case _ =>
            throw new InvalidTypeNameException(location, variable.name)
        }
    }
  }

  private def mergeSchemas(mayBeAnnotationSchemas: Option[Schema], mayBeSpecifiedSchemas: Option[Schema]): Option[Schema] = {
    (mayBeAnnotationSchemas, mayBeSpecifiedSchemas) match {
      case (Some(schema), None) => Some(schema)
      case (None, Some(schema)) => Some(schema)
      case (Some(annotationSchemas), Some(specifiedSchemas)) => Some(Schema(annotationSchemas, specifiedSchemas))
      case _ => None
    }
  }
}

class TypeReferenceWithTypeParamNode(val name: String, val theTypeRef: ValueNode[_], val schemaConstraint: Option[SchemaNode], val metadata: Option[SchemaNode]) extends ValueNode[Type] with Product3[ValueNode[_], Option[SchemaNode], Option[SchemaNode]] {
  override def doExecute(implicit ctx: ExecutionContext): Value[Type] = {
    val activeFrame = ctx.executionStack().activeFrame()
    val typeResolver = referenceResolver(activeFrame)
    val schemaProvider = metadata.map(m => {
      val typeSchemaResolver = new SchemaNodeBasedSchemaResolver(activeFrame, m)
      SchemaProvider(typeSchemaResolver)
    })
    TypeValue(ReferenceType(name, typeResolver), this, schemaProvider)
  }

  private def referenceResolver(frame: Frame): EvaluationContext => Type = (ectx: EvaluationContext) => {
    implicit val context: ExecutionContext = ectx match {
      case executionContext: ExecutionContext =>
        executionContext
      case _ =>
        ExecutionContext(frame, ectx)
    }
    context.runInFrame(
      frame, {
      val evaluate = theTypeRef.execute(context)
      evaluate match {
        case typeValue: Value[Type] if TypeType.accepts(typeValue) => typeValue.evaluate.withSchema(schemaConstraint.map(_.doExecute.evaluate))
        case _ => throw new InvalidTypeNameException(location(), name)
      }
    })
  }

  override def _1: ValueNode[_] = theTypeRef

  override def _2: Option[SchemaNode] = schemaConstraint

  override def _3: Option[SchemaNode] = metadata

  override def shouldNotify: Boolean = false
}