package org.mule.weave.v2.interpreted.node.structure

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.ExecutionNode
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.interpreted.node.structure.TypeNode.schemaProvider
import org.mule.weave.v2.interpreted.node.structure.schema.SchemaNode
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.FunctionParamType
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.types.IntersectionType
import org.mule.weave.v2.model.types.KeyType
import org.mule.weave.v2.model.types.KeyValuePairType
import org.mule.weave.v2.model.types.NameType
import org.mule.weave.v2.model.types.NameValuePairType
import org.mule.weave.v2.model.types.NamespaceType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.types.UnionType
import org.mule.weave.v2.model.values.SchemaProvider
import org.mule.weave.v2.model.values.TypeSchemaResolver
import org.mule.weave.v2.model.values.TypeValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.location.WeaveLocation

class RLiteralTypeNode(delegate: ValueNode[Type]) extends ValueNode[Type] with Product1[ValueNode[_]] {

  var theValue: Value[Type] = _

  override protected def doExecute(implicit ctx: ExecutionContext): Value[Type] = {
    if (theValue == null) {
      this.synchronized {
        if (theValue == null) {
          theValue = delegate.execute
        }
      }
    }
    theValue
  }

  override def _1: ValueNode[_] = delegate

  override def location(): WeaveLocation = delegate.location()

  override def shouldNotify: Boolean = false
}

class TypeNode(val typeValue: Type, val asSchema: Option[SchemaNode] = None, val asTypeSchema: Option[SchemaNode] = None) extends ValueNode[Type] with Product3[Type, Option[SchemaNode], Option[SchemaNode]] {
  override def doExecute(implicit ctx: ExecutionContext): Value[Type] = {
    val frame = ctx.executionStack().activeFrame()
    TypeValue(typeValue.withSchema(asSchema.map(_.execute.evaluate)), this, schemaProvider(asTypeSchema, frame))
  }

  override def _1: Type = typeValue

  override def _2: Option[SchemaNode] = asSchema

  override def _3: Option[SchemaNode] = asTypeSchema

  override def shouldNotify: Boolean = false
}

class ArrayTypeNode(val arrayItem: ValueNode[Type], val asSchema: Option[SchemaNode] = None, val asTypeSchema: Option[SchemaNode] = None) extends ValueNode[Type] with Product3[ValueNode[Type], Option[SchemaNode], Option[SchemaNode]] {
  override def doExecute(implicit ctx: ExecutionContext): Value[Type] = {
    val frame = ctx.executionStack().activeFrame()
    val maybeSchema: Option[Schema] = asSchema.map(_.execute.evaluate)
    val arrayType: Type = new ArrayType(arrayItem.execute.evaluate)
    TypeValue(arrayType.withSchema(maybeSchema), this, schemaProvider(asTypeSchema, frame))
  }

  override def _1: ValueNode[Type] = arrayItem

  override def _2: Option[SchemaNode] = asSchema

  override def _3: Option[SchemaNode] = asTypeSchema

  override def shouldNotify: Boolean = false
}

class RFunctionTypeNode(val params: Seq[RFunctionParamTypeNode], returnTypeNode: ValueNode[Type], val asSchema: Option[SchemaNode] = None, val asTypeSchema: Option[SchemaNode] = None) extends ValueNode[Type] with Product2[Seq[RFunctionParamTypeNode], ValueNode[Type]] {
  override protected def doExecute(implicit ctx: ExecutionContext): TypeValue = {
    val frame = ctx.executionStack().activeFrame()
    val paramsType = params.map((p) => FunctionParamType(p.paramType.execute.evaluate, p.optional, p.name))
    val returnType = returnTypeNode.execute.evaluate
    TypeValue(
      new FunctionType(Some(paramsType.toArray), Some(returnType)).withSchema(asSchema.map(_.execute.evaluate)),
      this,
      schemaProvider(asTypeSchema, frame))
  }

  override def _1: Seq[RFunctionParamTypeNode] = params

  override def _2: ValueNode[Type] = returnTypeNode

  override def shouldNotify: Boolean = false
}

class RFunctionParamTypeNode(val paramType: ValueNode[Type], val name: Option[String], val optional: Boolean) extends Product3[ValueNode[Type], Option[String], Boolean] with ExecutionNode {
  override def _1: ValueNode[Type] = paramType

  override def _2: Option[String] = name

  override def _3: Boolean = optional

}

class RObjectTypeNode(val kvpTypeValues: Seq[ValueNode[Type]], open: Boolean, val asSchema: Option[SchemaNode] = None, val asTypeSchema: Option[SchemaNode] = None) extends ValueNode[Type] with Product1[Seq[ValueNode[Type]]] {
  override protected def doExecute(implicit ctx: ExecutionContext): TypeValue = {
    val types = kvpTypeValues.map(_.execute.evaluate.asInstanceOf[KeyValuePairType])
    val frame = ctx.executionStack().activeFrame()
    TypeValue(new ObjectType(types, open).withSchema(asSchema.map(_.execute.evaluate)), this, schemaProvider(asTypeSchema, frame))
  }

  override def _1: Seq[ValueNode[Type]] = kvpTypeValues

  override def shouldNotify: Boolean = false
}

class RKeyValuePairTypeNode(val keyTypeNode: ValueNode[Type], val valueTypeNode: ValueNode[Type], val optional: Boolean, val repeated: Boolean) extends ValueNode[Type] with Product3[ValueNode[Type], ValueNode[Type], Boolean] {
  override protected def doExecute(implicit ctx: ExecutionContext) = {
    val key = keyTypeNode.execute.evaluate
    val value = valueTypeNode.execute.evaluate
    TypeValue(new KeyValuePairType(key, value, optional, repeated))
  }

  override def _1: ValueNode[Type] = keyTypeNode

  override def _2: ValueNode[Type] = valueTypeNode

  override def _3: Boolean = optional

  override def shouldNotify: Boolean = false
}

class RNameValuePairTypeNode(val nameTypeNode: ValueNode[NameType], val valueTypeNode: ValueNode[Type], val optional: Boolean) extends ValueNode[Type] with Product3[ValueNode[Type], ValueNode[Type], Boolean] {
  override protected def doExecute(implicit ctx: ExecutionContext) = {
    val key = nameTypeNode.execute.evaluate
    val value = valueTypeNode.execute.evaluate
    TypeValue(new NameValuePairType(key, value, optional))
  }

  override def _1: ValueNode[Type] = nameTypeNode

  override def _2: ValueNode[Type] = valueTypeNode

  override def _3: Boolean = optional

  override def shouldNotify: Boolean = false
}

class RKeyTypeNode(val nameTypeNode: ValueNode[Type], val attrs: Seq[ValueNode[NameValuePairType]], val metadata: Option[SchemaNode] = None) extends ValueNode[Type] with Product1[ValueNode[Type]] {
  override protected def doExecute(implicit ctx: ExecutionContext) = {
    val nameType = nameTypeNode.execute.evaluate.asInstanceOf[NameType]
    val attrsType: Seq[NameValuePairType] = attrs.map(_.execute.evaluate)
    TypeValue(new KeyType(Some(nameType), attrsType), this, schemaProvider(metadata))
  }

  override def _1: ValueNode[Type] = nameTypeNode

  override def shouldNotify: Boolean = false
}

class RNameTypeNode(localName: String, ns: Option[NamespaceNode], asTypeSchema: Option[SchemaNode]) extends ValueNode[Type] with Product3[String, Option[NamespaceNode], Option[SchemaNode]] {
  override protected def doExecute(implicit ctx: ExecutionContext): Value[Type] = {
    TypeValue(new NameType(Some(localName), ns.map((nsNode) => new NamespaceType(Some(nsNode.execute.evaluate.uri)))), UnknownLocationCapable, schemaProvider(asTypeSchema))
  }

  override def _1: String = localName

  override def _2: Option[NamespaceNode] = ns

  override def _3: Option[SchemaNode] = asTypeSchema

  override def shouldNotify: Boolean = false
}

class TypeParameterNode(val typeValue: ValueNode[Type]) extends ValueNode[Type] with Product1[ValueNode[Type]] {
  override def doExecute(implicit ctx: ExecutionContext): Value[Type] = {
    TypeValue(typeValue.execute.evaluate, this)
  }

  override def _1: ValueNode[Type] = typeValue

  override def shouldNotify: Boolean = false
}

class UnionTypeNode(val typeValue: Seq[ValueNode[Type]], val asSchema: Option[SchemaNode] = None, val asTypeSchema: Option[SchemaNode] = None) extends ValueNode[Type] with Product1[Seq[ValueNode[Type]]] {
  override def doExecute(implicit ctx: ExecutionContext): Value[Type] = {
    val elements = typeValue.map(_.execute.evaluate)
    val frame = ctx.executionStack().activeFrame()
    TypeValue(UnionType(elements).withSchema(asSchema.map(_.execute.evaluate)), this, schemaProvider(asTypeSchema, frame))
  }

  override def _1: Seq[ValueNode[Type]] = typeValue

  override def shouldNotify: Boolean = false
}

class IntersectionTypeNode(val typeValue: Seq[ValueNode[Type]], val asSchema: Option[SchemaNode] = None, val asTypeSchema: Option[SchemaNode] = None) extends ValueNode[Type] with Product1[Seq[ValueNode[Type]]] {
  override def doExecute(implicit ctx: ExecutionContext): Value[Type] = {
    val frame = ctx.executionStack().activeFrame()
    val elements = typeValue.map(_.execute.evaluate)
    TypeValue(IntersectionType(elements).withSchema(asSchema.map(_.execute.evaluate)), this, schemaProvider(asTypeSchema, frame))
  }

  override def _1: Seq[ValueNode[Type]] = typeValue

  override def shouldNotify: Boolean = false
}

object TypeNode {

  def schemaProvider(maybeNode: Option[SchemaNode], frame: Frame): Option[SchemaProvider] = {
    maybeNode.map(node => {
      val typeSchemaResolver = new SchemaNodeBasedSchemaResolver(frame, node)
      SchemaProvider(typeSchemaResolver)
    })
  }

  def schemaProvider(maybeNode: Option[SchemaNode])(implicit ctx: ExecutionContext): Option[SchemaProvider] = {
    val activeFrame = ctx.executionStack().activeFrame()
    schemaProvider(maybeNode, activeFrame)
  }
}

trait FrameBasedSchemaResolver extends TypeSchemaResolver {

  def frame: Frame

  override def resolve(ctx: EvaluationContext): Option[Schema] = {
    ctx match {
      case executionContext: ExecutionContext =>
        executionContext.runInFrame(frame, {
          resolveSchema(executionContext)
        })
      case _ =>
        val executionContext = ExecutionContext(frame, ctx)
        resolveSchema(executionContext)
    }
  }

  def resolveSchema(executionContext: ExecutionContext): Option[Schema]
}

class SchemaNodeBasedSchemaResolver(val frame: Frame, metadata: SchemaNode) extends FrameBasedSchemaResolver {

  override def resolveSchema(executionContext: ExecutionContext): Option[Schema] = {
    val schema = metadata.execute(executionContext).evaluate(executionContext)
    Some(schema)
  }
}