package org.mule.weave.v2.interpreted.node.executors

import org.mule.weave.v2.core.functions.BaseTernaryFunctionValue
import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.core.exception.UnexpectedFunctionCallTypesException
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.allTargets
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.findMatchingFunctionWithCoercion
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.indexOfFunction
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.materializeOverloadedFunctionArgs
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.location.WeaveLocation

import java.util.concurrent.atomic.AtomicInteger
import java.util.concurrent.atomic.AtomicReference

/**
  * This class has the logic to execute Ternary functions. It knows how to dispatch how to do coercions and such
  *
  * @param node                  The function to call
  * @param name                  The name of the function
  * @param firstArgConstantType  If the first type is constant. This mean we can validate at compile time that the type of this param will never change
  * @param secondArgConstantType If the second type is constant. This mean we can validate at compile time that the type of this param will never change
  * @param thirdArgConstantType  If the third type is constant. This mean we can validate at compile time that the type of this param will never change
  */
class TernaryOverloadedStaticExecutor(
  val node: ValueNode[_],
  val name: String,
  val firstArgConstantType: Boolean,
  val secondArgConstantType: Boolean,
  val thirdArgConstantType: Boolean,
  val showInStacktrace: Boolean = true,
  override val location: WeaveLocation) extends TernaryExecutor with Product5[ValueNode[_], String, Boolean, Boolean, Boolean] {

  val cachedDispatchIndex: AtomicInteger = new AtomicInteger(-1)
  val cachedCoercedOperation: AtomicReference[(Int, Type, Type, Type, Boolean, Boolean, Boolean)] = new AtomicReference()

  override def execute(arguments: Array[Value[_]])(implicit ctx: ExecutionContext): Value[Any] = {
    //This should allways be like this as they are operators
    executeTernary(arguments(0), arguments(1), arguments(2))
  }

  def executeTernary(fv: Value[_], sv: Value[_], tv: Value[_])(implicit ctx: ExecutionContext): Value[Any] = {
    val activeFrame: Frame = ctx.executionStack().activeFrame()
    try {
      activeFrame.updateCallSite(node)

      val targetFunction: FunctionValue = target()
      val resolvedTargets: Array[_ <: FunctionValue] = allTargets(targetFunction)
      //FAST PATH
      // If the operation is cached then try with this ternary operation
      // Most times calling over same types so we should dispatch over same function
      val dispatchIndex = cachedDispatchIndex.get()
      if (dispatchIndex != -1) {
        val operation: BaseTernaryFunctionValue = resolvedTargets(dispatchIndex).asInstanceOf[BaseTernaryFunctionValue]

        //If values are literal we do not need to validate every just the first time when we load the operation
        //Then is ok every time as its type will NEVER change. Also we don't need to materialize
        //As we are not going to validate against the type
        val firstValue = if (!firstArgConstantType && operation.firstParam.typeRequiresMaterialization) {
          fv.materialize
        } else {
          fv
        }

        val secondValue = if (!secondArgConstantType && operation.secondParam.typeRequiresMaterialization) {
          sv.materialize
        } else {
          sv
        }

        val thirdValue = if (!thirdArgConstantType && operation.thirdParam.typeRequiresMaterialization) {
          tv.materialize
        } else {
          tv
        }

        if ((firstArgConstantType || operation.First.accepts(firstValue))
          && (secondArgConstantType || operation.Second.accepts(secondValue))
          && (thirdArgConstantType || operation.Third.accepts(thirdValue))) {
          return doCall(operation, firstValue, secondValue, thirdValue)
        }
      }

      val firstValue: Value[_] = if (targetFunction.paramsTypesRequiresMaterialize) {
        materializeOverloadedFunctionArgs(resolvedTargets, 0, fv)
      } else {
        fv
      }
      val secondValue: Value[_] = if (targetFunction.paramsTypesRequiresMaterialize) {
        materializeOverloadedFunctionArgs(resolvedTargets, 1, sv)
      } else {
        sv
      }
      val thirdValue: Value[_] = if (targetFunction.paramsTypesRequiresMaterialize) {
        materializeOverloadedFunctionArgs(resolvedTargets, 2, tv)
      } else {
        tv
      }

      val coercedOperation = cachedCoercedOperation.get
      if (coercedOperation != null) {
        //If values are literal we do not need to validate every just the first time when we load the coerced operation
        //Then is ok every time as its type will NEVER change
        if ((firstArgConstantType || coercedOperation._2.accepts(firstValue)) &&
          (secondArgConstantType || coercedOperation._3.accepts(secondValue)) &&
          (thirdArgConstantType || coercedOperation._4.accepts(thirdValue))) {

          val functionToDispatch = resolvedTargets(coercedOperation._1).asInstanceOf[BaseTernaryFunctionValue]

          val maybeFirstValue = if (!coercedOperation._5) {
            Some(firstValue)
          } else {
            functionToDispatch.First.coerceMaybe(firstValue)
          }
          val maybeSecondValue = if (!coercedOperation._6) {
            Some(secondValue)
          } else {
            functionToDispatch.Second.coerceMaybe(secondValue)
          }
          val maybeThirdValue = if (!coercedOperation._7) {
            Some(thirdValue)
          } else {
            functionToDispatch.Third.coerceMaybe(thirdValue)
          }

          if (maybeFirstValue.isDefined && maybeSecondValue.isDefined && maybeThirdValue.isDefined) {
            return doCall(functionToDispatch, maybeFirstValue.get, maybeSecondValue.get, maybeThirdValue.get)
          }
        }
      }

      val matchingOp = findMatchingOp(resolvedTargets, firstValue, secondValue, thirdValue)
      if (matchingOp != -1) {
        //Update it here should we use any strategy
        if (targetFunction.dispatchCanBeCached) {
          cachedDispatchIndex.set(matchingOp)
        }
        doCall(resolvedTargets(matchingOp).asInstanceOf[BaseTernaryFunctionValue], firstValue, secondValue, thirdValue)
      } else {
        val materializedValues: Array[Value[Any]] = Array(firstValue.materialize, secondValue.materialize, thirdValue.materialize)
        val argTypes: Array[Type] = materializedValues.map(_.valueType)
        val sortedOperators: Array[FunctionValue] = FunctionDispatchingHelper.sortByParameterTypeWeight(resolvedTargets, argTypes)
        val functionToCallWithCoercion: Option[(Int, Array[Value[_]], Seq[Int])] = findMatchingFunctionWithCoercion(materializedValues, sortedOperators, this)
        functionToCallWithCoercion match {
          case Some((dispatchFunction, argumentsWithCoercion, paramsToCoerce)) => {
            //Cache the coercion use the base type to avoid Memory Leaks as Types may have references to Streams or Objects
            val ternaryFunction: BaseTernaryFunctionValue = sortedOperators(dispatchFunction).asInstanceOf[BaseTernaryFunctionValue]
            if (targetFunction.dispatchCanBeCached) {
              cachedCoercedOperation.set((indexOfFunction(resolvedTargets, ternaryFunction), firstValue.valueType.baseType, secondValue.valueType.baseType, thirdValue.valueType.baseType, paramsToCoerce.contains(0), paramsToCoerce.contains(1), paramsToCoerce.contains(2)))
            }
            val firstCoercedValue: Value[_] = argumentsWithCoercion(0)
            val secondCoercedValue = argumentsWithCoercion(1)
            val thirdCoercedValue = argumentsWithCoercion(2)
            doCall(ternaryFunction, firstCoercedValue, secondCoercedValue, thirdCoercedValue)
          }
          case None =>
            throw new UnexpectedFunctionCallTypesException(node.location(), name, materializedValues, sortedOperators.map(_.parameters.map(_.wtype)))
        }
      }
    } finally {
      activeFrame.cleanCallSite()
    }
  }

  private def doCall(operation: BaseTernaryFunctionValue, firstValue: Value[_], secondValue: Value[_], thirdValue: Value[_])(implicit ctx: ExecutionContext) = {
    try {
      operation.call(firstValue, secondValue, thirdValue)
    } catch {
      case ex: ExecutionException =>
        if (showInStacktrace) {
          ex.addCallToStacktrace(location, name())
        }
        throw ex
    }
  }

  def findMatchingOp(operations: Array[_ <: FunctionValue], firstValue: Value[_], secondValue: Value[_], thirdValue: Value[_])(implicit ctx: ExecutionContext): Int = {
    var i = 0
    while (i < operations.length) {
      val operation: BaseTernaryFunctionValue = operations(i).asInstanceOf[BaseTernaryFunctionValue]
      if (operation.First.accepts(firstValue) && operation.Second.accepts(secondValue) && operation.Third.accepts(thirdValue)) {
        return i
      }
      i = i + 1
    }
    -1
  }

  def target()(implicit ctx: ExecutionContext): FunctionValue = {
    node.execute.asInstanceOf[FunctionValue]
  }

  override def name()(implicit ctx: ExecutionContext): String = this.name

  override def _1: ValueNode[_] = node

  override def _2: String = name

  override def _3: Boolean = firstArgConstantType

  override def _4: Boolean = secondArgConstantType

  override def _5: Boolean = thirdArgConstantType
}