package org.mule.weave.v2.runtime.core.functions.runtime

import org.mule.weave.v2.core.functions.SecureQuinaryFunctionValue
import org.mule.weave.v2.core.functions.WriteFunctionValue
import org.mule.weave.v2.core.functions.WriteFunctionValue.toDwString
import org.mule.weave.v2.core.util.ObjectValueUtils.selectFunction
import org.mule.weave.v2.core.util.ObjectValueUtils.selectInputStream
import org.mule.weave.v2.core.util.ObjectValueUtils.selectNumber
import org.mule.weave.v2.core.util.ObjectValueUtils.selectObject
import org.mule.weave.v2.core.util.ObjectValueUtils.selectString
import org.mule.weave.v2.core.util.ObjectValueUtils.selectStringAnyMap
import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.core.exception.InvalidParameterException
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.model
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.UserDefinedServicesProvider
import org.mule.weave.v2.model.WeaveServicesProvider
import org.mule.weave.v2.model.service.CompositeRuntimeSettings
import org.mule.weave.v2.model.service.CompositeSecurityMangerService
import org.mule.weave.v2.model.service.CpuLimitService
import org.mule.weave.v2.model.service.CpuLimitedPatternService
import org.mule.weave.v2.model.service.DefaultCpuLimitService
import org.mule.weave.v2.model.service.DefaultPatternService
import org.mule.weave.v2.model.service.InMemoryLoggingService
import org.mule.weave.v2.model.service.LoggingService
import org.mule.weave.v2.model.service.PatternService
import org.mule.weave.v2.model.service.PropertiesSettings
import org.mule.weave.v2.model.service.RuntimeSettings
import org.mule.weave.v2.model.service.SecurityManagerService
import org.mule.weave.v2.model.service.SettingsService
import org.mule.weave.v2.model.service.SimpleSettingsService
import org.mule.weave.v2.model.service.WeaveRuntimePrivilege
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types
import org.mule.weave.v2.model.types.BooleanType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.math
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.ObjectValueBuilder
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.ValueProvider
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.exception.LocatableException
import org.mule.weave.v2.parser.exception.WeaveRuntimeException
import org.mule.weave.v2.parser.location.UnknownLocation
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.CompilationException
import org.mule.weave.v2.runtime.BindingValue
import org.mule.weave.v2.runtime.DataWeaveScript
import org.mule.weave.v2.runtime.DataWeaveScriptingEngine
import org.mule.weave.v2.runtime.DynamicModuleComponentFactory
import org.mule.weave.v2.runtime.InputType
import org.mule.weave.v2.runtime.ScriptingBindings
import org.mule.weave.v2.runtime.core.functions.runtime.ExceptionHandling.HANDLE
import org.mule.weave.v2.runtime.core.functions.runtime.ExceptionHandling.isHandle
import org.mule.weave.v2.runtime.utils.ResultTypeBuilder
import org.mule.weave.v2.sdk.EmptyWeaveResourceResolver
import org.mule.weave.v2.sdk.NameIdentifierHelper
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceResolver

import java.io.File
import java.io.InputStream
import java.net.URL
import java.nio.charset.Charset
import java.util.Properties

abstract class BaseRunFunctionValue extends SecureQuinaryFunctionValue {

  protected val localResourceResolver: ThreadLocal[WeaveResourceResolver] = new ThreadLocal[WeaveResourceResolver]()

  protected var engine: DataWeaveScriptingEngine = _

  override val First: StringType = StringType

  override val Second: ObjectType = ObjectType

  override val Third: ObjectType = ObjectType

  override val Forth: ObjectType = ObjectType

  override val Fifth: ObjectType = ObjectType

  override val firstParamName: String = "fileToExecute"

  override val secondParamName: String = "fs"

  override val thirdParamName: String = "readerInputs"

  override val forthParamName: String = "inputValues"

  override val fifthParamName: String = "configuration"

  override val thirdDefaultValue: Option[ValueProvider] = Some(ValueProvider(ObjectValue.empty))

  override val forthDefaultValue: Option[ValueProvider] = Some(ValueProvider(ObjectValue.empty))

  override val fifthDefaultValue: Option[ValueProvider] = Some(ValueProvider(ObjectValue.empty))

  override val requiredPrivilege: WeaveRuntimePrivilege = WeaveRuntimePrivilege.EXECUTION

  protected def initEngine(ctx: EvaluationContext): Unit = {
    if (engine == null) {
      this.synchronized {
        if (engine == null) {
          engine = DataWeaveScriptingEngine(
            DynamicModuleComponentFactory(
              ctx.serviceManager.weaveResourceResolver,
              () => {
                val resolver = localResourceResolver.get()
                if (resolver == null) {
                  EmptyWeaveResourceResolver
                } else {
                  resolver
                }
              },
              systemFirst = true))
        }
      }
    }
  }

  private def buildErrorObject(logger: InMemoryLoggingService, e: CompilationException): ObjectValue = {
    val failureResult = new ObjectValueBuilder()
    failureResult.addPair("kind", TryFunctionValue.getExceptionKind(e))
    failureResult.addPair("message", value = e.getMessage)
    failureResult.addPair("location", value = LocationBuilder.buildLocation(e.messages.errorMessages.head._1))
    failureResult.addPair("logs", buildLogs(logger))
    failureResult.build
  }

  private def buildErrorObject(logger: InMemoryLoggingService, e: LocatableException): ObjectValue = {
    val failureResult = new ObjectValueBuilder()
    failureResult.addPair("kind", TryFunctionValue.getExceptionKind(e))
    failureResult.addPair("message", value = e.getMessage)
    failureResult.addPair("location", value = LocationBuilder.buildLocation(e.location))
    e match {
      case ee: ExecutionException => {
        failureResult.addPair("stack", value = ArrayValue(ee.weaveStacktrace.buildStackTrace().map(_.stringValue())))
      }
      case _ =>
    }

    failureResult.addPair("logs", buildLogs(logger))
    failureResult.build
  }

  private def buildErrorObject(logger: InMemoryLoggingService, e: Throwable): ObjectValue = {
    val failureResult = new ObjectValueBuilder()
    failureResult.addPair("kind", TryFunctionValue.getExceptionKind(e))
    failureResult.addPair("message", value = e.getClass.getCanonicalName + " " + Option(e.getMessage).getOrElse(""))
    failureResult.addPair("location", value = LocationBuilder.buildLocation(UnknownLocation))
    failureResult.addPair("logs", buildLogs(logger))
    failureResult.build
  }

  def buildLogs(logger: InMemoryLoggingService): Value[_] = {
    ArrayValue(logger.logs.map((le) => {
      le.toDwValue
    }))
  }

  protected def loadBindings(inputsValue: Value[RunScriptFunctionValue.Forth.T], inputs: Seq[KeyValuePair])(implicit ctx: EvaluationContext) = {
    val bindings = new ScriptingBindings()
    inputs.foreach(x => {
      val inputValue: Value[model.types.ObjectType.T] = ObjectType.coerce(x._2)
      val input: types.ObjectType.T = inputValue.materialize.evaluate
      val inputName: String = x._1.evaluate.name
      val contentType: String = selectString(input, "mimeType").getOrElse(throw InvalidParameterException("inputs", s"Missing `mimeType` property on : ${toDwString(inputValue, WriteFunctionValue.maxValueLength)}.", inputsValue.location()))
      val content: InputStream = selectInputStream(input, "value").getOrElse(throw InvalidParameterException("inputs", s"Missing `value` property on : ${toDwString(inputValue, WriteFunctionValue.maxValueLength)}.", inputsValue.location()))
      val encoding: Charset = Charset.forName(selectString(input, "encoding").getOrElse("UTF-8"))
      val options: Map[String, Any] = selectStringAnyMap(input, "properties").getOrElse(Map())
      bindings.addBinding(inputName, BindingValue(content, contentType, options, charset = encoding))
    })

    inputsValue.evaluate
      .toIterator()
      .foreach((kvp) => {
        bindings.addBinding(kvp._1.evaluate.name, kvp._2.materialize)
      })
    bindings
  }

  def doExecute[T <: Value[_]](mainFileValue: First.V, fileSystemValue: Second.V, readerValues: Third.V, inputsValue: Forth.V, runtimeConfigurationValue: Fifth.V, callback: (DataWeaveScript, ScriptingBindings, ServiceManager, InMemoryLoggingService) => T)(implicit ctx: EvaluationContext): Value[_] = {
    initEngine(ctx)
    val mainFile: String = mainFileValue.evaluate.toString
    val resources: ObjectSeq = fileSystemValue.materialize.evaluate
    val inputs: Seq[KeyValuePair] = readerValues.materialize.evaluate.toSeq()
    val runtimeConfiguration: RunScriptFunctionValue.Fifth.T = runtimeConfigurationValue.materialize.evaluate
    val timeout: math.Number = selectNumber(runtimeConfiguration, "timeOut").getOrElse(-1)
    val executionStackSize: math.Number = selectNumber(runtimeConfiguration, "maxStackSize").getOrElse(256)
    val securityManager: Option[Array[Value[_]] => Value[_]] = selectFunction(runtimeConfiguration, "securityManager")
    val loggerService: Option[ObjectSeq] = selectObject(runtimeConfiguration, "loggerService")
    val defaultOutputMimeType: Option[String] = selectString(runtimeConfiguration, "outputMimeType")
    val defaultWriterProperties: Option[Map[String, Any]] = selectStringAnyMap(runtimeConfiguration, "writerProperties")
    val onException = selectString(runtimeConfiguration, "onException").getOrElse(HANDLE)
    val maybeFunction: Option[Array[Value[_]] => Value[_]] = selectFunction(runtimeConfiguration, "onUnhandledTimeout")

    val script = selectString(resources, mainFile)
      .getOrElse(throw new WeaveRuntimeException(s"Unable to find resource with name ${mainFile}", UnknownLocation))

    val bindings: ScriptingBindings = loadBindings(inputsValue, inputs)

    localResourceResolver.set(new ObjectBasedResourceResolver(resources))

    val inMemoryLoggingService = new InMemoryLoggingService()

    val logger: LoggingService = loggerService match {
      case Some(loggerService) => {
        new WeaveFunctionLoggingService(loggerService, UnknownLocation)
      }
      case None =>
        inMemoryLoggingService
    }
    val builder = ResultTypeBuilder()
    try {
      val inputTypes = bindings.entries().map((name) => new InputType(name, None)).toArray
      val nameIdentifier = NameIdentifierHelper.fromWeaveFilePath(mainFile)
      val weaveScript: DataWeaveScript =
        if (defaultOutputMimeType.isDefined) {
          val config = engine
            .newConfig()
            .withScript(script)
            .withNameIdentifier(nameIdentifier)
            .withInputs(inputTypes)
            .withDefaultOutputType(defaultOutputMimeType.get)
            .withDefaultWriterProperties(defaultWriterProperties.getOrElse(Map()))
            .withMaxTime(timeout.toLong)
          engine.compileWith(config)
        } else {
          val config = engine
            .newConfig()
            .withScript(script)
            .withNameIdentifier(nameIdentifier)
            .withInputs(inputTypes)
            .withMaxTime(timeout.toLong)
          engine.compileWith(config)
        }

      val currentServiceManager: ServiceManager = ctx.serviceManager
      val currentSettingsService = currentServiceManager.settingsService

      ctx match {
        case context: ExecutionContext => {
          //We need to copy the listeners so that things like watchdog debugger or codecoverage work
          val listeners = context.notificationManager().valueNodeListeners
          listeners.foreach((listener) => {
            weaveScript.addExecutionListener(listener)
          })
          // Propagate materialize values execution context (Required by debugger)
          if (context.materializeValues()) {
            weaveScript.materializeValues(true)
          }
        }
        case _ =>
      }

      val securityManger: SecurityManagerService =
        if (securityManager.isDefined) {
          new CompositeSecurityMangerService(currentServiceManager.securityManager, new WeaveFunctionSecurityManager(securityManager.get))
        } else {
          currentServiceManager.securityManager
        }

      val runtimeProperties = new Properties()
      runtimeProperties.setProperty(RuntimeSettings.STACK_SIZE, Math.min(executionStackSize.toInt, currentSettingsService.execution().stackSize).toString)
      val settingsService = new SimpleSettingsService(new CompositeRuntimeSettings(new PropertiesSettings(None, runtimeProperties), currentSettingsService.rootSettings()), currentSettingsService.languageLevelService, currentSettingsService.notificationService)
      val cpuLimitService = new DefaultCpuLimitService(settingsService, Some(currentServiceManager.cpuLimitService), maybeFunction)
      val servicesProvider = UserDefinedServicesProvider(Map(
        classOf[SecurityManagerService] -> securityManger,
        classOf[CpuLimitService] -> cpuLimitService,
        classOf[SettingsService] -> settingsService,
        classOf[PatternService] -> (if (timeout.toLong > 0) new CpuLimitedPatternService() else DefaultPatternService) //Pattern Service to limit Pattern Matching
      ))
      val manager = new ServiceManager(logger, serviceProvider = WeaveServicesProvider(servicesProvider, currentServiceManager.serviceProvider))
      weaveScript.maxTime(timeout.toLong)
      try {
        logger match {
          case wf: WeaveFunctionLoggingService => wf.initialize()
          case _                               =>
        }
        val result = callback(weaveScript, bindings, manager, inMemoryLoggingService)
        builder.withResult(result)
      } finally {
        logger match {
          case wf: WeaveFunctionLoggingService => wf.shutdown()
          case _                               =>
        }
      }
    } catch {
      case cr: CompilationException if (isHandle(onException)) => {
        val error = buildErrorObject(inMemoryLoggingService, cr)
        builder.withError(error)
      }
      case le: LocatableException if (isHandle(onException)) => {
        val error = buildErrorObject(inMemoryLoggingService, le)
        builder.withError(error)
      }
      case ex: Throwable if (isHandle(onException)) => {
        ex.printStackTrace()
        val error = buildErrorObject(inMemoryLoggingService, ex)
        builder.withError(error)
      }
    } finally {
      localResourceResolver.set(null)
    }

    builder.build()
  }
}

class ObjectBasedResourceResolver(resources: ObjectSeq)(implicit context: EvaluationContext) extends WeaveResourceResolver {
  override def resolve(name: NameIdentifier): Option[WeaveResource] = {
    val url = NameIdentifierHelper.toWeaveFilePath(name)
    val formedUrl = new File(url).toURI.toURL.toString
    val maybeKeyValuePair = resources.keyValueOf(KeyValue(url))
    maybeKeyValuePair.map((kvp) => {
      WeaveResource(formedUrl, StringType.coerce(kvp._2).evaluate.toString)
    })
  }
}

object RunScriptFunction {
  val value = Seq(RunScriptFunctionValue, EvalScriptFunctionValue)
}

object ExceptionHandling {
  val HANDLE = "HANDLE"
  val FAIL = "FAIL"

  def isFail(handle: String): Boolean = {
    FAIL.equals(handle)
  }

  def isHandle(handle: String): Boolean = {
    HANDLE.equals(handle)
  }
}

class WeaveFunctionSecurityManager(func: (Array[Value[_]]) => Value[_])(implicit ctx: EvaluationContext) extends SecurityManagerService {
  override protected def doSupports(privilege: WeaveRuntimePrivilege, args: Array[Value[_]]): Boolean = {
    BooleanType.coerce(func.apply(Array(StringValue(privilege.name), ArrayValue(args)))).evaluate
  }
}

class WeaveFunctionLoggingService(loggingService: ObjectSeq, location: WeaveLocation)(implicit ctx: EvaluationContext) extends LoggingService {

  private lazy val logger = selectFunction(loggingService, "log").getOrElse(throw new WeaveRuntimeException("`log` field is required in LoggingService at the RuntimeConfiguration", location))
  private val INFO_LEVEL = StringValue("INFO")
  private val WARN_LEVEL = StringValue("WARN")
  private val ERROR_LEVEL = StringValue("ERROR")

  private var context: Value[_] = ObjectValue(ObjectSeq.empty)

  def initialize(): Unit = {
    context = selectFunction(loggingService, "initialize")
      .map((func) => {
        func.apply(Array())
      })
      .getOrElse(ObjectValue(ObjectSeq.empty))
  }

  override def isInfoEnabled(): Boolean = {
    true
  }

  override def logInfo(msg: String): Unit = {
    logger.apply(Array(INFO_LEVEL, StringValue(msg), context))
  }

  override def logError(msg: String): Unit = {
    logger.apply(Array(ERROR_LEVEL, StringValue(msg), context))
  }

  override def logWarn(msg: String): Unit = {
    logger.apply(Array(WARN_LEVEL, StringValue(msg), context))
  }

  def shutdown(): Unit = {
    selectFunction(loggingService, "shutdown").foreach((func) => func.apply(Array()))
  }
}
