package org.mule.weave.v2.runtime.tools

import java.io.BufferedWriter
import java.io.File
import java.io.FileWriter

import org.mule.weave.v2.interpreted.RuntimeModuleNodeCompiler
import org.mule.weave.v2.interpreted.extension.ParsingContextCreator
import org.mule.weave.v2.interpreted.extension.WeaveBasedDataFormatExtensionLoaderService
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.ServiceManager
import org.mule.weave.v2.model.service.DefaultLoggingService
import org.mule.weave.v2.module.CompositeDataFormatExtensionsLoaderService
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.DataFormatExtensionsLoaderService
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.DefaultDataFormatExtensionsLoaderService
import org.mule.weave.v2.module.option.ModuleOption
import org.mule.weave.v2.parser.phase.ModuleParsingPhasesManager
import org.mule.weave.v2.sdk.ClassLoaderWeaveResourceResolver
import org.mule.weave.v2.sdk.ParsingContextFactory

object DataFormatDocGenerator {

  def emitModuleOption(option: ModuleOption): String = {
    val description = option.description + (if (option.possibleValues.isEmpty) "" else s"\n\nValid values are ${option.possibleValues.map((a) => "`" + a.toString + "`").mkString(" or ")}.")
    val defaultValue: Any = option.defaultValue
    val value: String = toStringValue(defaultValue)
    val sufix: String = if (option.required) "(Required)" else ""
    s"|`${option.name}` ${sufix} |`${option.dataType}`|${value}|${description}"
  }

  private def toStringValue(defaultValue: Any): String = {
    val value = defaultValue match {
      case Some(s) => toStringValue(s)
      case value => {
        value match {
          case st: String => {
            if (st.equals(System.getProperty("line.separator"))) {
              "`New Line`"
            } else {
              s"`'${st}'`"
            }
          }
          case null | None => "`null`"
          case theValue    => s"`${String.valueOf(theValue)}`"
        }
      }
      case _ => ""
    }
    value
  }

  def main(args: Array[String]): Unit = {

    if (args.length == 0) {
      println("Missing target directory param")
      return
    }
    val folder = args(0)
    val targetFolder = new File(folder)
    if (targetFolder.isFile) {
      println("Missing target should be a directory")
      return
    }
    targetFolder.mkdirs()

    val muleSdkParserManager = ModuleParsingPhasesManager(ParsingContextFactory.createDefaultModuleLoaderManager())

    val loaderService = WeaveBasedDataFormatExtensionLoaderService(ParsingContextCreator(muleSdkParserManager), ClassLoaderWeaveResourceResolver.noContextClassloader(), RuntimeModuleNodeCompiler())
    val manager =
      ServiceManager(
        DefaultLoggingService,
        Map(
          //Map Of Services
          classOf[DataFormatExtensionsLoaderService] -> CompositeDataFormatExtensionsLoaderService(DefaultDataFormatExtensionsLoaderService, loaderService)))
    val evaluationContext: EvaluationContext = EvaluationContext(manager)
    val modules: Seq[DataFormat[_, _]] = DataFormatManager.modules(evaluationContext)

    modules.foreach((module) => {

      val targetFile = new File(targetFolder, "dataweave-formats-" + module.name() + ".adoc")
      println(s"Creating documentation for ${module.name()} at ${targetFile.getAbsolutePath}")

      val dataFormatHeaderBegin =
        s"""
           |= ${module.label()} Format
           |ifndef::env-site,env-github[]
           |include::_attributes.adoc[]
           |endif::[]
           |:keywords: format, ${module.name()}, ${module.acceptedMimeTypes.mkString(", ")}, ${module.defaultMimeType}
           |
           |MIME type: `${module.defaultMimeType.toString}`
           |
           |ID: `${module.name()}`
           |
           |""".stripMargin

      val dataFormatHeaderEnd =
        s"""
           |
           |[[properties]]
           |== Configuration Properties
           |
           |DataWeave supports the following configuration properties for this format.
           |
           |""".stripMargin

      val dataFormatHeader = dataFormatHeaderBegin + module.docs() + dataFormatHeaderEnd

      val centerPart = generateReaderWriterPropsDoc(module)

      val footer =
        s"""
           |
           |[[mime_types]]
           |== Supported MIME Types
           |
           |This format supports the following MIME types.
           |
           |[cols="1", options="header"]
           ||===
           || MIME Type
          ${
          module.acceptedMimeTypes.map((mime) => {
            s"||`${mime.toString}`"
          }).mkString("\n")
        }
           ||===
           |""".stripMargin

      val content = dataFormatHeader +
        centerPart +
        footer

      val writer = new BufferedWriter(new FileWriter(targetFile))
      try {
        writer.write(content)
      } finally {
        writer.close()
      }
    })

  }

  private def generateReaderWriterPropsDoc(module: DataFormat[_, _]): String = {
    val result: StringBuilder = new StringBuilder

    val writerOptions = module.writerOptions()
    val readerOptions = module.readerOptions()

    if (readerOptions.nonEmpty) {
      val prefix =
        s"""
           |[[reader_properties]]
           |=== Reader Properties
           |
           |This format accepts properties that provide instructions for reading input data.
           |
           |
           |[cols="1,1,1,3a", options="header"]
           ||===
           ||Parameter |Type |Default|Description
           |""".stripMargin
      result.append(prefix)
      readerOptions.toSeq
        .sortBy(_._1)
        .foreach((option) => {
          if (!option._2.internal())
            result.append(emitModuleOption(option._2)).append("\n")
        })
      result.append("|===\n")
    } else {
      result.append(s"""
           |[[reader_properties]]
           |=== Reader Properties
           |
           |There are no reader properties for this format.
           |
           | """.stripMargin)
    }

    if (writerOptions.nonEmpty) {
      val prefix =
        s"""
           |[[writer_properties]]
           |=== Writer Properties
           |
           |This format accepts properties that provide instructions for writing output data.
           |
           |[cols="1,1,1,3a", options="header"]
           ||===
           ||Parameter |Type |Default|Description
           |""".stripMargin
      result.append(prefix)
      writerOptions.toSeq
        .sortBy(_._1)
        .foreach((option) => {
          if (!option._2.internal())
            result.append(emitModuleOption(option._2)).append("\n")
        })
      result.append("|===\n")
    } else {
      result.append(s"""
           |[[writer_properties]]
           |=== Writer Properties
           |
           |There are no writer properties for this format.
           |
           | """.stripMargin)
    }
    result.toString()
  }
}
