package org.mule.weave.v2.runtime.core.functions.stringops

import org.mule.weave.v2.core.functions.UnaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.BooleanValue
import org.mule.weave.v2.model.values.Value

abstract class StringFiltersFunctions extends UnaryFunctionValue {
  override val R = StringType

  override protected def doExecute(v: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val numberStr = v.evaluate.toString
    if (numberStr.isEmpty) {
      onEmptyString
    } else {
      var result = BooleanValue.TRUE_BOOL
      val length = numberStr.length
      var i = 0
      while (i < length && result.evaluate) {
        val c = numberStr.charAt(i)
        if (!accepts(c)) {
          result = BooleanValue.FALSE_BOOL
        }
        i = i + 1
      }
      result
    }
  }

  protected def onEmptyString: BooleanValue = {
    BooleanValue.FALSE_BOOL
  }

  protected def accepts(c: Char): Boolean
}

object IsWhitespaceStringFunctionValue extends StringFiltersFunctions {

  override protected def onEmptyString: BooleanValue = {
    BooleanValue.TRUE_BOOL
  }

  protected def accepts(c: Char): Boolean = {
    Character.isWhitespace(c)
  }
}

object IsNumericStringFunctionValue extends StringFiltersFunctions {
  override protected def accepts(c: Char): Boolean = {
    Character.isDigit(c)
  }
}

object IsAlphaStringFunctionValue extends StringFiltersFunctions {
  override protected def accepts(c: Char): Boolean = {
    Character.isLetter(c)
  }
}

object IsAlphaNumericStringFunctionValue extends StringFiltersFunctions {
  override protected def accepts(c: Char): Boolean = {
    Character.isLetterOrDigit(c)
  }
}

object IsUpperCaseStringFunctionValue extends StringFiltersFunctions {
  override protected def accepts(c: Char): Boolean = {
    Character.isUpperCase(c)
  }
}

object IsLowerCaseStringFunctionValue extends StringFiltersFunctions {
  override protected def accepts(c: Char): Boolean = {
    Character.isLowerCase(c)
  }
}
